/**
 * LaTeX Parser v3 - 引用系统处理器
 * 
 * 负责引用系统的解析、验证和HTML生成
 */

class ReferenceProcessor {
  constructor(options = {}) {
    this.options = {
      generateTooltips: true,
      useTypeColors: true,
      enableHover: true,
      ...options
    };
  }

  /**
   * 渲染普通引用为HTML
   * @param {object} refData - 引用数据
   * @returns {string} - 渲染后的HTML
   */
  renderReference(refData) {
    const {
      refId,
      type = 'ref',
      targetType,
      targetNumber,
      targetTitle = ''
    } = refData;

    let cssClass = `latex-${type}`;
    if (targetType) cssClass += ` ${targetType}`;

    let html = `<a href="#${refId}" class="${cssClass}"`;
    
    if (this.options.generateTooltips && targetTitle) {
      html += ` title="${this.escapeHtml(targetTitle)}"`;
    }
    
    html += `>${targetNumber}</a>`;
    return html;
  }

  /**
   * 渲染页面引用为HTML
   * @param {object} refData - 引用数据
   * @returns {string} - 渲染后的HTML
   */
  renderPageReference(refData) {
    const {
      refId,
      targetNumber
    } = refData;

    return `<a href="#${refId}" class="latex-pageref">${targetNumber}</a>`;
  }

  /**
   * 渲染引用错误为HTML
   * @param {string} refId - 引用ID
   * @returns {string} - 渲染后的HTML
   */
  renderReferenceError(refId) {
    return `<span class="latex-ref-error" title="未定义的引用: ${refId}">??</span>`;
  }

  /**
   * 生成引用系统CSS样式
   * @returns {string}
   */
  generateReferenceCSS() {
    return `
.latex-ref,
.latex-pageref {
  color: #0066cc;
  text-decoration: none;
  cursor: pointer;
  font-weight: 500;
  border-bottom: 1px dotted #0066cc;
}

.latex-ref:hover,
.latex-pageref:hover {
  background-color: #e6f2ff;
  text-decoration: none;
}

.latex-ref-error {
  color: #cc0000;
  font-weight: bold;
  background-color: #ffe6e6;
  padding: 1px 3px;
  border-radius: 2px;
}

/* 不同类型引用的颜色区分 */
.latex-ref.section {
  color: #006600;
  border-bottom-color: #006600;
}

.latex-ref.figure {
  color: #cc6600;
  border-bottom-color: #cc6600;
}

.latex-ref.table {
  color: #6600cc;
  border-bottom-color: #6600cc;
}

.latex-ref.equation {
  color: #cc0066;
  border-bottom-color: #cc0066;
}
    `.trim();
  }

  /**
   * 生成类型化引用CSS样式
   * @returns {string}
   */
  generateTypedReferenceCSS() {
    return `
.latex-ref.section {
  color: #006600;
  border-bottom-color: #006600;
}

.latex-ref.section:hover {
  background-color: #e6ffe6;
}

.latex-ref.figure {
  color: #cc6600;
  border-bottom-color: #cc6600;
}

.latex-ref.figure:hover {
  background-color: #fff2e6;
}

.latex-ref.table {
  color: #6600cc;
  border-bottom-color: #6600cc;
}

.latex-ref.table:hover {
  background-color: #f2e6ff;
}

.latex-ref.equation {
  color: #cc0066;
  border-bottom-color: #cc0066;
}

.latex-ref.equation:hover {
  background-color: #ffe6f2;
}
    `.trim();
  }

  /**
   * 验证引用完整性
   * @param {object} references - 引用列表
   * @param {object} labels - 标签列表
   * @returns {object} 验证结果
   */
  validateReferences(references, labels) {
    const errors = [];
    const warnings = [];
    const validRefs = [];

    references.forEach(ref => {
      const { refId, type } = ref;
      
      if (!labels[refId]) {
        errors.push(`未定义的引用: ${refId}`);
      } else {
        // 检查自引用
        if (ref.sourceLabel === refId) {
          warnings.push(`自引用检测: ${refId}`);
        }
        validRefs.push(ref);
      }
    });

    return {
      valid: errors.length === 0,
      errors,
      warnings,
      validReferences: validRefs
    };
  }

  /**
   * HTML转义
   * @param {string} text
   * @returns {string}
   */
  escapeHtml(text) {
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 生成引用统计信息
   * @param {object} labels - 标签列表
   * @param {array} references - 引用列表
   * @returns {object}
   */
  generateReferenceStats(labels, references) {
    const labelsByType = {};
    const referencesByType = {};
    
    // 统计标签
    Object.values(labels).forEach(label => {
      const type = label.type;
      labelsByType[type] = (labelsByType[type] || 0) + 1;
    });
    
    // 统计引用
    references.forEach(ref => {
      const targetLabel = labels[ref.refId];
      if (targetLabel) {
        const type = targetLabel.type;
        referencesByType[type] = (referencesByType[type] || 0) + 1;
      }
    });

    return {
      totalLabels: Object.keys(labels).length,
      totalReferences: references.length,
      labelsByType,
      referencesByType,
      unusedLabels: Object.keys(labels).filter(labelId => 
        !references.some(ref => ref.refId === labelId)
      )
    };
  }
}

module.exports = ReferenceProcessor;