/**
 * LaTeX Parser v3 - 文档元数据处理器
 * 
 * 负责元数据的解析、验证和HTML生成
 */

class MetadataProcessor {
  constructor(options = {}) {
    this.options = {
      enableFormatting: true,
      dateLocale: 'zh-CN',
      authorSeparator: ' 和 ',
      titleCentered: true,
      ...options
    };
  }

  /**
   * 生成标题页HTML
   * @param {object} metadata - 元数据对象
   * @returns {string} - 渲染后的HTML
   */
  generateTitlePage(metadata) {
    if (!metadata || Object.keys(metadata).length === 0) {
      return '';
    }

    let html = '<div class="document-title">\n';

    // 生成标题
    if (metadata.title && metadata.title.content) {
      const titleContent = this.options.enableFormatting 
        ? this.processFormatting(metadata.title.content)
        : this.escapeHtml(metadata.title.content);
      html += `  <h1 class="title">${titleContent}</h1>\n`;
    }

    // 生成作者
    if (metadata.author && metadata.author.content) {
      html += '  <div class="authors">\n';
      
      if (metadata.author.authors && metadata.author.authors.length > 0) {
        // 多作者处理
        const authorElements = metadata.author.authors.map(author => {
          const processedAuthor = this.options.enableFormatting
            ? this.processFormatting(author.trim())
            : this.escapeHtml(author.trim());
          return `    <span class="author">${processedAuthor}</span>`;
        });
        html += authorElements.join('\n');
      } else {
        // 单作者处理
        const authorContent = this.options.enableFormatting
          ? this.processFormatting(metadata.author.content)
          : this.escapeHtml(metadata.author.content);
        html += `    <span class="author">${authorContent}</span>`;
      }
      
      html += '\n  </div>\n';
    }

    // 生成日期
    if (metadata.date && metadata.date.content) {
      const dateContent = this.options.enableFormatting
        ? this.processFormatting(metadata.date.content)
        : this.escapeHtml(metadata.date.content);
      html += `  <div class="date">${dateContent}</div>\n`;
    }

    html += '</div>\n';
    return html;
  }

  /**
   * 生成元数据CSS样式
   * @returns {string}
   */
  generateMetadataCSS() {
    return `
.document-title {
  text-align: center;
  margin: 2em 0 3em 0;
  padding: 1em;
  border-bottom: 1px solid #ccc;
}

.document-title .title {
  font-size: 2em;
  font-weight: bold;
  margin: 0 0 0.5em 0;
  color: #333;
  line-height: 1.2;
}

.document-title .authors {
  font-size: 1.2em;
  margin: 1em 0;
  color: #555;
}

.document-title .author {
  display: inline-block;
  margin: 0 0.5em;
}

.document-title .author:not(:last-child):after {
  content: "，";
}

.document-title .date {
  font-size: 1em;
  margin-top: 1em;
  color: #666;
  font-style: italic;
}

/* 感谢信息样式 */
.thanks {
  font-size: 0.8em;
  vertical-align: super;
  color: #0066cc;
}

/* 单位信息样式 */
.affiliation {
  font-size: 0.9em;
  margin-top: 0.5em;
  color: #777;
}
    `.trim();
  }

  /**
   * 验证元数据完整性
   * @param {object} metadata - 元数据对象
   * @returns {object} 验证结果
   */
  validateMetadata(metadata) {
    const errors = [];
    const warnings = [];

    // 验证标题
    if (metadata.title) {
      if (!metadata.title.content || !metadata.title.content.trim()) {
        errors.push('标题不能为空');
      }
    }

    // 验证作者
    if (metadata.author) {
      if (!metadata.author.content || !metadata.author.content.trim()) {
        warnings.push('作者信息为空');
      }
    }

    // 验证日期
    if (metadata.date) {
      if (metadata.date.content && !metadata.date.isEmpty) {
        // 简单的日期格式检查
        if (!/\d/.test(metadata.date.content) && !metadata.date.isToday) {
          warnings.push('日期格式可能不正确');
        }
      }
    }

    return {
      valid: errors.length === 0,
      errors,
      warnings
    };
  }

  /**
   * 处理格式化命令
   * @param {string} content - 内容文本
   * @returns {string} - 处理后的HTML内容
   */
  processFormatting(content) {
    let processedContent = content;

    // 处理基本格式命令
    processedContent = processedContent
      .replace(/\\textbf\{([^}]+)\}/g, '<strong>$1</strong>')
      .replace(/\\textit\{([^}]+)\}/g, '<em>$1</em>')
      .replace(/\\emph\{([^}]+)\}/g, '<em>$1</em>')
      .replace(/\\underline\{([^}]+)\}/g, '<u>$1</u>');

    // 处理感谢信息
    processedContent = processedContent
      .replace(/\\thanks\{([^}]+)\}/g, '<sup class="thanks">$1</sup>');

    // 处理换行
    processedContent = processedContent
      .replace(/\\\\/g, '<br>');

    return this.escapeHtml(processedContent);
  }

  /**
   * 解析作者信息
   * @param {string} authorContent - 作者内容
   * @returns {object} - 解析结果
   */
  parseAuthorContent(authorContent) {
    const result = {
      content: authorContent,
      authors: [],
      hasThanks: false,
      hasAffiliation: false
    };

    // 检查是否有感谢信息
    if (/\\thanks\{/.test(authorContent)) {
      result.hasThanks = true;
    }

    // 检查是否有单位信息（通过\\\\换行判断）
    if (/\\\\/.test(authorContent)) {
      result.hasAffiliation = true;
    }

    // 分割作者（通过\\and分割）
    const authorParts = authorContent.split(/\\and/);
    result.authors = authorParts
      .map(author => author.trim())
      .filter(author => author.length > 0);

    return result;
  }

  /**
   * 处理今日日期
   * @returns {string} - 格式化的今日日期
   */
  formatTodayDate() {
    const today = new Date();
    const year = today.getFullYear();
    const month = today.getMonth() + 1;
    const day = today.getDate();
    
    return `${year}年${month}月${day}日`;
  }

  /**
   * HTML转义
   * @param {string} text
   * @returns {string}
   */
  escapeHtml(text) {
    // 简化版HTML转义，避免双重转义格式化内容
    if (text.includes('<') && text.includes('>')) {
      return text; // 已经是HTML格式，不再转义
    }
    
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 生成元数据统计信息
   * @param {object} metadata - 元数据对象
   * @returns {object}
   */
  generateMetadataStats(metadata) {
    const stats = {
      hasTitle: false,
      hasAuthor: false,
      hasDate: false,
      authorCount: 0,
      titleLength: 0,
      hasMaketitle: false
    };

    if (metadata.title && metadata.title.content) {
      stats.hasTitle = true;
      stats.titleLength = metadata.title.content.length;
    }

    if (metadata.author && metadata.author.content) {
      stats.hasAuthor = true;
      stats.authorCount = metadata.author.authors ? metadata.author.authors.length : 1;
    }

    if (metadata.date && metadata.date.content) {
      stats.hasDate = true;
    }

    return stats;
  }
}

module.exports = MetadataProcessor;