/**
 * LaTeX Parser v3 - 数学公式处理器
 * 
 * 负责数学公式的解析、渲染和KaTeX集成
 */

class MathProcessor {
  constructor(options = {}) {
    this.options = {
      enableKaTeX: true,
      displayMode: false,
      throwOnError: false,
      ...options
    };
  }

  /**
   * 渲染行内数学公式
   * @param {string} formula - LaTeX数学公式
   * @returns {string} - 渲染后的HTML
   */
  renderInlineMath(formula) {
    try {
      if (this.isKaTeXAvailable()) {
        return global.katex.renderToString(formula, {
          displayMode: false,
          throwOnError: this.options.throwOnError
        });
      }
      return `<span class="latex-inline-math">${this.escapeHtml(formula)}</span>`;
    } catch (error) {
      return `<span class="latex-math-error">${this.escapeHtml(formula)}</span>`;
    }
  }

  /**
   * 渲染块级数学公式
   * @param {string} formula - LaTeX数学公式
   * @returns {string} - 渲染后的HTML
   */
  renderDisplayMath(formula) {
    try {
      if (this.isKaTeXAvailable()) {
        return global.katex.renderToString(formula, {
          displayMode: true,
          throwOnError: this.options.throwOnError
        });
      }
      return `<div class="latex-display-math">${this.escapeHtml(formula)}</div>`;
    } catch (error) {
      return `<div class="latex-math-error">${this.escapeHtml(formula)}</div>`;
    }
  }

  /**
   * 检查KaTeX是否可用
   * @returns {boolean}
   */
  isKaTeXAvailable() {
    return typeof global !== 'undefined' && 
           global.katex && 
           typeof global.katex.renderToString === 'function';
  }

  /**
   * HTML转义
   * @param {string} text
   * @returns {string}
   */
  escapeHtml(text) {
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 生成数学公式CSS样式
   * @returns {string}
   */
  generateMathCSS() {
    return `
.latex-inline-math {
  display: inline;
  margin: 0 2px;
}

.latex-display-math {
  display: block;
  margin: 16px 0;
  text-align: center;
}

.latex-equation {
  display: block;
  margin: 16px 0;
  text-align: center;
  position: relative;
}

.latex-equation::after {
  content: counter(equation);
  position: absolute;
  right: 0;
  top: 50%;
  transform: translateY(-50%);
}

.latex-align {
  display: block;
  margin: 16px 0;
}

.latex-math-error {
  background-color: #ffebee;
  color: #c62828;
  padding: 2px 4px;
  border-radius: 3px;
  font-family: monospace;
}
    `.trim();
  }
}

module.exports = MathProcessor;