/**
 * LaTeX Parser v3 - 脚注处理器
 * 
 * 负责脚注的解析、编号和HTML生成
 */

class FootnoteProcessor {
  constructor(options = {}) {
    this.options = {
      numberingStyle: 'numeric', // numeric, roman, alpha, symbols
      startNumber: 1,
      longContentThreshold: 200,
      enableFormatting: true,
      ...options
    };
    
    this.numberingStyles = {
      numeric: (n) => n.toString(),
      roman: (n) => this.toRoman(n),
      alpha: (n) => String.fromCharCode(96 + n), // a, b, c...
      symbols: (n) => ['*', '†', '‡', '§', '¶', '#'][n - 1] || n.toString()
    };
  }

  /**
   * 渲染脚注链接为HTML
   * @param {object} footnoteData - 脚注数据
   * @returns {string} - 渲染后的HTML
   */
  renderFootnoteLink(footnoteData) {
    const { id, number } = footnoteData;
    return `<sup><a href="#${id}" class="footnote-link">${number}</a></sup>`;
  }

  /**
   * 渲染脚注区域为HTML
   * @param {array} footnotes - 脚注数组
   * @returns {string} - 渲染后的HTML
   */
  renderFootnotesSection(footnotes) {
    if (!footnotes || footnotes.length === 0) {
      return '';
    }

    let html = `<div class="footnotes-section">
<h3>脚注</h3>
<ol class="footnotes-list">`;

    footnotes.forEach(footnote => {
      const content = this.options.enableFormatting 
        ? this.processFootnoteContent(footnote.content)
        : this.escapeHtml(footnote.content);
        
      html += `
<li id="${footnote.id}" class="footnote-item">
  <span class="footnote-content">${content}</span>
  <a href="#footnote-ref-${footnote.number}" class="footnote-backref">↩</a>
</li>`;
    });

    html += `
</ol>
</div>`;

    return html;
  }

  /**
   * 为脚注分配编号
   * @param {array} footnotes - 脚注数组
   * @returns {array} - 带编号的脚注数组
   */
  assignNumbers(footnotes) {
    const numberingFunc = this.numberingStyles[this.options.numberingStyle];
    
    return footnotes.map((footnote, index) => ({
      ...footnote,
      number: numberingFunc(index + this.options.startNumber),
      id: footnote.id || `footnote-${index + 1}`
    }));
  }

  /**
   * 处理脚注内容的格式化
   * @param {string} content - 脚注内容
   * @returns {string} - 处理后的HTML内容
   */
  processFootnoteContent(content) {
    let processedContent = content;

    // 处理基本格式命令
    processedContent = processedContent
      .replace(/\\textbf\{([^}]+)\}/g, '<strong>$1</strong>')
      .replace(/\\textit\{([^}]+)\}/g, '<em>$1</em>')
      .replace(/\\emph\{([^}]+)\}/g, '<em>$1</em>');

    // 处理URL链接
    processedContent = processedContent
      .replace(/\\url\{([^}]+)\}/g, '<a href="$1" target="_blank" rel="noopener">$1</a>');

    // 处理行内数学公式（简化处理）
    processedContent = processedContent
      .replace(/\$([^$]+)\$/g, '<span class="latex-inline-math">$1</span>');

    return this.escapeHtml(processedContent);
  }

  /**
   * 生成脚注CSS样式
   * @returns {string}
   */
  generateFootnoteCSS() {
    return `
.footnote-link {
  color: #0066cc;
  text-decoration: none;
  font-size: 0.8em;
  font-weight: normal;
}

.footnote-link:hover {
  text-decoration: underline;
}

.footnotes-section {
  margin-top: 2em;
  padding-top: 1em;
  border-top: 1px solid #ccc;
  font-size: 0.9em;
}

.footnotes-section h3 {
  font-size: 1.1em;
  margin-bottom: 0.5em;
  color: #333;
}

.footnotes-list {
  list-style-type: decimal;
  padding-left: 1.5em;
  margin: 0;
}

.footnote-item {
  margin-bottom: 0.5em;
  line-height: 1.4;
}

.footnote-content {
  display: inline;
}

.footnote-backref {
  margin-left: 0.5em;
  color: #666;
  text-decoration: none;
  font-size: 0.9em;
}

.footnote-backref:hover {
  color: #0066cc;
}

/* 长脚注样式 */
.footnote-item.long {
  margin-bottom: 1em;
}

.footnote-item.long .footnote-content {
  display: block;
  margin-bottom: 0.3em;
}
    `.trim();
  }

  /**
   * 数字转罗马数字
   * @param {number} num
   * @returns {string}
   */
  toRoman(num) {
    const romanNumerals = [
      { value: 10, symbol: 'x' },
      { value: 9, symbol: 'ix' },
      { value: 5, symbol: 'v' },
      { value: 4, symbol: 'iv' },
      { value: 1, symbol: 'i' }
    ];
    
    let result = '';
    for (const { value, symbol } of romanNumerals) {
      while (num >= value) {
        result += symbol;
        num -= value;
      }
    }
    return result;
  }

  /**
   * HTML转义
   * @param {string} text
   * @returns {string}
   */
  escapeHtml(text) {
    // 简化版HTML转义，避免双重转义格式化内容
    if (text.includes('<') && text.includes('>')) {
      return text; // 已经是HTML格式，不再转义
    }
    
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 验证脚注完整性
   * @param {array} footnotes - 脚注数组
   * @returns {object} 验证结果
   */
  validateFootnotes(footnotes) {
    const errors = [];
    const warnings = [];
    const stats = {
      totalFootnotes: footnotes.length,
      longFootnotes: 0,
      formattedFootnotes: 0,
      emptyFootnotes: 0
    };

    footnotes.forEach((footnote, index) => {
      if (!footnote.content || !footnote.content.trim()) {
        stats.emptyFootnotes++;
        warnings.push(`空的脚注内容 (脚注 ${index + 1})`);
      }

      if (footnote.content && footnote.content.length > this.options.longContentThreshold) {
        stats.longFootnotes++;
        footnote.isLong = true;
      }

      if (footnote.content && /\\[a-zA-Z]+\{/.test(footnote.content)) {
        stats.formattedFootnotes++;
        footnote.hasFormatting = true;
      }

      if (footnote.content && /\\ref\{/.test(footnote.content)) {
        footnote.hasReferences = true;
      }
    });

    return {
      valid: errors.length === 0,
      errors,
      warnings,
      stats
    };
  }
}

module.exports = FootnoteProcessor;