/**
 * LaTeX Parser v3 - 图形处理器
 * 
 * 负责图形的解析、渲染和样式生成
 */

class FigureProcessor {
  constructor(options = {}) {
    this.options = {
      responsive: true,
      defaultImagePath: '',
      supportedFormats: ['png', 'jpg', 'jpeg', 'gif', 'svg', 'pdf'],
      ...options
    };
  }

  /**
   * 渲染figure环境为HTML
   * @param {object} figureData - 图形数据
   * @returns {string} - 渲染后的HTML
   */
  renderFigure(figureData) {
    const {
      images = [],
      caption,
      label,
      centered = false,
      position = '',
      subfigures = [],
      type = 'figure'
    } = figureData;

    // 构建CSS类
    let cssClasses = `latex-${type}`;
    if (centered) cssClasses += ' centered';
    if (position && typeof position === 'string') {
      cssClasses += ` position-${position.replace(/[!]/g, '')}`;
    }

    // 构建HTML结构
    let html = `<figure class="${cssClasses}"`;
    if (label) html += ` id="${label}"`;
    html += '>';

    // 渲染图片
    if (subfigures.length > 0) {
      // 子图环境
      html += '<div class="subfigures-container">';
      subfigures.forEach(subfig => {
        html += this.renderSubfigure(subfig);
      });
      html += '</div>';
    } else {
      // 普通图片
      images.forEach(image => {
        html += this.renderImage(image);
      });
    }

    // 添加标题
    if (caption) {
      html += `<figcaption>${this.escapeHtml(caption)}</figcaption>`;
    }

    html += '</figure>';
    return html;
  }

  /**
   * 渲染子图
   * @param {object} subfigData - 子图数据
   * @returns {string}
   */
  renderSubfigure(subfigData) {
    const { image, caption, width = '50%' } = subfigData;
    
    let html = `<div class="latex-subfigure" style="width: ${width}">`;
    if (image) {
      html += this.renderImage(image);
    }
    if (caption) {
      html += `<div class="subfigure-caption">${this.escapeHtml(caption)}</div>`;
    }
    html += '</div>';
    
    return html;
  }

  /**
   * 渲染单个图片
   * @param {object} imageData - 图片数据
   * @returns {string}
   */
  renderImage(imageData) {
    const {
      src,
      width,
      height,
      scale,
      angle,
      alt = ''
    } = imageData;

    // 构建样式
    const styles = [];
    if (width) styles.push(`width: ${width}`);
    if (height) styles.push(`height: ${height}`);
    if (scale) styles.push(`transform: scale(${scale})`);
    if (angle) styles.push(`transform: rotate(${angle}deg)`);

    let html = `<img src="${src}"`;
    if (alt) html += ` alt="${this.escapeHtml(alt)}"`;
    if (styles.length > 0) {
      html += ` style="${styles.join('; ')}"`;
    }
    html += '>';

    return html;
  }

  /**
   * 生成图形CSS样式
   * @returns {string}
   */
  generateFigureCSS() {
    return `
.latex-figure {
  display: block;
  margin: 16px 0;
  text-align: left;
}

.latex-figure.centered {
  text-align: center;
}

.latex-figure img {
  max-width: 100%;
  height: auto;
}

.latex-figure figcaption {
  margin-top: 8px;
  font-size: 0.9em;
  font-style: italic;
  color: #666;
  text-align: center;
}

.latex-figure.position-h {
  float: none;
}

.latex-figure.position-t {
  margin-bottom: 16px;
}

.latex-figure.position-b {
  margin-top: 16px;
}

.latex-subfigure {
  display: inline-block;
  vertical-align: top;
  margin: 0 8px;
}

.latex-subfigure img {
  width: 100%;
  height: auto;
}

.subfigure-caption {
  text-align: center;
  font-size: 0.8em;
  margin-top: 4px;
}

.subfigures-container {
  text-align: center;
}

.latex-wrapfigure {
  margin: 8px;
}

.latex-wrapfigure.position-r {
  float: right;
  margin-left: 16px;
}

.latex-wrapfigure.position-l {
  float: left;
  margin-right: 16px;
}
    `.trim();
  }

  /**
   * 生成响应式图片CSS
   * @returns {string}
   */
  generateResponsiveImageCSS() {
    return `
.latex-figure img {
  max-width: 100%;
  height: auto;
}

@media (max-width: 768px) {
  .latex-figure {
    margin: 12px 0;
  }
  
  .latex-subfigure {
    display: block;
    width: 100% !important;
    margin: 8px 0;
  }
  
  .latex-wrapfigure {
    float: none !important;
    margin: 12px 0 !important;
    width: 100% !important;
  }
}
    `.trim();
  }

  /**
   * HTML转义
   * @param {string} text
   * @returns {string}
   */
  escapeHtml(text) {
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 验证图片格式
   * @param {string} src - 图片路径
   * @returns {boolean}
   */
  isValidImageFormat(src) {
    const ext = src.split('.').pop().toLowerCase();
    return this.options.supportedFormats.includes(ext);
  }
}

module.exports = FigureProcessor;