/**
 * LaTeX Parser v3 - 文档类和包管理处理器
 * 
 * 负责文档类和包的解析、验证和配置生成
 */

class DocumentClassProcessor {
  constructor(options = {}) {
    this.options = {
      enableValidation: true,
      enableWarnings: true,
      defaultFontSize: '10pt',
      defaultPageFormat: 'letterpaper',
      ...options
    };

    // 已知文档类型
    this.knownDocumentClasses = [
      'article', 'report', 'book', 'beamer', 'letter', 'memoir', 'scrartcl', 'scrreprt', 'scrbook'
    ];

    // 已知包列表
    this.knownPackages = [
      'amsmath', 'amsfonts', 'amssymb', 'amsthm', 'graphicx', 'color', 'xcolor',
      'geometry', 'hyperref', 'babel', 'inputenc', 'fontenc', 'fontspec',
      'tikz', 'pgfplots', 'listings', 'algorithm2e', 'float', 'subfig', 'subcaption',
      'cite', 'natbib', 'biblatex', 'fancyhdr', 'setspace', 'microtype'
    ];

    // 字体大小映射
    this.fontSizeMapping = {
      '10pt': '10pt',
      '11pt': '11pt', 
      '12pt': '12pt',
      '14pt': '14pt',
      '17pt': '17pt',
      '20pt': '20pt'
    };

    // 页面格式映射
    this.pageFormatMapping = {
      'a4paper': { width: '210mm', height: '297mm', name: 'A4' },
      'letterpaper': { width: '8.5in', height: '11in', name: 'Letter' },
      'legalpaper': { width: '8.5in', height: '14in', name: 'Legal' },
      'a5paper': { width: '148mm', height: '210mm', name: 'A5' },
      'b5paper': { width: '176mm', height: '250mm', name: 'B5' }
    };
  }

  /**
   * 生成文档配置
   * @param {object} documentClass - 文档类信息
   * @param {array} packages - 包列表
   * @returns {object} - 文档配置
   */
  generateDocumentConfig(documentClass, packages) {
    const config = {
      documentClass: {
        type: documentClass.type,
        options: documentClass.options || [],
        isKnown: this.knownDocumentClasses.includes(documentClass.type)
      },
      packages: this.processPackageList(packages),
      fontSettings: this.extractFontSettings(documentClass.options || []),
      pageSettings: this.extractPageSettings(documentClass.options || [])
    };

    return config;
  }

  /**
   * 生成文档类CSS样式
   * @param {object} documentClass - 文档类信息
   * @returns {string}
   */
  generateDocumentCSS(documentClass) {
    const options = documentClass.options || [];
    const fontSize = this.extractFontSize(options);
    const pageFormat = this.extractPageFormat(options);
    const columns = this.extractColumnCount(options);
    
    let css = `
.document-${documentClass.type} {
  font-family: "Times New Roman", "SimSun", serif;
  font-size: ${fontSize};
  line-height: 1.6;
  color: #333;
  margin: 0 auto;
  padding: 2em;
}`;

    // 页面格式设置
    if (this.pageFormatMapping[pageFormat]) {
      const format = this.pageFormatMapping[pageFormat];
      css += `

.document-${documentClass.type} {
  max-width: ${format.width};
  min-height: ${format.height};
}`;
    }

    // 分栏设置
    css += `

.document-${documentClass.type} .content {
  column-count: ${columns};
  column-gap: 2em;
}`;

    // 特殊文档类型样式
    if (documentClass.type === 'beamer') {
      css += `

.document-beamer {
  background: #f8f9fa;
  border: 1px solid #dee2e6;
  border-radius: 8px;
  padding: 1.5em;
}

.document-beamer .frame {
  margin-bottom: 2em;
  padding: 1em;
  background: white;
  border-radius: 4px;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}`;
    }

    if (documentClass.type === 'book') {
      css += `

.document-book {
  text-align: justify;
}

.document-book .chapter {
  page-break-before: always;
  margin-top: 3em;
}

.document-book .chapter-title {
  font-size: 2em;
  font-weight: bold;
  margin-bottom: 1.5em;
  text-align: center;
}`;
    }

    return css.trim();
  }

  /**
   * 验证包依赖关系
   * @param {array} packages - 包列表
   * @returns {object} 验证结果
   */
  validatePackageDependencies(packages) {
    const warnings = [];
    const knownPackages = [];
    const unknownPackages = [];

    packages.forEach(pkg => {
      const packageNames = this.parsePackageNames(pkg.name);
      
      packageNames.forEach(name => {
        if (this.knownPackages.includes(name)) {
          knownPackages.push(name);
        } else {
          unknownPackages.push(name);
          warnings.push(`未知包: ${name}`);
        }
      });
    });

    // 检查包冲突
    this.checkPackageConflicts(knownPackages, warnings);

    return {
      valid: true, // 基础验证通过，未知包不影响解析
      warnings,
      knownPackages,
      unknownPackages
    };
  }

  /**
   * 生成完整文档配置
   * @param {object} documentClass - 文档类
   * @param {array} packages - 包列表
   * @returns {object}
   */
  generateFullDocumentConfig(documentClass, packages) {
    const baseConfig = this.generateDocumentConfig(documentClass, packages);
    
    return {
      ...baseConfig,
      languageSettings: this.extractLanguageSettings(packages),
      geometrySettings: this.extractGeometrySettings(packages),
      hyperlinkSettings: this.extractHyperlinkSettings(packages),
      mathSettings: this.extractMathSettings(packages)
    };
  }

  /**
   * 处理包列表
   * @private
   */
  processPackageList(packages) {
    return packages.map(pkg => ({
      name: pkg.name,
      options: pkg.options || [],
      packageList: this.parsePackageNames(pkg.name),
      isKnown: this.parsePackageNames(pkg.name).every(name => 
        this.knownPackages.includes(name)
      )
    }));
  }

  /**
   * 提取字体设置
   * @private
   */
  extractFontSettings(options) {
    return {
      size: this.extractFontSize(options),
      family: 'serif', // 默认字体族
      encoding: this.extractOption(options, /^[A-Z]\d+$/) || 'T1' // 字体编码
    };
  }

  /**
   * 提取页面设置
   * @private
   */
  extractPageSettings(options) {
    const format = this.extractPageFormat(options);
    const columns = this.extractColumnCount(options);
    const twoside = options.includes('twoside');
    const landscape = options.includes('landscape');
    
    return {
      format,
      columns,
      twoside,
      landscape,
      openright: options.includes('openright'),
      openany: options.includes('openany')
    };
  }

  /**
   * 提取字体大小
   * @private
   */
  extractFontSize(options) {
    const sizeOption = options.find(opt => this.fontSizeMapping[opt]);
    return sizeOption ? this.fontSizeMapping[sizeOption] : this.options.defaultFontSize;
  }

  /**
   * 提取页面格式
   * @private
   */
  extractPageFormat(options) {
    const formatOption = options.find(opt => this.pageFormatMapping[opt]);
    return formatOption || this.options.defaultPageFormat;
  }

  /**
   * 提取分栏数量
   * @private
   */
  extractColumnCount(options) {
    if (options.includes('twocolumn')) return 2;
    if (options.includes('onecolumn')) return 1;
    return 1; // 默认单栏
  }

  /**
   * 提取选项值
   * @private
   */
  extractOption(options, pattern) {
    return options.find(opt => pattern.test(opt));
  }

  /**
   * 解析包名列表
   * @private
   */
  parsePackageNames(packageString) {
    return packageString.split(',').map(name => name.trim()).filter(name => name);
  }

  /**
   * 检查包冲突
   * @private
   */
  checkPackageConflicts(packages, warnings) {
    const conflicts = [
      { packages: ['amsmath', 'amstex'], message: 'amsmath和amstex包可能冲突' },
      { packages: ['color', 'xcolor'], message: '建议只使用xcolor包，避免与color包冲突' },
      { packages: ['subfig', 'subcaption'], message: 'subfig和subcaption包冲突，建议只使用一个' }
    ];

    conflicts.forEach(conflict => {
      const hasConflict = conflict.packages.every(pkg => packages.includes(pkg));
      if (hasConflict) {
        warnings.push(conflict.message);
      }
    });
  }

  /**
   * 提取语言设置
   * @private
   */
  extractLanguageSettings(packages) {
    const babelPkg = packages.find(pkg => pkg.name.includes('babel'));
    const languages = [];
    
    if (babelPkg && babelPkg.options) {
      languages.push(...babelPkg.options);
    }

    return {
      languages,
      hasMultilingual: languages.length > 1,
      mainLanguage: languages[languages.length - 1] || 'english'
    };
  }

  /**
   * 提取几何设置
   * @private
   */
  extractGeometrySettings(packages) {
    const geometryPkg = packages.find(pkg => pkg.name.includes('geometry'));
    const settings = {};
    
    if (geometryPkg && geometryPkg.options) {
      geometryPkg.options.forEach(option => {
        const [key, value] = option.split('=');
        if (value) {
          settings[key] = value;
        }
      });
    }

    return settings;
  }

  /**
   * 提取超链接设置
   * @private
   */
  extractHyperlinkSettings(packages) {
    const hyperrefPkg = packages.find(pkg => pkg.name.includes('hyperref'));
    const settings = { enabled: false };
    
    if (hyperrefPkg) {
      settings.enabled = true;
      settings.options = {};
      
      if (hyperrefPkg.options) {
        hyperrefPkg.options.forEach(option => {
          const [key, value] = option.split('=');
          if (value) {
            settings.options[key] = value === 'true' ? true : value === 'false' ? false : value;
          }
        });
      }
    }

    return settings;
  }

  /**
   * 提取数学设置
   * @private
   */
  extractMathSettings(packages) {
    const mathPackages = packages.filter(pkg => 
      ['amsmath', 'amsfonts', 'amssymb', 'amsthm', 'mathtools'].some(math => 
        pkg.name.includes(math)
      )
    );

    return {
      enabled: mathPackages.length > 0,
      packages: mathPackages.map(pkg => pkg.name),
      hasAdvancedMath: mathPackages.some(pkg => 
        pkg.name.includes('amsmath') || pkg.name.includes('mathtools')
      )
    };
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = DocumentClassProcessor;
  console.log('[DocumentClassProcessor] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境
  window.DocumentClassProcessor = DocumentClassProcessor;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.DocumentClassProcessor = DocumentClassProcessor;
  }
  console.log('[DocumentClassProcessor] 模块已加载到浏览器环境');
}