/**
 * LaTeX Parser v3 - 文献引用处理器
 * 
 * 负责文献引用的解析、验证和HTML生成
 */

class BibliographyProcessor {
  constructor(options = {}) {
    this.options = {
      enableNumbering: true,
      enablePageInfo: true,
      citationStyle: 'numeric', // 'numeric', 'author-year'
      ...options
    };

    // 引用计数器
    this.citationCounter = 0;
    this.citationMap = new Map(); // key -> number
  }

  /**
   * 重置引用计数器
   */
  resetCounters() {
    this.citationCounter = 0;
    this.citationMap.clear();
  }

  /**
   * 格式化引用
   * @param {array} keys - 引用键列表
   * @param {array} citations - 所有引用信息
   * @param {string} pageInfo - 页码信息
   * @returns {string} 格式化后的引用
   */
  formatCitation(keys, citations, pageInfo = '') {
    if (!keys || keys.length === 0) return '[?]';

    const numbers = keys.map(key => {
      const citation = citations.find(c => c.key === key);
      return citation ? parseInt(citation.number) : null;
    }).filter(n => n !== null).sort((a, b) => a - b);

    if (numbers.length === 0) return '[?]';

    // 格式化连续数字为范围
    const formatted = this.compressNumberSequence(numbers);
    const baseText = `[${formatted}]`;
    
    return pageInfo ? `[${formatted}, ${pageInfo}]` : baseText;
  }

  /**
   * 压缩连续数字序列
   * @private
   */
  compressNumberSequence(numbers) {
    if (numbers.length <= 1) return numbers.join(',');

    const ranges = [];
    let start = numbers[0];
    let end = numbers[0];

    for (let i = 1; i < numbers.length; i++) {
      if (numbers[i] === end + 1) {
        end = numbers[i];
      } else {
        if (start === end) {
          ranges.push(start.toString());
        } else if (end === start + 1) {
          ranges.push(`${start},${end}`);
        } else {
          ranges.push(`${start}-${end}`);
        }
        start = end = numbers[i];
      }
    }

    // 处理最后一个范围
    if (start === end) {
      ranges.push(start.toString());
    } else if (end === start + 1) {
      ranges.push(`${start},${end}`);
    } else {
      ranges.push(`${start}-${end}`);
    }

    return ranges.join(',');
  }

  /**
   * 生成文献列表HTML
   * @param {array} items - 文献条目列表
   * @returns {string} HTML字符串
   */
  generateBibliographyHTML(items) {
    if (!items || items.length === 0) {
      return `
<div class="bibliography">
  <div class="bib-title">参考文献</div>
  <div class="bib-empty">暂无参考文献</div>
</div>`.trim();
    }

    let html = `
<div class="bibliography">
  <div class="bib-title">参考文献</div>
  <div class="bib-list">`;

    items.forEach(item => {
      html += `
    <div class="bib-item" id="bib-${item.key}">
      <span class="bib-number">[${item.number}]</span>
      <span class="bib-content">${item.content}</span>
    </div>`;
    });

    html += `
  </div>
</div>`;

    return html.trim();
  }

  /**
   * 生成bibliography注意HTML
   * @param {string} filename - bib文件名
   * @returns {string}
   */
  generateBibliographyNoticeHTML(filename) {
    return `
<div class="bibliography-notice">
  <div class="notice-title">参考文献</div>
  <div class="notice-content">文献数据来源: ${filename}</div>
  <div class="notice-hint">注：实际文献列表需要LaTeX编译环境生成</div>
</div>`.trim();
  }

  /**
   * 生成文献样式CSS
   * @returns {string} CSS字符串
   */
  generateBibliographyCSS() {
    return `
/* 文献引用样式 */
.citation {
  color: #2c3e50;
  text-decoration: none;
  font-weight: bold;
  padding: 0 2px;
}

.citation:hover {
  color: #3498db;
  text-decoration: underline;
}

/* 文献列表样式 */
.bibliography {
  margin: 2em 0;
  padding: 1.5em;
  background: #f8f9fa;
  border: 1px solid #e9ecef;
  border-radius: 6px;
}

.bib-title {
  font-size: 1.4em;
  font-weight: bold;
  color: #2c3e50;
  margin-bottom: 1em;
  text-align: center;
  border-bottom: 2px solid #3498db;
  padding-bottom: 0.5em;
}

.bib-list {
  margin: 0;
  padding: 0;
}

.bib-item {
  margin: 0.8em 0;
  padding: 0.5em 0;
  line-height: 1.6;
  display: flex;
  align-items: flex-start;
}

.bib-number {
  color: #2c3e50;
  font-weight: bold;
  margin-right: 0.5em;
  min-width: 2em;
  flex-shrink: 0;
}

.bib-content {
  text-align: justify;
  flex: 1;
}

.bib-empty {
  color: #6c757d;
  font-style: italic;
  text-align: center;
  margin: 1em 0;
}

/* bibliography数据库文件提示 */
.bibliography-notice {
  margin: 2em 0;
  padding: 1.5em;
  background: #fff3cd;
  border: 1px solid #ffeaa7;
  border-radius: 6px;
  border-left: 5px solid #f39c12;
}

.notice-title {
  font-size: 1.2em;
  font-weight: bold;
  color: #856404;
  margin-bottom: 0.5em;
}

.notice-content {
  color: #856404;
  margin-bottom: 0.5em;
}

.notice-hint {
  color: #6c757d;
  font-size: 0.9em;
  font-style: italic;
}

/* 响应式样式 */
@media (max-width: 768px) {
  .bibliography, .bibliography-notice {
    margin: 1.5em 0;
    padding: 1em;
  }
  
  .bib-item {
    margin: 0.6em 0;
    padding: 0.3em 0;
  }
}`.trim();
  }

  /**
   * 验证引用完整性
   * @param {array} citations - 引用列表
   * @param {array} bibliography - 文献条目列表
   * @returns {object} 验证结果
   */
  validateCitations(citations, bibliography) {
    const warnings = [];
    const missingReferences = [];
    const unusedReferences = [];

    // 创建文献键集合
    const bibKeys = new Set(bibliography.map(item => item.key));
    const citedKeys = new Set(citations.map(citation => citation.key));

    // 检查缺失的文献
    citations.forEach(citation => {
      if (!bibKeys.has(citation.key)) {
        missingReferences.push(citation.key);
        warnings.push(`未找到引用的文献: ${citation.key}`);
      }
    });

    // 检查未使用的文献
    bibliography.forEach(item => {
      if (!citedKeys.has(item.key)) {
        unusedReferences.push(item.key);
        warnings.push(`文献未被引用: ${item.key}`);
      }
    });

    return {
      valid: missingReferences.length === 0,
      warnings,
      missingReferences,
      unusedReferences
    };
  }

  /**
   * 获取引用编号
   * @param {string} key - 引用键
   * @returns {string} 编号
   */
  getCitationNumber(key) {
    if (this.citationMap.has(key)) {
      return this.citationMap.get(key).toString();
    }

    this.citationCounter++;
    this.citationMap.set(key, this.citationCounter);
    return this.citationCounter.toString();
  }

  /**
   * 解析cite命令中的多个引用
   * @param {string} keyString - 引用键字符串
   * @returns {array} 引用键数组
   */
  parseCitationKeys(keyString) {
    if (!keyString || keyString.trim() === '') return [];
    
    return keyString.split(',')
      .map(key => key.trim())
      .filter(key => key !== '');
  }

  /**
   * 解析页码信息
   * @param {string} pageString - 页码字符串
   * @returns {string} 清理后的页码
   */
  parsePageInfo(pageString) {
    if (!pageString) return '';
    return pageString.trim();
  }

  /**
   * 生成引用链接
   * @param {string} key - 引用键
   * @param {string} text - 显示文本
   * @returns {string} 链接HTML
   */
  generateCitationLink(key, text) {
    return `<a href="#bib-${key}" class="citation">${text}</a>`;
  }
}

module.exports = BibliographyProcessor;