/**
 * LaTeX Parser v3 - 高级环境处理器
 * 
 * 负责高级环境的解析、验证和HTML生成
 */

class AdvancedEnvironmentProcessor {
  constructor(options = {}) {
    this.options = {
      enableNumbering: true,
      enableTitles: true,
      enableProofEnd: true,
      ...options
    };

    // 环境类型映射
    this.environmentTypes = {
      'theorem': { name: '定理', numbered: true, showEnd: false },
      'proof': { name: '证明', numbered: false, showEnd: true },
      'abstract': { name: '摘要', numbered: false, showEnd: false },
      'definition': { name: '定义', numbered: true, showEnd: false },
      'lemma': { name: '引理', numbered: true, showEnd: false },
      'corollary': { name: '推论', numbered: true, showEnd: false },
      'proposition': { name: '命题', numbered: true, showEnd: false },
      'remark': { name: '注释', numbered: false, showEnd: false },
      'example': { name: '例子', numbered: true, showEnd: false }
    };

    // 环境计数器
    this.counters = {};
    this.resetCounters();
  }

  /**
   * 重置环境计数器
   */
  resetCounters() {
    this.counters = {
      'theorem': 0,
      'definition': 0,
      'lemma': 0,
      'corollary': 0,
      'proposition': 0,
      'example': 0
    };
  }

  /**
   * 生成环境HTML
   * @param {object} environment - 环境信息
   * @returns {string} HTML字符串
   */
  generateEnvironmentHTML(environment) {
    const envType = this.environmentTypes[environment.type];
    if (!envType) {
      return `<div class="unknown-environment">${environment.content}</div>`;
    }

    const idAttr = environment.id ? `id="${environment.id}"` : '';
    const headerText = this.generateHeaderText(environment, envType);
    const endText = envType.showEnd && this.options.enableProofEnd ? 
      '<div class="proof-end">∎</div>' : '';

    return `
<div class="${environment.type}" ${idAttr}>
  <div class="${environment.type}-header">${headerText}</div>
  <div class="${environment.type}-content">${environment.content}</div>
  ${endText}
</div>`.trim();
  }

  /**
   * 生成标题文本
   * @private
   */
  generateHeaderText(environment, envType) {
    let header = envType.name;
    
    if (envType.numbered && environment.number) {
      header += ` ${environment.number}`;
    }
    
    if (environment.title) {
      header += ` (${environment.title})`;
    }
    
    return header;
  }

  /**
   * 生成高级环境CSS样式
   * @returns {string} CSS字符串
   */
  generateAdvancedEnvironmentCSS() {
    return `
/* 通用环境样式 */
.theorem, .proof, .abstract, .definition, .lemma, .corollary, .proposition, .remark, .example {
  margin: 1.5em 0;
  padding: 1em;
  border-radius: 6px;
  background: #f8f9fa;
}

/* 环境标题样式 */
.theorem-header, .definition-header, .lemma-header, .corollary-header, .proposition-header, .example-header {
  font-weight: bold;
  color: #2c3e50;
  margin-bottom: 0.8em;
  padding: 0.3em 0.8em;
  background: #e8f4f8;
  border-radius: 4px;
  border-left: 4px solid #3498db;
}

.proof-header {
  font-weight: bold;
  color: #27ae60;
  margin-bottom: 0.8em;
  font-style: italic;
}

.abstract-header {
  font-weight: bold;
  color: #8e44ad;
  margin-bottom: 0.8em;
  text-align: center;
  font-size: 1.1em;
  text-transform: uppercase;
  letter-spacing: 1px;
}

.remark-header {
  font-weight: bold;
  color: #f39c12;
  margin-bottom: 0.8em;
  font-style: italic;
}

/* 环境内容样式 */
.theorem-content, .definition-content, .lemma-content, .corollary-content, .proposition-content, .example-content {
  line-height: 1.6;
  text-align: justify;
}

.proof-content {
  line-height: 1.6;
  text-align: justify;
  font-style: italic;
}

.abstract-content {
  line-height: 1.7;
  text-align: justify;
  max-width: 90%;
  margin: 0 auto;
}

.remark-content {
  line-height: 1.6;
  font-style: italic;
  color: #666;
}

/* 特殊环境样式 */
.theorem {
  background: linear-gradient(145deg, #f0f8ff, #e6f3ff);
  border-left: 5px solid #3498db;
}

.proof {
  background: linear-gradient(145deg, #f0fff4, #e6ffe6);
  border-left: 5px solid #27ae60;
  position: relative;
}

.abstract {
  background: linear-gradient(145deg, #faf0ff, #f0e6ff);
  border: 2px solid #8e44ad;
  text-align: center;
}

.definition {
  background: linear-gradient(145deg, #fffaf0, #fff0e6);
  border-left: 5px solid #e67e22;
}

.lemma {
  background: linear-gradient(145deg, #f5f0ff, #ebe6ff);
  border-left: 5px solid #9b59b6;
}

/* 证明结束符号 */
.proof-end {
  text-align: right;
  font-size: 1.2em;
  color: #27ae60;
  font-weight: bold;
  margin-top: 0.5em;
}

.proof-end::after {
  content: "∎";
}

/* 响应式样式 */
@media (max-width: 768px) {
  .theorem, .proof, .abstract, .definition, .lemma, .corollary, .proposition, .remark, .example {
    margin: 1em 0;
    padding: 0.8em;
  }
  
  .abstract-content {
    max-width: 100%;
  }
}`.trim();
  }

  /**
   * 验证环境结构
   * @param {array} environments - 环境列表
   * @returns {object} 验证结果
   */
  validateEnvironmentStructure(environments) {
    const warnings = [];
    let valid = true;

    environments.forEach((env, index) => {
      // 检查未知环境类型
      if (!this.environmentTypes[env.type]) {
        warnings.push(`未知环境类型: ${env.type}`);
      }

      // 检查空内容
      if (!env.content || env.content.trim() === '') {
        warnings.push(`空内容环境: ${env.type} (第${index + 1}个)`);
      }

      // 检查编号一致性
      if (this.environmentTypes[env.type]?.numbered && !env.number) {
        warnings.push(`缺少编号的${env.type}环境 (第${index + 1}个)`);
      }
    });

    return { valid, warnings };
  }

  /**
   * 生成环境编号
   * @param {string} type - 环境类型
   * @returns {string} 编号
   */
  generateNumber(type) {
    const envType = this.environmentTypes[type];
    if (!envType || !envType.numbered) {
      return '';
    }

    if (!this.counters[type]) {
      this.counters[type] = 0;
    }
    
    this.counters[type]++;
    return this.counters[type].toString();
  }

  /**
   * 生成自动ID
   * @param {string} type - 环境类型
   * @param {number} index - 环境索引
   * @returns {string}
   */
  generateAutoId(type, index) {
    return `auto-${type}-${index}`;
  }

  /**
   * 解析环境标题
   * @param {string} titleString - 标题字符串
   * @returns {string} 清理后的标题
   */
  parseTitle(titleString) {
    if (!titleString) return '';
    
    // 移除方括号并清理标题
    return titleString.replace(/^\[|\]$/g, '').trim();
  }

  /**
   * 检查环境匹配
   * @param {string} text - 输入文本
   * @returns {object} 匹配检查结果
   */
  checkEnvironmentMatching(text) {
    const beginMatches = [];
    const endMatches = [];
    const errors = [];

    // 收集所有begin和end
    const beginRegex = /\\begin\{(\w+)\}/g;
    const endRegex = /\\end\{(\w+)\}/g;
    
    let match;
    while ((match = beginRegex.exec(text)) !== null) {
      if (this.environmentTypes[match[1]]) {
        beginMatches.push({
          type: match[1],
          position: match.index,
          text: match[0]
        });
      }
    }
    
    while ((match = endRegex.exec(text)) !== null) {
      if (this.environmentTypes[match[1]]) {
        endMatches.push({
          type: match[1],
          position: match.index,
          text: match[0]
        });
      }
    }

    // 检查嵌套匹配 - 模拟栈结构
    const allMatches = [...beginMatches.map(m => ({...m, isBegin: true})), ...endMatches.map(m => ({...m, isBegin: false}))]
      .sort((a, b) => a.position - b.position);

    const stack = [];
    
    allMatches.forEach(match => {
      if (match.isBegin) {
        stack.push(match);
      } else {
        const lastBegin = stack.pop();
        if (!lastBegin) {
          errors.push(`多余的结束标签: ${match.type}`);
        } else if (lastBegin.type !== match.type) {
          errors.push(`环境不匹配: 期望\\end{${lastBegin.type}}，但发现\\end{${match.type}}`);
        }
      }
    });

    // 检查未闭合的环境
    stack.forEach(openEnv => {
      errors.push(`未闭合的环境: ${openEnv.type} (1个)`);
    });

    return {
      valid: errors.length === 0,
      beginMatches,
      endMatches,
      errors
    };
  }
}

module.exports = AdvancedEnvironmentProcessor;