# LaTeX Parser V3 使用指南

高性能的 LaTeX 解析器，将 LaTeX 内容转换为结构化数据和 HTML 输出。

## 快速开始

### Node.js

```javascript
const LaTeXParser = require('./lib/latex-parser-v3');

const parser = new LaTeXParser();
const result = parser.parseComplete('$E = mc^2$');
console.log(result.processedText);
```

### 浏览器

```html
<!-- 按顺序加载文件 -->
<script src="lib/latex-parser-v3/core/DocumentParser.js"></script>
<script src="lib/latex-parser-v3/core/BlockElementParser.js"></script>
<script src="lib/latex-parser-v3/core/InlineElementParser.js"></script>
<script src="lib/latex-parser-v3/core/MathParser.js"></script>
<script src="lib/latex-parser-v3/core/SpecialParser.js"></script>
<script src="lib/latex-parser-v3/core/LaTeXParser.js"></script>
<script src="lib/latex-parser-v3/browser.js"></script>

<script>
  const parser = new LaTeXParser();
  const result = parser.parseComplete('\\section{Hello}');
</script>
```

## 主要功能

- ✅ **完整的 LaTeX 语法支持** - 文档结构、章节、列表、表格、数学公式等
- ✅ **数学公式处理** - 支持行内公式 `$...$` 和显示公式 `$$...$$`
- ✅ **文本格式化** - 粗体、斜体、下划线、字体大小等
- ✅ **环境支持** - itemize、enumerate、equation、align 等
- ✅ **错误恢复** - 遇到错误时优雅降级，继续处理
- ✅ **性能优化** - 平均处理时间 < 100ms

## API 参考

### new LaTeXParser(config)

创建解析器实例。

**参数：**
- `config` {Object} - 可选配置
  - `mode` {String} - 'tolerant'（默认）或 'strict'
  - `enabledLayers` {Array} - 启用的解析层，默认全部：`['document', 'block', 'inline', 'math', 'special']`
  - `maxRecursionDepth` {Number} - 最大递归深度，默认 100

### parser.parseComplete(content)

解析 LaTeX 内容。

**参数：**
- `content` {String} - LaTeX 文本

**返回：**
```javascript
{
  success: Boolean,        // 是否成功
  processedText: String,   // HTML 输出
  blocks: Array,          // 结构化数据块
  metadata: Object,        // 处理元信息
  warnings: Array,        // 警告列表
  errors: Array          // 错误列表
}
```

## 使用示例

### 基础用法

```javascript
const LaTeXParser = require('./lib/latex-parser-v3');
const parser = new LaTeXParser();

// 解析数学公式
const result = parser.parseComplete('$E = mc^2$');
console.log(result.processedText);
// 输出: <span class="katex">E = mc^2</span>
```

### 文档结构

```javascript
const input = `
\\section{介绍}
这是一个示例文档。

\\subsection{特性}
\\begin{itemize}
\\item \\textbf{粗体文本}
\\item \\textit{斜体文本}
\\item 数学公式 $a^2 + b^2 = c^2$
\\end{itemize}
`;

const result = parser.parseComplete(input);
console.log(result.processedText);
```

### 数学环境

```javascript
const mathInput = `
\\begin{equation}
  \\int_{0}^{\\infty} e^{-x} dx = 1
\\end{equation}

\\begin{align}
  a &= b + c \\\\
  d &= e + f
\\end{align}
`;

const result = parser.parseComplete(mathInput);
```

### 错误处理

```javascript
const result = parser.parseComplete('\\unknown{cmd}');

if (result.warnings.length > 0) {
  console.log('警告:', result.warnings);
}

if (!result.success) {
  console.error('错误:', result.errors);
}
```

### 浏览器环境

```javascript
// 使用 browser.js 提供的便捷函数
const result = parseLatex('\\section{Test}', {
  mode: 'tolerant'
});

// 或使用工厂函数
const parser = createLaTeXParser({
  enabledLayers: ['document', 'math']
});
```

## 支持的 LaTeX 语法

### 数学公式
- 行内公式：`$...$`、`\(...\)`
- 显示公式：`$$...$$`、`\[...\]`
- 环境：equation、align、gather、matrix、cases 等

### 文档结构
- `\section{...}`、`\subsection{...}`、`\subsubsection{...}`
- `\chapter{...}`、`\paragraph{...}`
- `\begin{document}...\end{document}`

### 文本格式
- `\textbf{...}` - 粗体
- `\textit{...}` - 斜体
- `\emph{...}` - 强调
- `\underline{...}` - 下划线
- `\texttt{...}` - 等宽字体

### 列表环境
- `\begin{itemize}...\end{itemize}` - 无序列表
- `\begin{enumerate}...\end{enumerate}` - 有序列表
- `\begin{description}...\end{description}` - 描述列表

### 表格
- `\begin{tabular}{...}...\end{tabular}`
- `\begin{array}{...}...\end{array}`

### 特殊命令
- `\usepackage{...}` - 包引用
- `\includegraphics{...}` - 图片插入
- `\footnote{...}` - 脚注
- `\cite{...}` - 引用
- `\ref{...}`、`\label{...}` - 交叉引用

## 配置选项

### 解析模式

- **tolerant**（默认）- 遇到错误时尝试恢复，继续处理
- **strict** - 严格模式，遇到错误立即停止

### 启用层控制

通过 `enabledLayers` 选项控制启用哪些解析层：

```javascript
const parser = new LaTeXParser({
  enabledLayers: ['document', 'math']  // 只处理文档结构和数学
});
```

可用的层：
- `document` - 文档结构（章节、标题等）
- `block` - 块级元素（列表、表格等）
- `inline` - 行内元素（格式化、特殊字符等）
- `math` - 数学公式
- `special` - 特殊功能（包引用、图片等）

### 性能优化

对于大文档，可以：

1. 只启用需要的解析层
2. 调整 `maxRecursionDepth` 限制递归深度
3. 分段处理文档

```javascript
const parser = new LaTeXParser({
  enabledLayers: ['math'],      // 只处理数学
  maxRecursionDepth: 50        // 限制递归深度
});
```

## 与其他模块集成

### 处理 Markdown 中的 LaTeX

LaTeX Parser V3 是纯 LaTeX 处理器。如需处理 Markdown 文档中的 LaTeX 代码块：

```javascript
// 使用 LaTeXIntegration 模块
const LaTeXIntegration = require('./lib/latex-integration');

const markdown = `
# 标题

普通文本

\`\`\`latex
\\section{LaTeX 内容}
\`\`\`
`;

const result = LaTeXIntegration.process(markdown);
```

### 与 KaTeX 集成

数学公式会被标记为 KaTeX 可识别的格式：

```javascript
const result = parser.parseComplete('$E = mc^2$');
// processedText 包含 KaTeX 可渲染的内容
```

### 与 PDF 生成集成

```javascript
const latexContent = '\\section{Title}\nContent';
const parsed = parser.parseComplete(latexContent);

// 使用 parsed.processedText 生成 PDF
// 使用 parsed.css 应用样式
```


## 调试和故障排除

### 检查处理结果

```javascript
const result = parser.parseComplete(input);

// 查看处理时间
console.log('处理时间:', result.metadata.processingTime, 'ms');

// 查看各层统计
console.log('各层处理:', result.metadata.layerStats);

// 查看警告
if (result.warnings.length > 0) {
  console.log('警告:', result.warnings);
}

// 查看错误
if (!result.success) {
  console.log('错误:', result.errors);
}
```

### 常见问题

**Q: 浏览器环境报错 "LaTeXParser is not defined"**  
A: 确保按正确顺序加载所有文件，browser.js 必须最后加载。

**Q: 数学公式没有正确渲染**  
A: 检查是否已加载 KaTeX，数学内容会被标记但需要 KaTeX 渲染。

**Q: ~~独立的 $$ 公式被 `<br>` 标签破坏~~（已修复）**  
A: ✅ 此问题已在新的占位符架构中解决。DocumentParser 层会保护 `$$` 块，后续层不会再插入 `<br>` 标签。

**Q: ~~输出的 HTML 中 $$ 公式被包在 `<p>` 标签中~~（已修复）**  
A: ✅ 新架构确保 `$$` 公式保持原始格式，不会被额外的 HTML 标签包围。

**Q: 处理大文档很慢**  
A: 尝试只启用需要的解析层，或分段处理文档。

**Q: 某些 LaTeX 命令不被识别**  
A: 检查 warnings 数组，未知命令会生成警告但不会中断处理。

**Q: LaTeX Parser V3 应该处理 Markdown 中的数学公式吗？**  
A: 不推荐。V3 设计用于处理纯 LaTeX 内容或 LaTeX 代码块。Markdown 中的数学公式（如 `$$...$$`）应该：
1. 直接传递给 KaTeX 的 `renderMathInElement` 函数
2. 或使用专门的 Markdown 数学扩展
3. V3 主要处理 ```latex 代码块中的内容


## 高级功能

### 自定义渲染器

```javascript
const HTMLRenderer = require('./lib/latex-parser-v3/renderers/HTMLRenderer');
const CSSGenerator = require('./lib/latex-parser-v3/renderers/CSSGenerator');

const parser = new LaTeXParser();
const htmlRenderer = new HTMLRenderer();
const cssGenerator = new CSSGenerator();

const parseResult = parser.parseComplete(latexContent);
const html = htmlRenderer.render(parseResult);
const css = cssGenerator.generate(parseResult);
```

### 处理器扩展

可用的额外处理器：
- `MathProcessor` - 高级数学处理
- `FigureProcessor` - 图表处理
- `ReferenceProcessor` - 引用处理
- `FootnoteProcessor` - 脚注处理
- `MetadataProcessor` - 元数据提取
- `TOCProcessor` - 目录生成
- `BibliographyProcessor` - 参考文献处理

## 性能指标

- 平均处理时间：< 100ms（1000行文档）
- 内存使用：< 50MB（标准文档）
- 测试覆盖率：> 90%
- 支持递归深度：100 层

## 版本信息

- **当前版本**: 3.0.0
- **架构**: 五层流水线（Five-Layer Pipeline）
- **Node.js**: >= 12.0.0
- **浏览器**: Chrome/Firefox/Safari/Edge（现代版本）

## 许可证

MIT