#!/usr/bin/env node

/**
 * LaTeX Parser V3 Core 测试运行器
 * 
 * 提供便捷的测试运行和管理功能
 */

const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');

// 导入测试配置
const testSuite = require('./index.js');

/**
 * 颜色输出工具
 */
const colors = {
  reset: '\x1b[0m',
  green: '\x1b[32m',
  red: '\x1b[31m',
  yellow: '\x1b[33m',
  blue: '\x1b[34m',
  magenta: '\x1b[35m',
  cyan: '\x1b[36m'
};

function colorize(text, color) {
  return `${colors[color]}${text}${colors.reset}`;
}

/**
 * 显示帮助信息
 */
function showHelp() {
  console.log(colorize('\nLaTeX Parser V3 Core 测试运行器', 'cyan'));
  console.log('='.repeat(50));
  console.log('\n用法: node run-tests.js [选项] [测试文件]');
  console.log('\n选项:');
  console.log('  --help, -h     显示帮助信息');
  console.log('  --all, -a      运行所有测试 (默认)');
  console.log('  --coverage, -c 生成覆盖率报告');
  console.log('  --watch, -w    监视模式运行');
  console.log('  --verbose, -v  详细输出模式');
  console.log('  --validate, -V 验证测试环境');
  console.log('  --stats, -s    显示测试统计信息');
  console.log('  --layer <n>    运行特定层级的测试 (0-5)');
  
  console.log('\n测试文件:');
  console.log('  LaTeXParser    主控制器测试');
  console.log('  DocumentParser 文档解析层测试 (L1)');
  console.log('  BlockElementParser 块级解析层测试 (L2)');
  console.log('  InlineElementParser 行内解析层测试 (L3)');
  console.log('  MathParser     数学解析层测试 (L4)');
  console.log('  SpecialParser  特殊功能解析层测试 (L5)');
  
  console.log('\n示例:');
  console.log('  node run-tests.js                    # 运行所有测试');
  console.log('  node run-tests.js --coverage         # 带覆盖率运行所有测试');
  console.log('  node run-tests.js LaTeXParser        # 运行主控制器测试');
  console.log('  node run-tests.js --layer 1          # 运行L1层测试');
  console.log('  node run-tests.js --watch MathParser # 监视模式运行数学解析测试');
  console.log();
}

/**
 * 验证测试环境
 */
function validateEnvironment() {
  console.log(colorize('🔍 验证测试环境...', 'blue'));
  
  const validation = testSuite.validateTestEnvironment();
  
  if (validation.valid) {
    console.log(colorize('✅ 测试环境验证通过', 'green'));
    return true;
  } else {
    console.log(colorize('❌ 测试环境验证失败:', 'red'));
    validation.issues.forEach(issue => {
      console.log(colorize(`   - ${issue}`, 'red'));
    });
    return false;
  }
}

/**
 * 显示测试统计信息
 */
function showStatistics() {
  console.log(colorize('\n📊 测试统计信息', 'cyan'));
  console.log('='.repeat(30));
  
  const stats = testSuite.getTestStatistics();
  const modules = testSuite.getTestModules();
  
  console.log(`总测试模块数: ${colorize(stats.totalModules, 'yellow')}`);
  console.log(`解析层级数: ${colorize(stats.layersCount, 'yellow')}`);
  console.log(`预估测试用例数: ${colorize(stats.estimatedTestCount, 'yellow')}`);
  
  console.log(colorize('\n解析层级结构:', 'blue'));
  modules.forEach(module => {
    const layerText = module.layer === 0 ? '主控制器' : `L${module.layer}层`;
    console.log(`  ${colorize(layerText, 'magenta')}: ${module.name} - ${module.description}`);
  });
  
  console.log(colorize('\n测试配置:', 'blue'));
  console.log(`  超时时间: ${stats.configuration.timeout}ms`);
  console.log(`  最大并发: ${stats.configuration.maxConcurrency}`);
  console.log(`  覆盖率要求: ${stats.configuration.coverageThreshold.global.lines}%`);
  console.log();
}

/**
 * 构建Jest命令
 */
function buildJestCommand(options) {
  let cmd = 'npx jest';
  
  // 添加测试文件路径
  if (options.testFile) {
    cmd += ` tests/${options.testFile}.spec.js`;
  } else {
    cmd += ' tests/';
  }
  
  // 添加选项
  if (options.coverage) {
    cmd += ' --coverage';
  }
  
  if (options.watch) {
    cmd += ' --watch';
  }
  
  if (options.verbose) {
    cmd += ' --verbose';
  }
  
  // 添加其他Jest选项
  cmd += ' --colors';
  
  return cmd;
}

/**
 * 运行特定层级的测试
 */
function runLayerTests(layer) {
  const modules = testSuite.getTestModules();
  const layerModules = modules.filter(m => m.layer === parseInt(layer));
  
  if (layerModules.length === 0) {
    console.log(colorize(`❌ 未找到层级 ${layer} 的测试模块`, 'red'));
    return;
  }
  
  console.log(colorize(`🚀 运行 L${layer} 层测试...`, 'blue'));
  layerModules.forEach(module => {
    console.log(colorize(`   📝 ${module.name}: ${module.description}`, 'cyan'));
  });
  
  const testFiles = layerModules.map(m => `tests/${m.name}.spec.js`).join(' ');
  const cmd = `npx jest ${testFiles} --colors --verbose`;
  
  try {
    execSync(cmd, { stdio: 'inherit' });
  } catch (error) {
    console.log(colorize(`❌ 层级 ${layer} 测试执行失败`, 'red'));
    process.exit(1);
  }
}

/**
 * 主函数
 */
function main() {
  const args = process.argv.slice(2);
  const options = {
    help: false,
    all: true,
    coverage: false,
    watch: false,
    verbose: false,
    validate: false,
    stats: false,
    layer: null,
    testFile: null
  };
  
  // 解析命令行参数
  for (let i = 0; i < args.length; i++) {
    const arg = args[i];
    
    switch (arg) {
      case '--help':
      case '-h':
        options.help = true;
        break;
      case '--all':
      case '-a':
        options.all = true;
        break;
      case '--coverage':
      case '-c':
        options.coverage = true;
        break;
      case '--watch':
      case '-w':
        options.watch = true;
        break;
      case '--verbose':
      case '-v':
        options.verbose = true;
        break;
      case '--validate':
      case '-V':
        options.validate = true;
        break;
      case '--stats':
      case '-s':
        options.stats = true;
        break;
      case '--layer':
        if (i + 1 < args.length) {
          options.layer = args[++i];
          options.all = false;
        }
        break;
      default:
        // 检查是否是测试文件名
        const validTestFiles = ['LaTeXParser', 'DocumentParser', 'BlockElementParser', 
                              'InlineElementParser', 'MathParser', 'SpecialParser'];
        if (validTestFiles.includes(arg)) {
          options.testFile = arg;
          options.all = false;
        } else {
          console.log(colorize(`❌ 未知选项: ${arg}`, 'red'));
          showHelp();
          process.exit(1);
        }
        break;
    }
  }
  
  // 处理选项
  if (options.help) {
    showHelp();
    return;
  }
  
  if (options.validate) {
    const isValid = validateEnvironment();
    if (!isValid) {
      process.exit(1);
    }
    if (args.length === 1) return; // 只验证环境
  }
  
  if (options.stats) {
    showStatistics();
    if (args.length === 1) return; // 只显示统计
  }
  
  // 检查测试环境
  console.log(colorize('🔧 LaTeX Parser V3 Core 测试运行器', 'cyan'));
  console.log('='.repeat(40));
  
  if (!validateEnvironment()) {
    process.exit(1);
  }
  
  // 运行特定层级测试
  if (options.layer !== null) {
    runLayerTests(options.layer);
    return;
  }
  
  // 运行测试
  console.log(colorize('\n🚀 开始运行测试...', 'blue'));
  
  const cmd = buildJestCommand(options);
  console.log(colorize(`执行命令: ${cmd}`, 'yellow'));
  
  try {
    execSync(cmd, { stdio: 'inherit' });
    console.log(colorize('\n✅ 测试完成!', 'green'));
  } catch (error) {
    console.log(colorize('\n❌ 测试执行失败', 'red'));
    process.exit(1);
  }
}

// 运行主函数
if (require.main === module) {
  main();
}

module.exports = {
  validateEnvironment,
  showStatistics,
  runLayerTests,
  buildJestCommand,
  main
};