/**
 * LaTeX Parser V3 Core 模块测试套件入口
 * 
 * 基于 interface.md 规范的完整测试套件
 * 提供统一的测试入口和配置管理
 */

// 测试模块路径 (不直接导入，避免非Jest环境下的错误)

/**
 * 测试套件配置
 */
const TEST_CONFIG = {
  // 测试超时时间 (毫秒)
  timeout: 10000,
  
  // 测试并发设置
  maxConcurrency: 4,
  
  // 错误处理模式
  stopOnFirstFailure: false,
  
  // 详细输出模式
  verbose: true,
  
  // 覆盖率设置
  collectCoverage: true,
  coverageThreshold: {
    global: {
      branches: 80,
      functions: 80,
      lines: 80,
      statements: 80
    }
  }
};

/**
 * 测试模块元信息
 */
const TEST_MODULES = [
  {
    name: 'LaTeXParser',
    description: '主控制器 - 五层解析流水线协调',
    file: './LaTeXParser.spec.js',
    layer: 0,
    priority: 1
  },
  {
    name: 'DocumentParser',
    description: '文档结构解析层 (L1) - 章节、标题、文档环境',
    file: './DocumentParser.spec.js', 
    layer: 1,
    priority: 2
  },
  {
    name: 'BlockElementParser',
    description: '块级结构解析层 (L2) - 列表、表格、浮动环境',
    file: './BlockElementParser.spec.js',
    layer: 2, 
    priority: 3
  },
  {
    name: 'InlineElementParser',
    description: '行内元素解析层 (L3) - 文本格式化和修饰',
    file: './InlineElementParser.spec.js',
    layer: 3,
    priority: 4
  },
  {
    name: 'MathParser', 
    description: '数学解析层 (L4) - 数学公式和数学环境',
    file: './MathParser.spec.js',
    layer: 4,
    priority: 5
  },
  {
    name: 'SpecialParser',
    description: '特殊功能解析层 (L5) - 特殊功能和扩展命令',
    file: './SpecialParser.spec.js',
    layer: 5,
    priority: 6
  }
];

/**
 * 获取测试配置
 */
function getTestConfig() {
  return { ...TEST_CONFIG };
}

/**
 * 获取测试模块信息
 */
function getTestModules() {
  return [...TEST_MODULES];
}

/**
 * 运行特定层级的测试
 * @param {number} layer - 解析层级 (0-5)
 */
function runLayerTests(layer) {
  const modules = TEST_MODULES.filter(m => m.layer === layer);
  if (modules.length === 0) {
    console.warn(`No test modules found for layer ${layer}`);
    return;
  }
  
  console.log(`Running tests for Layer ${layer}:`);
  modules.forEach(module => {
    console.log(`  - ${module.name}: ${module.description}`);
  });
}

/**
 * 运行所有测试的统计信息
 */
function getTestStatistics() {
  return {
    totalModules: TEST_MODULES.length,
    layersCount: 6, // 0-5层
    estimatedTestCount: TEST_MODULES.length * 15, // 估算每个模块约15个测试
    configuration: TEST_CONFIG
  };
}

/**
 * 验证测试环境
 */
function validateTestEnvironment() {
  const issues = [];
  
  // 检查必要的依赖
  try {
    require('../LaTeXParser');
  } catch (error) {
    issues.push('LaTeXParser 模块无法加载');
  }
  
  try {
    require('../DocumentParser');
  } catch (error) {
    issues.push('DocumentParser 模块无法加载');
  }
  
  try {
    require('../BlockElementParser');
  } catch (error) {
    issues.push('BlockElementParser 模块无法加载');
  }
  
  try {
    require('../InlineElementParser');
  } catch (error) {
    issues.push('InlineElementParser 模块无法加载');
  }
  
  try {
    require('../MathParser');
  } catch (error) {
    issues.push('MathParser 模块无法加载');
  }
  
  try {
    require('../SpecialParser');
  } catch (error) {
    issues.push('SpecialParser 模块无法加载');
  }
  
  return {
    valid: issues.length === 0,
    issues: issues
  };
}

module.exports = {
  // 配置和元信息
  getTestConfig,
  getTestModules,
  getTestStatistics,
  
  // 工具函数
  runLayerTests,
  validateTestEnvironment,
  
  // 常量
  TEST_CONFIG,
  TEST_MODULES
};