/**
 * SpecialParser 测试
 * 
 * 基于 interface.md 规范的第5层解析器测试
 * 测试特殊功能和扩展命令处理功能
 */

const SpecialParser = require('../SpecialParser');

describe('SpecialParser - 特殊功能解析层 (L5)', () => {
  let parser;
  
  beforeEach(() => {
    parser = new SpecialParser();
  });
  
  describe('核心方法接口', () => {
    test('应该具备 process 方法', () => {
      expect(typeof parser.process).toBe('function');
    });
    
    test('应该返回符合 LayerResult 格式的结果', () => {
      const result = parser.process('\\usepackage{amsmath}');
      
      // 验证 LayerResult 格式
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('placeholders');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.text).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(Array.isArray(result.placeholders)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该支持传入已有占位符数组', () => {
      const existingPlaceholders = [
        {
          id: '__L4_MATH_1__',
          content: '$x^2$',
          type: 'MATH',
          layer: 4
        }
      ];
      
      const result = parser.process('\\footnote{test}', existingPlaceholders);
      
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('placeholders');
    });
  });
  
  describe('包引用处理', () => {
    test('应该处理 \\usepackage 命令', () => {
      const text = '\\usepackage{amsmath}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查包引用是否被正确处理
      const hasPackage = result.blocks.some(block => 
        block.type === 'package' || JSON.stringify(block).includes('amsmath')
      ) || result.placeholders.some(p => p.content.includes('usepackage'));
      expect(hasPackage || result.text.includes('amsmath')).toBe(true);
    });
    
    test('应该处理带选项的包引用', () => {
      const text = '\\usepackage[utf8]{inputenc}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理多个包引用', () => {
      const text = `
        \\usepackage{amsmath}
        \\usepackage{graphicx}
        \\usepackage[T1]{fontenc}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理包引用列表', () => {
      const text = '\\usepackage{amsmath,amssymb,amsfonts}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('图片插入处理', () => {
    test('应该处理 \\includegraphics 命令', () => {
      const text = '\\includegraphics{image.png}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查图片插入是否被正确处理
      const hasImage = result.blocks.some(block => 
        block.type === 'image' || JSON.stringify(block).includes('image.png')
      ) || result.placeholders.some(p => p.content.includes('includegraphics'));
      expect(hasImage || result.text.includes('image')).toBe(true);
    });
    
    test('应该处理带参数的图片插入', () => {
      const text = '\\includegraphics[width=0.5\\textwidth]{figure.jpg}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的图片参数', () => {
      const text = '\\includegraphics[width=5cm,height=3cm,angle=90]{rotated.pdf}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理图片路径', () => {
      const text = '\\includegraphics{./images/subfolder/photo.png}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('脚注系统处理', () => {
    test('应该处理 \\footnote 命令', () => {
      const text = 'Main text\\footnote{This is a footnote} continues.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查脚注是否被正确处理
      const hasFootnote = result.blocks.some(block => 
        block.type === 'footnote' || JSON.stringify(block).includes('footnote')
      ) || result.placeholders.some(p => p.content.includes('footnote'));
      expect(hasFootnote || result.text.includes('footnote')).toBe(true);
    });
    
    test('应该处理多个脚注', () => {
      const text = 'First\\footnote{First note} and second\\footnote{Second note}.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的脚注内容', () => {
      const text = 'Text\\footnote{Footnote with \\textbf{bold} and $x^2$ math}.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理脚注标记', () => {
      const text = '\\footnotemark[5] Text \\footnotetext[5]{Marked footnote}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('交叉引用处理', () => {
    test('应该处理 \\cite 命令', () => {
      const text = 'According to Smith\\cite{smith2024}, the result is clear.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查引用是否被正确处理
      const hasCite = result.blocks.some(block => 
        block.type === 'citation' || JSON.stringify(block).includes('smith2024')
      ) || result.placeholders.some(p => p.content.includes('cite'));
      expect(hasCite || result.text.includes('smith2024')).toBe(true);
    });
    
    test('应该处理 \\ref 命令', () => {
      const text = 'See equation \\ref{eq:important} for details.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\label 命令', () => {
      const text = '\\section{Introduction}\\label{sec:intro}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理多种引用类型', () => {
      const text = `
        See \\cite{author1,author2} and equation \\ref{eq:1}.
        Figure \\ref{fig:graph} shows \\pageref{sec:results}.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理引用选项', () => {
      const text = '\\cite[p.~42]{smith2024} and \\cite[see][]{jones2023}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('自定义命令处理', () => {
    test('应该处理 \\newcommand 定义', () => {
      const text = '\\newcommand{\\mycommand}[1]{\\textbf{#1}}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查自定义命令是否被处理
      const hasNewCommand = result.blocks.some(block => 
        block.type === 'newcommand' || JSON.stringify(block).includes('mycommand')
      ) || result.placeholders.some(p => p.content.includes('newcommand'));
      expect(hasNewCommand || result.text.includes('mycommand')).toBe(true);
    });
    
    test('应该处理带参数的自定义命令', () => {
      const text = '\\newcommand{\\vector}[2]{\\mathbf{#1}_{#2}}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\renewcommand', () => {
      const text = '\\renewcommand{\\abstractname}{Summary}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\providecommand', () => {
      const text = '\\providecommand{\\email}[1]{\\texttt{#1}}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('占位符机制', () => {
    test('应该创建符合格式的占位符', () => {
      const text = '\\footnote{test footnote}';
      const result = parser.process(text);
      
      // 检查占位符格式：__L{layer}_{TYPE}_{index}__
      result.placeholders.forEach(placeholder => {
        expect(placeholder).toHaveProperty('id');
        expect(placeholder).toHaveProperty('content');
        expect(placeholder).toHaveProperty('type');
        expect(placeholder).toHaveProperty('layer');
        
        // 验证 ID 格式
        expect(placeholder.id).toMatch(/^__L\d+_\w+_\d+__$/);
        expect(placeholder.layer).toBe(5); // SpecialParser 是第5层
      });
    });
    
    test('应该保护现有占位符', () => {
      const existingPlaceholders = [
        {
          id: '__L4_MATH_1__',
          content: '$x^2$',
          type: 'MATH',
          layer: 4
        }
      ];
      
      const text = 'Text with __L4_MATH_1__ and \\footnote{note}';
      const result = parser.process(text, existingPlaceholders);
      
      // 现有占位符应该被保护
      expect(result.text).toContain('__L4_MATH_1__');
    });
  });
  
  describe('复杂特殊功能混合', () => {
    test('应该处理包含多种特殊功能的文档', () => {
      const text = `
        \\usepackage{graphicx}
        \\usepackage{amsmath}
        
        \\newcommand{\\myref}[1]{\\ref{#1}}
        
        Text with citation \\cite{author2024} and footnote\\footnote{Important note}.
        
        \\includegraphics[width=0.5\\textwidth]{figure1.png}
        
        See section \\ref{sec:results} for details.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      expect(result.blocks.length + result.placeholders.length).toBeGreaterThan(0);
    });
    
    test('应该处理嵌套的特殊命令', () => {
      const text = '\\footnote{See \\cite{ref} and \\includegraphics{img.png}}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('错误处理', () => {
    test('应该处理未闭合的特殊命令', () => {
      const text = '\\footnote{unclosed footnote';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理无效的包名', () => {
      const text = '\\usepackage{nonexistentpackage}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理无效的图片路径', () => {
      const text = '\\includegraphics{/invalid/path/image.png}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该继续处理，可能有警告
    });
    
    test('应该处理空的脚注', () => {
      const text = 'Text\\footnote{} continues.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理无效的引用格式', () => {
      const text = '\\cite{} and \\ref{}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该继续处理，可能有警告
    });
  });
  
  describe('边界条件', () => {
    test('应该处理空字符串', () => {
      const result = parser.process('');
      
      expect(result.text).toBe('');
      expect(result.blocks).toHaveLength(0);
      expect(result.placeholders).toHaveLength(0);
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理不包含特殊命令的纯文本', () => {
      const text = 'This is just plain text without any special LaTeX commands.';
      const result = parser.process(text);
      
      expect(result.text).toContain('plain text');
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理超长的特殊命令参数', () => {
      const longNote = 'Very '.repeat(200) + 'long footnote text';
      const text = `\\footnote{${longNote}}`;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理大量的引用', () => {
      const manyRefs = Array.from({length: 20}, (_, i) => `\\cite{ref${i}}`).join(' ');
      const result = parser.process(manyRefs);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的自定义命令', () => {
      const complexCommand = '\\newcommand{\\complex}[3][default]{#1: \\textbf{#2} = \\textit{#3}}';
      const result = parser.process(complexCommand);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理Unicode内容的特殊命令', () => {
      const unicodeText = '\\footnote{中文脚注 العربية Русский}';
      const result = parser.process(unicodeText);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      expect(result.text.includes('中文') || result.placeholders.some(p => p.content.includes('中文'))).toBe(true);
    });
    
    test('应该处理网络图片URL', () => {
      const text = '\\includegraphics{https://example.com/image.png}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
});