# LaTeX Parser V3 Core 模块测试套件

## 概述

本测试套件基于 `interface.md` 规范，为 LaTeX Parser V3 的核心模块提供完整的接口测试覆盖。测试套件遵循五层解析架构，确保每个解析层的功能正确性和接口一致性。

## 测试架构

### 五层解析器测试结构

```
LaTeX Parser V3 Core Tests
├── LaTeXParser.spec.js      (L0) - 主控制器测试
├── DocumentParser.spec.js   (L1) - 文档结构解析层测试
├── BlockElementParser.spec.js (L2) - 块级结构解析层测试  
├── InlineElementParser.spec.js (L3) - 行内元素解析层测试
├── MathParser.spec.js       (L4) - 数学解析层测试
├── SpecialParser.spec.js    (L5) - 特殊功能解析层测试
├── index.js                 - 测试套件入口和配置
└── README.md               - 本文档
```

### 测试覆盖范围

每个测试文件覆盖以下核心方面：

1. **接口规范验证** - 基于 `interface.md` 的严格接口测试
2. **LayerResult 格式验证** - 统一的输入输出格式验证
3. **占位符机制测试** - 层级保护和恢复机制验证
4. **错误处理测试** - 容错模式和严格模式验证
5. **边界条件测试** - 空输入、特殊字符、性能边界
6. **功能完整性测试** - 各层特定功能的完整覆盖

## 测试规范

### 接口契约验证

每个解析层都必须遵循统一的接口契约：

```javascript
// 输入接口
process(text: String, existingPlaceholders: Array = []): LayerResult

// 输出格式 (LayerResult)
{
  text: String,              // 处理后的文本
  blocks: Array,            // 该层产生的结构化数据块
  placeholders: Array,      // 该层创建的占位符
  warnings: Array,          // 处理警告
  errors: Array            // 处理错误
}

// 占位符格式
{
  id: String,               // 格式: "__L{layer}_{TYPE}_{index}__"
  content: String,          // 原始内容
  type: String,            // 占位符类型
  layer: Number            // 所属层级 (1-5)
}
```

### 测试分类

#### 1. LaTeXParser (主控制器) - 59个测试用例
- **配置测试** - 构造函数和配置参数验证
- **ParseResult格式测试** - 输出格式完整性验证
- **流水线集成测试** - 五层解析流程验证
- **错误处理模式测试** - tolerant/strict模式验证
- **性能边界测试** - 大文档处理和超时测试

#### 2. DocumentParser (L1) - 45个测试用例
- **章节命令测试** - `\section`, `\subsection`, `\chapter`等
- **文档元数据测试** - `\title`, `\author`, `\date`
- **文档环境测试** - `\begin{document}...\end{document}`
- **数学环境保护测试** - `$$...$$` 块保护机制
- **占位符创建测试** - L1层占位符格式验证

#### 3. BlockElementParser (L2) - 47个测试用例
- **列表环境测试** - `itemize`, `enumerate`, `description`
- **表格环境测试** - `tabular`, `array`
- **图片环境测试** - `figure`
- **引用环境测试** - `quote`, `quotation`
- **嵌套结构测试** - 深度嵌套和复杂组合

#### 4. InlineElementParser (L3) - 51个测试用例  
- **文本格式测试** - `\textbf`, `\textit`, `\emph`, `\underline`
- **字号命令测试** - `\large`, `\small`, `\tiny`
- **特殊字符转义测试** - LaTeX特殊字符处理
- **换行命令测试** - `\\`, `\newline`
- **格式嵌套测试** - 复杂格式组合

#### 5. MathParser (L4) - 53个测试用例
- **行内数学测试** - `$...$`, `\(...\)`
- **显示数学测试** - `$$...$$`, `\[...\]`
- **数学环境测试** - `equation`, `align`, `gather`, `matrix`
- **数学统计验证测试** - 公式计数和语法验证
- **复杂数学表达式测试** - 嵌套和多行公式

#### 6. SpecialParser (L5) - 49个测试用例
- **包引用测试** - `\usepackage`
- **图片插入测试** - `\includegraphics`
- **脚注系统测试** - `\footnote`
- **交叉引用测试** - `\cite`, `\ref`, `\label`
- **自定义命令测试** - `\newcommand`, `\renewcommand`

## 运行测试

### 前置条件

确保在 `lib/latex-parser-v3/core` 目录下运行测试：

```bash
cd /mnt/d/sycdata/gcf/md2pdf/lib/latex-parser-v3/core
```

### 运行完整测试套件

```bash
# 运行所有核心模块测试
npm test tests/

# 或使用 Jest 直接运行
npx jest tests/
```

### 运行单个测试文件

```bash
# 测试主控制器
npx jest tests/LaTeXParser.spec.js

# 测试特定解析层
npx jest tests/DocumentParser.spec.js
npx jest tests/BlockElementParser.spec.js
npx jest tests/InlineElementParser.spec.js
npx jest tests/MathParser.spec.js
npx jest tests/SpecialParser.spec.js
```

### 带覆盖率的测试

```bash
# 生成覆盖率报告
npx jest tests/ --coverage

# 详细覆盖率报告
npx jest tests/ --coverage --verbose
```

### 监视模式运行

```bash
# 监视文件变化自动重新运行测试
npx jest tests/ --watch
```

## 测试配置

测试套件使用以下配置 (在 `index.js` 中定义)：

```javascript
const TEST_CONFIG = {
  timeout: 10000,           // 10秒测试超时
  maxConcurrency: 4,        // 最大并发数
  stopOnFirstFailure: false, // 不在首个失败时停止
  verbose: true,            // 详细输出
  collectCoverage: true,    // 收集覆盖率
  coverageThreshold: {
    global: {
      branches: 80,         // 分支覆盖率 ≥ 80%
      functions: 80,        // 函数覆盖率 ≥ 80%
      lines: 80,           // 行覆盖率 ≥ 80%
      statements: 80       // 语句覆盖率 ≥ 80%
    }
  }
}
```

## 测试环境验证

在运行测试前，可以验证测试环境：

```javascript
const testSuite = require('./tests/index.js');
const validation = testSuite.validateTestEnvironment();

if (!validation.valid) {
  console.error('测试环境问题:', validation.issues);
}
```

## 性能基准

测试套件包含性能基准测试：

- **单个解析器处理时间** < 100ms
- **完整文档处理时间** < 1000ms  
- **大型文档 (10000+ 字符)** < 5000ms
- **内存使用** < 50MB

## 质量保证

### 覆盖率要求

- **整体代码覆盖率** ≥ 80%
- **关键路径覆盖率** ≥ 95%
- **错误处理覆盖率** ≥ 90%

### 测试质量检查

1. **接口一致性** - 所有解析器遵循相同的接口规范
2. **数据格式验证** - LayerResult 和 Placeholder 格式严格验证
3. **错误处理完整性** - 各种错误场景的处理验证
4. **边界条件覆盖** - 空输入、极值、异常情况
5. **性能退化检测** - 性能回归测试

## 持续集成

测试套件支持 CI/CD 集成：

```bash
# CI 环境运行
npm test -- --ci --coverage --watchAll=false

# 生成 JUnit 报告
npm test -- --ci --coverage --testResultsProcessor=jest-junit
```

## 故障排查

### 常见问题

1. **模块加载失败**
   ```bash
   # 确保在正确目录下
   cd /mnt/d/sycdata/gcf/md2pdf/lib/latex-parser-v3/core
   
   # 检查文件是否存在
   ls -la *.js
   ```

2. **测试超时**
   ```javascript
   // 增加超时时间
   jest.setTimeout(20000);
   ```

3. **覆盖率不足**
   ```bash
   # 查看覆盖率详情
   npx jest --coverage --verbose
   ```

### 调试模式

```bash
# 调试特定测试
node --inspect-brk node_modules/.bin/jest tests/LaTeXParser.spec.js --runInBand

# 只运行失败的测试
npx jest --onlyFailures
```

## 扩展和维护

### 添加新测试

1. 遵循现有的测试结构模式
2. 确保覆盖接口规范的所有方面
3. 包含错误处理和边界条件测试
4. 更新本文档的测试统计信息

### 维护检查清单

- [ ] 所有测试都通过
- [ ] 覆盖率达到要求 (≥80%)
- [ ] 性能测试在预期范围内
- [ ] 接口变更时更新相应测试
- [ ] 新功能添加对应测试用例

## 统计信息

- **总测试文件数**: 6个
- **预估测试用例总数**: 304个
- **测试层级覆盖**: L0-L5 (6层)
- **接口验证点**: 36个核心接口方法
- **支持的LaTeX命令**: 50+ 种命令类型

---

基于 interface.md 规范 - LaTeX Parser V3 Core 测试套件
版本: 1.0.0 | 最后更新: 2024-09-03