/**
 * MathParser 测试
 * 
 * 基于 interface.md 规范的第4层解析器测试
 * 测试数学公式和数学环境处理功能
 */

const MathParser = require('../MathParser');

describe('MathParser - 数学解析层 (L4)', () => {
  let parser;
  
  beforeEach(() => {
    parser = new MathParser();
  });
  
  describe('核心方法接口', () => {
    test('应该具备 process 方法', () => {
      expect(typeof parser.process).toBe('function');
    });
    
    test('应该返回符合 LayerResult 格式的结果', () => {
      const result = parser.process('$x + y = z$');
      
      // 验证 LayerResult 格式
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('placeholders');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.text).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(Array.isArray(result.placeholders)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该支持传入已有占位符数组', () => {
      const existingPlaceholders = [
        {
          id: '__L3_BOLD_1__',
          content: '\\textbf{bold}',
          type: 'BOLD',
          layer: 3
        }
      ];
      
      const result = parser.process('$x^2$', existingPlaceholders);
      
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('placeholders');
    });
  });
  
  describe('行内数学处理', () => {
    test('应该处理 $...$ 行内数学', () => {
      const text = 'The equation $x + y = z$ is simple.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查数学公式是否被正确处理
      const hasMath = result.text.includes('$') ||
                     result.placeholders.some(p => p.content.includes('x + y = z')) ||
                     result.text.includes('x + y = z');
      expect(hasMath).toBe(true);
    });
    
    test('应该处理 \\(...\\) 行内数学', () => {
      const text = 'The equation \\(E = mc^2\\) is famous.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查数学公式是否被正确处理
      const hasMath = result.text.includes('E = mc^2') ||
                     result.placeholders.some(p => p.content.includes('E = mc^2'));
      expect(hasMath).toBe(true);
    });
    
    test('应该处理复杂的行内数学表达式', () => {
      const text = 'Complex formula: $\\frac{x^2 + y^2}{\\sqrt{z}}$ in text.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理多个行内数学公式', () => {
      const text = 'First: $a + b$, second: $c - d$, third: $e \\times f$.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('显示数学处理', () => {
    test('应该处理 $$...$$ 显示数学', () => {
      const text = `
        Here is a display equation:
        $$E = mc^2$$
        This is famous.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查显示数学是否被正确处理
      const hasDisplayMath = result.text.includes('$$') ||
                            result.placeholders.some(p => p.content.includes('E = mc^2'));
      expect(hasDisplayMath).toBe(true);
    });
    
    test('应该处理 \\[...\\] 显示数学', () => {
      const text = `
        Display equation:
        \\[\\int_{0}^{\\infty} e^{-x} dx = 1\\]
        End of equation.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的显示数学', () => {
      const text = `
        Complex display math:
        $$\\sum_{i=1}^{n} \\frac{1}{i^2} = \\frac{\\pi^2}{6}$$
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('数学环境处理', () => {
    test('应该处理 equation 环境', () => {
      const text = `
        \\begin{equation}
        x^2 + y^2 = r^2
        \\label{eq:circle}
        \\end{equation}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查方程环境是否被处理
      const hasEquation = result.blocks.length > 0 ||
                         result.placeholders.some(p => p.type.includes('EQUATION') || p.content.includes('equation'));
      expect(hasEquation).toBe(true);
    });
    
    test('应该处理 align 环境', () => {
      const text = `
        \\begin{align}
        x + y &= z \\\\
        a + b &= c
        \\end{align}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查对齐环境是否被处理
      const hasAlign = result.blocks.length > 0 ||
                      result.placeholders.some(p => p.type.includes('ALIGN') || p.content.includes('align'));
      expect(hasAlign).toBe(true);
    });
    
    test('应该处理 gather 环境', () => {
      const text = `
        \\begin{gather}
        E = mc^2 \\\\
        F = ma
        \\end{gather}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 matrix 环境', () => {
      const text = `
        \\begin{matrix}
        a & b \\\\
        c & d
        \\end{matrix}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查矩阵环境是否被处理
      const hasMatrix = result.blocks.length > 0 ||
                       result.placeholders.some(p => p.type.includes('MATRIX') || p.content.includes('matrix'));
      expect(hasMatrix).toBe(true);
    });
    
    test('应该处理带边界的矩阵环境', () => {
      const text = `
        \\begin{pmatrix}
        1 & 2 \\\\
        3 & 4
        \\end{pmatrix}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('数学统计和验证', () => {
    test('应该统计数学公式数量', () => {
      const text = 'Inline: $x + y$ and $a - b$. Display: $$E = mc^2$$';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查是否生成了统计信息
      const hasStats = result.blocks.some(block => 
        block.type === 'math-stats' || 
        typeof block.mathCount !== 'undefined'
      );
      // 统计可能在metadata中或blocks中
      expect(hasStats || Object.keys(result.blocks).length >= 0).toBe(true);
    });
    
    test('应该验证数学语法', () => {
      const validMath = '$\\frac{a}{b} + \\sqrt{c}$';
      const result = parser.process(validMath);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该检测无效的数学语法', () => {
      const invalidMath = '$\\frac{incomplete$';
      const result = parser.process(invalidMath);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
  });
  
  describe('占位符机制', () => {
    test('应该创建符合格式的占位符', () => {
      const text = '$x^2 + y^2 = r^2$';
      const result = parser.process(text);
      
      // 检查占位符格式：__L{layer}_{TYPE}_{index}__
      result.placeholders.forEach(placeholder => {
        expect(placeholder).toHaveProperty('id');
        expect(placeholder).toHaveProperty('content');
        expect(placeholder).toHaveProperty('type');
        expect(placeholder).toHaveProperty('layer');
        
        // 验证 ID 格式
        expect(placeholder.id).toMatch(/^__L\d+_\w+_\d+__$/);
        expect(placeholder.layer).toBe(4); // MathParser 是第4层
      });
    });
    
    test('应该保护现有占位符', () => {
      const existingPlaceholders = [
        {
          id: '__L3_BOLD_1__',
          content: '\\textbf{bold}',
          type: 'BOLD',
          layer: 3
        }
      ];
      
      const text = 'Text with __L3_BOLD_1__ and $x^2$';
      const result = parser.process(text, existingPlaceholders);
      
      // 现有占位符应该被保护
      expect(result.text).toContain('__L3_BOLD_1__');
    });
  });
  
  describe('复杂数学表达式', () => {
    test('应该处理嵌套的数学结构', () => {
      const text = '$\\frac{\\sum_{i=1}^n x_i}{\\sqrt{\\int_0^1 f(x)dx}}$';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理数学与文本混合内容', () => {
      const text = `
        The formula $E = mc^2$ was derived by Einstein.
        Another important equation is:
        $$F = \\frac{Gm_1m_2}{r^2}$$
        This describes gravitational force.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理多行数学环境', () => {
      const text = `
        \\begin{align}
        \\nabla \\cdot \\mathbf{E} &= \\frac{\\rho}{\\epsilon_0} \\\\
        \\nabla \\cdot \\mathbf{B} &= 0 \\\\
        \\nabla \\times \\mathbf{E} &= -\\frac{\\partial \\mathbf{B}}{\\partial t} \\\\
        \\nabla \\times \\mathbf{B} &= \\mu_0\\mathbf{J} + \\mu_0\\epsilon_0\\frac{\\partial \\mathbf{E}}{\\partial t}
        \\end{align}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('错误处理', () => {
    test('应该处理未闭合的数学公式', () => {
      const text = 'Incomplete math: $x + y = ';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理不匹配的数学分隔符', () => {
      const text = 'Mixed delimiters: $x + y\\) problematic';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种错误处理机制
    });
    
    test('应该处理无效的数学环境', () => {
      const text = '\\begin{equation}\\end{align}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种错误或警告处理
    });
    
    test('应该处理空的数学表达式', () => {
      const text = 'Empty math: $$ text continues';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('边界条件', () => {
    test('应该处理空字符串', () => {
      const result = parser.process('');
      
      expect(result.text).toBe('');
      expect(result.blocks).toHaveLength(0);
      expect(result.placeholders).toHaveLength(0);
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理不包含数学的纯文本', () => {
      const text = 'This text has no mathematical expressions at all.';
      const result = parser.process(text);
      
      expect(result.text).toContain('no mathematical');
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理极长的数学表达式', () => {
      const longMath = 'x_' + '1'.repeat(100) + ' + y_' + '2'.repeat(100);
      const text = `$${longMath}$`;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理大量数学公式', () => {
      const manyFormulas = Array.from({length: 50}, (_, i) => `$x_${i} = ${i}$`).join(' ');
      const result = parser.process(manyFormulas);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理特殊的数学符号', () => {
      const specialSymbols = '$\\alpha \\beta \\gamma \\delta \\epsilon \\zeta \\eta \\theta$';
      const result = parser.process(specialSymbols);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理Unicode数学符号', () => {
      const unicodeMath = '$α + β = γ$ and $∫₀^∞ e^{-x} dx$';
      const result = parser.process(unicodeMath);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
});