/**
 * MathParser 矩阵环境修复测试
 *
 * 测试修复后的wrapRemainingMathEnvironments方法
 * 确保行内公式中的多个矩阵环境不会被错误分割
 */

const MathParser = require('../MathParser');

describe('MathParser - 矩阵环境修复测试', () => {
  let parser;

  beforeEach(() => {
    parser = new MathParser();
  });

  describe('wrapRemainingMathEnvironments 矩阵处理', () => {
    test('应该为独立的矩阵环境添加$$包装', () => {
      const text = '这是一个独立的矩阵：\\begin{matrix} a & b \\\\ c & d \\end{matrix} 结束';
      const result = parser.wrapRemainingMathEnvironments(text);

      expect(result).toBe('这是一个独立的矩阵：$$\\begin{matrix} a & b \\\\ c & d \\end{matrix}$$ 结束');
    });

    test('应该为各种类型的独立矩阵环境添加$$包装', () => {
      const testCases = [
        { env: 'matrix', text: '\\begin{matrix} a & b \\\\ c & d \\end{matrix}' },
        { env: 'pmatrix', text: '\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}' },
        { env: 'bmatrix', text: '\\begin{bmatrix} x & y \\\\ z & w \\end{bmatrix}' },
        { env: 'vmatrix', text: '\\begin{vmatrix} a & b \\\\ c & d \\end{vmatrix}' },
        { env: 'Vmatrix', text: '\\begin{Vmatrix} 1 & 2 \\\\ 3 & 4 \\end{Vmatrix}' }
      ];

      testCases.forEach(({ env, text }) => {
        const result = parser.wrapRemainingMathEnvironments(text);
        expect(result).toBe(`$$${text}$$`);
      });
    });

    test('不应该为行内公式中的矩阵环境添加$$包装', () => {
      const text = '公式：$\\begin{matrix} a & b \\\\ c & d \\end{matrix} \\times \\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$ 结果';
      const result = parser.wrapRemainingMathEnvironments(text);

      // 应该保持原样，不添加额外的$$包装
      expect(result).toBe(text);
    });

    test('不应该为显示数学中的矩阵环境添加$$包装', () => {
      const testCases = [
        { text: '\\[\\begin{matrix} a & b \\\\ c & d \\end{matrix}\\]', expected: '\\[\\begin{matrix} a & b \\\\ c & d \\end{matrix}\\]' },
        { text: '$$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$', expected: '$$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$' }
      ];

      testCases.forEach(({ text, expected }) => {
        const result = parser.wrapRemainingMathEnvironments(text);
        expect(result).toBe(expected);
      });
    });

    test('应该正确处理混合情况', () => {
      const text = '独立矩阵：\\begin{matrix} a & b \\\\ c & d \\end{matrix} 和行内矩阵：$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$';
      const result = parser.wrapRemainingMathEnvironments(text);

      // 只有独立的矩阵应该被包装，行内公式中的矩阵保持原样
      expect(result).toBe('独立矩阵：$$\\begin{matrix} a & b \\\\ c & d \\end{matrix}$$ 和行内矩阵：$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$');
    });

    test('应该处理多个独立的矩阵环境', () => {
      const text = '第一个：\\begin{matrix} a & b \\\\ c & d \\end{matrix} 第二个：\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}';
      const result = parser.wrapRemainingMathEnvironments(text);

      expect(result).toBe('第一个：$$\\begin{matrix} a & b \\\\ c & d \\end{matrix}$$ 第二个：$$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$');
    });

    test('应该处理行内公式中的多个矩阵而不分割', () => {
      const text = '$\\begin{matrix} a \\\\ b \\end{matrix} + \\begin{pmatrix} c \\\\ d \\end{pmatrix} = \\begin{bmatrix} a+c \\\\ b+d \\end{bmatrix}$';
      const result = parser.wrapRemainingMathEnvironments(text);

      // 整个行内公式应该保持完整，不被分割
      expect(result).toBe(text);
    });

    test('应该处理复杂的嵌套情况', () => {
      const text = `
        独立矩阵：
        \\begin{vmatrix}
        \\alpha & \\beta \\\\
        \\gamma & \\delta
        \\end{vmatrix}

        行内公式：
        $\\begin{pmatrix} x_1 \\\\ x_2 \\end{pmatrix} = \\begin{bmatrix} a & b \\\\ c & d \\end{bmatrix} \\begin{pmatrix} y_1 \\\\ y_2 \\end{pmatrix}$

        显示公式：
        \\[\\begin{Vmatrix} 1 & 0 \\\\ 0 & 1 \\end{Vmatrix} = 1\\]
      `;

      const result = parser.wrapRemainingMathEnvironments(text);

      // 检查独立矩阵被包装
      expect(result).toContain('$$\\begin{vmatrix}');
      expect(result).toContain('\\end{vmatrix}$$');

      // 检查行内公式保持完整
      expect(result).toContain('$\\begin{pmatrix}');
      expect(result).toContain('\\begin{bmatrix}');
      expect(result).toContain('\\end{pmatrix}$');

      // 检查显示公式保持不变
      expect(result).toContain('\\[\\begin{Vmatrix}');
      expect(result).toContain('\\end{Vmatrix} = 1\\]');
    });

    test('应该处理边界情况', () => {
      // 空字符串
      expect(parser.wrapRemainingMathEnvironments('')).toBe('');

      // 只有文本
      expect(parser.wrapRemainingMathEnvironments('纯文本，没有矩阵')).toBe('纯文本，没有矩阵');

      // 矩阵在文档开头
      const text1 = '\\begin{matrix} a & b \\end{matrix} 开头';
      expect(parser.wrapRemainingMathEnvironments(text1)).toBe('$$\\begin{matrix} a & b \\end{matrix}$$ 开头');

      // 矩阵在文档末尾
      const text2 = '末尾 \\begin{pmatrix} 1 & 2 \\end{pmatrix}';
      expect(parser.wrapRemainingMathEnvironments(text2)).toBe('末尾 $$\\begin{pmatrix} 1 & 2 \\end{pmatrix}$$');
    });

    test('应该处理不完整的矩阵环境', () => {
      const text = '不完整的：\\begin{matrix} a & b';
      const result = parser.wrapRemainingMathEnvironments(text);

      // 不完整的矩阵环境不应该被处理
      expect(result).toBe(text);
    });
  });

  describe('完整处理流程测试', () => {
    test('process方法应该正确处理包含矩阵的文本', () => {
      const text = `
        这是一个包含矩阵的文档：

        独立矩阵：
        \\begin{matrix} 1 & 2 \\\\ 3 & 4 \\end{matrix}

        行内公式：
        $\\begin{pmatrix} x \\\\ y \\end{pmatrix} = \\begin{bmatrix} a & b \\\\ c & d \\end{bmatrix} \\begin{pmatrix} u \\\\ v \\end{pmatrix}$

        显示公式：
        $$\\det\\begin{vmatrix} \\alpha & \\beta \\\\ \\gamma & \\delta \\end{vmatrix} = \\alpha\\delta - \\beta\\gamma$$
      `;

      const result = parser.process(text);

      expect(result.text).toContain('$$\\begin{matrix}'); // 独立矩阵被包装
      expect(result.text).toContain('$\\begin{pmatrix}'); // 行内公式保持完整
      expect(result.errors).toHaveLength(0);
    });
  });
});