/**
 * LaTeXParser 核心控制器测试
 * 
 * 基于 interface.md 规范的完整接口测试
 * 测试主控制器的配置、解析流程和输出格式
 */

const LaTeXParser = require('../LaTeXParser');

describe('LaTeXParser - 主控制器', () => {
  
  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const parser = new LaTeXParser();
      
      expect(parser).toBeDefined();
      expect(typeof parser.parseComplete).toBe('function');
    });
    
    test('应该接受自定义配置参数', () => {
      const config = {
        mode: 'strict',
        enabledLayers: ['document', 'block', 'inline'],
        maxRecursionDepth: 50
      };
      
      const parser = new LaTeXParser(config);
      
      expect(parser).toBeDefined();
    });
    
    test('应该处理空配置对象', () => {
      const parser = new LaTeXParser({});
      
      expect(parser).toBeDefined();
      expect(typeof parser.parseComplete).toBe('function');
    });
  });
  
  describe('parseComplete 方法', () => {
    let parser;
    
    beforeEach(() => {
      parser = new LaTeXParser();
    });
    
    test('应该返回符合 ParseResult 格式的结果', () => {
      const result = parser.parseComplete('Hello World');
      
      // 验证 ParseResult 格式
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('processedText');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.success).toBe('boolean');
      expect(typeof result.processedText).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(typeof result.metadata).toBe('object');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该处理空字符串输入', () => {
      const result = parser.parseComplete('');
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBe('');
      expect(result.blocks).toHaveLength(0);
    });
    
    test('应该处理纯文本内容', () => {
      const text = 'This is plain text without LaTeX commands';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('This is plain text');
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理基本LaTeX命令', () => {
      const text = '\\textbf{Bold text} and \\textit{italic text}';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理数学公式', () => {
      const text = 'Inline math: $x + y = z$ and display math: $$E = mc^2$$';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBeTruthy();
    });
    
    test('应该处理章节结构', () => {
      const text = '\\section{Introduction}\\nSome content here.';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.blocks.length).toBeGreaterThan(0);
    });
  });
  
  describe('错误处理模式', () => {
    test('容错模式应该继续处理并生成警告', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      const text = '\\unknowncommand{test} normal text';
      
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBeTruthy();
      // 在容错模式下可能有警告，但不应有致命错误
    });
    
    test('严格模式应该在遇到错误时提供详细信息', () => {
      const parser = new LaTeXParser({ mode: 'strict' });
      const text = '\\section{Valid} \\invalidcommand{test}';
      
      const result = parser.parseComplete(text);
      
      // 严格模式下的行为可能因具体实现而异
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('errors');
    });
  });
  
  describe('配置参数测试', () => {
    test('应该支持启用特定解析层', () => {
      const config = {
        enabledLayers: ['document', 'inline']
      };
      const parser = new LaTeXParser(config);
      
      const text = '\\section{Title} \\textbf{bold} \\begin{itemize}\\item test\\end{itemize}';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
    });
    
    test('应该支持设置最大递归深度', () => {
      const config = {
        maxRecursionDepth: 5
      };
      const parser = new LaTeXParser(config);
      
      const result = parser.parseComplete('\\textbf{test}');
      
      expect(result.success).toBe(true);
    });
  });
  
  describe('复杂文档处理', () => {
    test('应该处理包含多层嵌套的复杂文档', () => {
      const complexText = `
        \\section{Introduction}
        This is a \\textbf{complex} document with \\textit{multiple} elements.
        
        \\begin{itemize}
        \\item First item with $x = y + z$
        \\item Second item with \\emph{emphasis}
        \\end{itemize}
        
        Display math:
        $$\\int_{0}^{\\infty} e^{-x} dx = 1$$
        
        \\subsection{Conclusion}
        Final paragraph.
      `;
      
      const result = parser.parseComplete(complexText);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBeTruthy();
      expect(result.blocks.length).toBeGreaterThan(0);
    });
    
    test('应该处理文档元数据', () => {
      const text = `
        \\title{Test Document}
        \\author{Test Author}
        \\date{2024-01-01}
        \\begin{document}
        Content here.
        \\end{document}
      `;
      
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.metadata).toBeDefined();
    });
  });
  
  describe('性能和边界条件', () => {
    test('应该处理大型文档', () => {
      const largeText = 'Normal text. '.repeat(1000) + '\\textbf{Bold at end}';
      
      const startTime = Date.now();
      const result = parser.parseComplete(largeText);
      const endTime = Date.now();
      
      expect(result.success).toBe(true);
      expect(endTime - startTime).toBeLessThan(5000); // 5秒内完成
    });
    
    test('应该处理特殊字符', () => {
      const text = 'Special chars: & % $ # _ { } ~ ^ \\';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toBeTruthy();
    });
    
    test('应该处理Unicode字符', () => {
      const text = 'Unicode: 中文 العربية Русский ñ';
      const result = parser.parseComplete(text);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('中文');
    });
  });
});