/**
 * InlineElementParser 测试
 * 
 * 基于 interface.md 规范的第3层解析器测试
 * 测试行内格式化和文本修饰功能
 */

const InlineElementParser = require('../InlineElementParser');

describe('InlineElementParser - 行内元素解析层 (L3)', () => {
  let parser;
  
  beforeEach(() => {
    parser = new InlineElementParser();
  });
  
  describe('核心方法接口', () => {
    test('应该具备 process 方法', () => {
      expect(typeof parser.process).toBe('function');
    });
    
    test('应该返回符合 LayerResult 格式的结果', () => {
      const result = parser.process('\\textbf{bold text}');
      
      // 验证 LayerResult 格式
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('placeholders');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.text).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(Array.isArray(result.placeholders)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该支持传入已有占位符数组', () => {
      const existingPlaceholders = [
        {
          id: '__L2_ITEMIZE_1__',
          content: '\\begin{itemize}\\item test\\end{itemize}',
          type: 'ITEMIZE',
          layer: 2
        }
      ];
      
      const result = parser.process('\\textbf{bold text}', existingPlaceholders);
      
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('placeholders');
    });
  });
  
  describe('文本格式处理', () => {
    test('应该处理 \\textbf 命令（粗体）', () => {
      const text = 'This is \\textbf{bold text} in a sentence.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查粗体文本是否被正确处理
      const hasBold = result.text.includes('<strong>') || 
                     result.text.includes('<b>') ||
                     result.placeholders.some(p => p.content.includes('textbf')) ||
                     result.text.includes('bold text');
      expect(hasBold).toBe(true);
    });
    
    test('应该处理 \\textit 命令（斜体）', () => {
      const text = 'This is \\textit{italic text} in a sentence.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查斜体文本是否被正确处理
      const hasItalic = result.text.includes('<em>') || 
                       result.text.includes('<i>') ||
                       result.placeholders.some(p => p.content.includes('textit')) ||
                       result.text.includes('italic text');
      expect(hasItalic).toBe(true);
    });
    
    test('应该处理 \\emph 命令（强调）', () => {
      const text = 'This is \\emph{emphasized text} in a sentence.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查强调文本是否被正确处理
      const hasEmph = result.text.includes('<em>') ||
                     result.placeholders.some(p => p.content.includes('emph')) ||
                     result.text.includes('emphasized text');
      expect(hasEmph).toBe(true);
    });
    
    test('应该处理 \\underline 命令（下划线）', () => {
      const text = 'This is \\underline{underlined text} in a sentence.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查下划线文本是否被正确处理
      const hasUnderline = result.text.includes('<u>') ||
                          result.text.includes('text-decoration') ||
                          result.placeholders.some(p => p.content.includes('underline')) ||
                          result.text.includes('underlined text');
      expect(hasUnderline).toBe(true);
    });
    
    test('应该处理嵌套的文本格式', () => {
      const text = '\\textbf{Bold with \\textit{italic inside} and more bold}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理组合的文本格式', () => {
      const text = '\\textbf{Bold}, \\textit{Italic}, \\emph{Emphasis}, \\underline{Underlined}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('字号命令处理', () => {
    test('应该处理 \\large 命令', () => {
      const text = 'Normal text and \\large{large text} here.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查大字体是否被处理
      const hasLarge = result.text.includes('large') ||
                      result.placeholders.some(p => p.content.includes('large'));
      expect(hasLarge).toBe(true);
    });
    
    test('应该处理 \\small 命令', () => {
      const text = 'Normal text and \\small{small text} here.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\tiny 命令', () => {
      const text = 'Normal text and \\tiny{tiny text} here.';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理多种字号组合', () => {
      const text = '\\large{Large} \\small{Small} \\tiny{Tiny} normal text';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('特殊字符转义处理', () => {
    test('应该处理基本的特殊字符转义', () => {
      const text = 'Special chars: \\& \\% \\$ \\# \\_ \\{ \\} \\~{} \\^{}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查特殊字符是否被正确转义
      const hasSpecialChars = result.text.includes('&') ||
                             result.text.includes('%') ||
                             result.text.includes('$') ||
                             result.text.includes('#');
      expect(hasSpecialChars).toBe(true);
    });
    
    test('应该处理反斜杠转义', () => {
      const text = 'Backslash: \\textbackslash{}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理引号', () => {
      const text = 'Quotes: `single quote\' and ``double quotes\'\'';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('换行命令处理', () => {
    test('应该处理 \\\\\\ 换行命令', () => {
      const text = 'First line\\\\Second line\\\\Third line';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查换行是否被正确处理
      const hasLineBreaks = result.text.includes('<br>') ||
                           result.text.includes('\\n') ||
                           result.text.includes('Second line');
      expect(hasLineBreaks).toBe(true);
    });
    
    test('应该处理 \\newline 命令', () => {
      const text = 'First line\\newline Second line';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理带参数的换行', () => {
      const text = 'Line one\\\\[2em]Line two with extra space';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('占位符机制', () => {
    test('应该创建符合格式的占位符', () => {
      const text = '\\textbf{bold text}';
      const result = parser.process(text);
      
      // 检查占位符格式：__L{layer}_{TYPE}_{index}__
      result.placeholders.forEach(placeholder => {
        expect(placeholder).toHaveProperty('id');
        expect(placeholder).toHaveProperty('content');
        expect(placeholder).toHaveProperty('type');
        expect(placeholder).toHaveProperty('layer');
        
        // 验证 ID 格式
        expect(placeholder.id).toMatch(/^__L\d+_\w+_\d+__$/);
        expect(placeholder.layer).toBe(3); // InlineElementParser 是第3层
      });
    });
    
    test('应该保护现有占位符', () => {
      const existingPlaceholders = [
        {
          id: '__L2_ITEMIZE_1__',
          content: '\\begin{itemize}\\item test\\end{itemize}',
          type: 'ITEMIZE',
          layer: 2
        }
      ];
      
      const text = 'Text with __L2_ITEMIZE_1__ and \\textbf{bold}';
      const result = parser.process(text, existingPlaceholders);
      
      // 现有占位符应该被保护
      expect(result.text).toContain('__L2_ITEMIZE_1__');
    });
  });
  
  describe('复杂内容处理', () => {
    test('应该处理包含多种行内元素的复杂文本', () => {
      const text = `
        This is a \\textbf{complex} paragraph with \\textit{multiple} formatting options.
        We have \\emph{emphasis}, \\underline{underlined text}, and \\large{large text}.
        
        Line breaks:\\\\
        New line with \\small{small text} and special chars: \\& \\% \\$.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理行内格式与数学公式的混合', () => {
      const text = 'Text with \\textbf{bold} and math $x^2$ and \\textit{italic}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 数学公式应该被保护
      const hasMath = result.text.includes('$') ||
                     result.text.includes('x^2');
      expect(hasMath).toBe(true);
    });
  });
  
  describe('错误处理', () => {
    test('应该处理未闭合的格式命令', () => {
      const text = '\\textbf{unclosed bold text';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理空的格式命令', () => {
      const text = '\\textbf{} empty bold';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理无效的字号命令', () => {
      const text = '\\invalidsize{text}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 未知命令应该保持原样或产生警告
    });
    
    test('应该处理格式命令的嵌套错误', () => {
      const text = '\\textbf{bold \\textit{italic} \\emph{emphasis}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种错误处理机制
    });
  });
  
  describe('边界条件', () => {
    test('应该处理空字符串', () => {
      const result = parser.process('');
      
      expect(result.text).toBe('');
      expect(result.blocks).toHaveLength(0);
      expect(result.placeholders).toHaveLength(0);
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理不包含行内格式的纯文本', () => {
      const text = 'This is plain text without any formatting commands.';
      const result = parser.process(text);
      
      expect(result.text).toContain('plain text');
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理超长的格式化文本', () => {
      const longText = 'Very '.repeat(500) + 'long text';
      const text = `\\textbf{${longText}}`;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理深度嵌套的格式命令', () => {
      const nestedText = '\\textbf{L1 \\textit{L2 \\emph{L3 \\underline{L4 text} L3} L2} L1}';
      const result = parser.process(nestedText);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理Unicode字符的格式化', () => {
      const text = '\\textbf{中文粗体} \\textit{العربية مائل} \\emph{Русский акцент}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      expect(result.text.includes('中文') || result.placeholders.some(p => p.content.includes('中文'))).toBe(true);
    });
  });
});