/**
 * DocumentParser 测试
 * 
 * 基于 interface.md 规范的第1层解析器测试
 * 测试文档级结构命令处理功能
 */

const DocumentParser = require('../DocumentParser');

describe('DocumentParser - 文档结构解析层 (L1)', () => {
  let parser;
  
  beforeEach(() => {
    parser = new DocumentParser();
  });
  
  describe('核心方法接口', () => {
    test('应该具备 process 方法', () => {
      expect(typeof parser.process).toBe('function');
    });
    
    test('应该返回符合 LayerResult 格式的结果', () => {
      const result = parser.process('\\section{Test}');
      
      // 验证 LayerResult 格式
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('placeholders');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.text).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(Array.isArray(result.placeholders)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该支持传入已有占位符数组', () => {
      const existingPlaceholders = [
        {
          id: '__L2_BLOCK_1__',
          content: 'test content',
          type: 'BLOCK',
          layer: 2
        }
      ];
      
      const result = parser.process('\\section{Test}', existingPlaceholders);
      
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('placeholders');
    });
  });
  
  describe('章节命令处理', () => {
    test('应该处理 \\section 命令', () => {
      const text = '\\section{Introduction}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.blocks.length).toBeGreaterThan(0);
      
      // 检查是否创建了占位符
      const hasSection = result.placeholders.some(p => 
        p.id.includes('L1') && p.content.includes('section')
      );
      expect(hasSection || result.text.includes('section')).toBe(true);
    });
    
    test('应该处理 \\subsection 命令', () => {
      const text = '\\subsection{Details}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\subsubsection 命令', () => {
      const text = '\\subsubsection{Specifics}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\chapter 命令', () => {
      const text = '\\chapter{First Chapter}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理嵌套章节结构', () => {
      const text = `
        \\chapter{Main Chapter}
        \\section{Section One}
        \\subsection{Subsection A}
        \\subsubsection{Details}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.blocks.length).toBeGreaterThan(0);
    });
  });
  
  describe('文档元数据处理', () => {
    test('应该处理 \\title 命令', () => {
      const text = '\\title{My Document Title}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 标题可能被转换为占位符或直接处理
      const hasTitle = result.placeholders.some(p => 
        p.content.includes('title') || p.content.includes('My Document Title')
      ) || result.text.includes('title') || result.blocks.some(b => 
        JSON.stringify(b).includes('title')
      );
      expect(hasTitle).toBe(true);
    });
    
    test('应该处理 \\author 命令', () => {
      const text = '\\author{John Doe}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 \\date 命令', () => {
      const text = '\\date{2024-01-01}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理组合的文档元数据', () => {
      const text = `
        \\title{Test Document}
        \\author{Jane Smith}
        \\date{\\today}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('文档环境处理', () => {
    test('应该处理 document 环境', () => {
      const text = `
        \\begin{document}
        This is document content.
        \\end{document}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理完整的文档结构', () => {
      const text = `
        \\documentclass{article}
        \\title{Complete Document}
        \\author{Test Author}
        \\begin{document}
        \\maketitle
        \\section{Introduction}
        Content here.
        \\end{document}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.blocks.length).toBeGreaterThan(0);
    });
  });
  
  describe('数学环境保护', () => {
    test('应该保护 $$...$$ 数学块', () => {
      const text = `
        Some text before.
        $$E = mc^2$$
        Some text after.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 数学块应该被保护，可能通过占位符或保持原样
      const hasMath = result.text.includes('$$') || 
                     result.placeholders.some(p => p.content.includes('$$')) ||
                     result.text.includes('E = mc');
      expect(hasMath).toBe(true);
    });
    
    test('应该处理多个数学块', () => {
      const text = `
        First equation: $$x + y = z$$
        Second equation: $$a^2 + b^2 = c^2$$
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('占位符机制', () => {
    test('应该创建符合格式的占位符', () => {
      const text = '\\section{Test Section}';
      const result = parser.process(text);
      
      // 检查占位符格式：__L{layer}_{TYPE}_{index}__
      result.placeholders.forEach(placeholder => {
        expect(placeholder).toHaveProperty('id');
        expect(placeholder).toHaveProperty('content');
        expect(placeholder).toHaveProperty('type');
        expect(placeholder).toHaveProperty('layer');
        
        // 验证 ID 格式
        expect(placeholder.id).toMatch(/^__L\d+_\w+_\d+__$/);
        expect(placeholder.layer).toBe(1); // DocumentParser 是第1层
      });
    });
    
    test('应该保护现有占位符', () => {
      const existingPlaceholders = [
        {
          id: '__L2_ITEMIZE_1__',
          content: '\\begin{itemize}\\item test\\end{itemize}',
          type: 'ITEMIZE',
          layer: 2
        }
      ];
      
      const text = '\\section{Test} __L2_ITEMIZE_1__ more text';
      const result = parser.process(text, existingPlaceholders);
      
      // 现有占位符应该被保护
      expect(result.text).toContain('__L2_ITEMIZE_1__');
    });
  });
  
  describe('错误处理', () => {
    test('应该处理格式错误的章节命令', () => {
      const text = '\\section{Unclosed section';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理空的章节标题', () => {
      const text = '\\section{}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理无效的文档环境', () => {
      const text = '\\begin{document} \\begin{invalid} content';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种形式的错误或警告处理
    });
  });
  
  describe('边界条件', () => {
    test('应该处理空字符串', () => {
      const result = parser.process('');
      
      expect(result.text).toBe('');
      expect(result.blocks).toHaveLength(0);
      expect(result.placeholders).toHaveLength(0);
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理只有空格的字符串', () => {
      const result = parser.process('   \\n\\t  ');
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理非常长的章节标题', () => {
      const longTitle = 'Very '.repeat(100) + 'Long Title';
      const text = `\\section{${longTitle}}`;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
});