/**
 * BlockElementParser 测试
 * 
 * 基于 interface.md 规范的第2层解析器测试
 * 测试块级结构元素处理功能
 */

const BlockElementParser = require('../BlockElementParser');

describe('BlockElementParser - 块级结构解析层 (L2)', () => {
  let parser;
  
  beforeEach(() => {
    parser = new BlockElementParser();
  });
  
  describe('核心方法接口', () => {
    test('应该具备 process 方法', () => {
      expect(typeof parser.process).toBe('function');
    });
    
    test('应该返回符合 LayerResult 格式的结果', () => {
      const result = parser.process('\\begin{itemize}\\item test\\end{itemize}');
      
      // 验证 LayerResult 格式
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('placeholders');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      // 验证数据类型
      expect(typeof result.text).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(Array.isArray(result.placeholders)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });
    
    test('应该支持传入已有占位符数组', () => {
      const existingPlaceholders = [
        {
          id: '__L1_SECTION_1__',
          content: '\\section{Test}',
          type: 'SECTION',
          layer: 1
        }
      ];
      
      const result = parser.process('\\begin{itemize}\\item test\\end{itemize}', existingPlaceholders);
      
      expect(result).toHaveProperty('text');
      expect(result).toHaveProperty('placeholders');
    });
  });
  
  describe('列表环境处理', () => {
    test('应该处理 itemize 环境', () => {
      const text = `
        \\begin{itemize}
        \\item First item
        \\item Second item
        \\item Third item
        \\end{itemize}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查是否创建了相应的块或占位符
      const hasItemize = result.blocks.length > 0 || 
                        result.placeholders.some(p => p.type.includes('ITEMIZE') || p.content.includes('itemize'));
      expect(hasItemize).toBe(true);
    });
    
    test('应该处理 enumerate 环境', () => {
      const text = `
        \\begin{enumerate}
        \\item First numbered item
        \\item Second numbered item
        \\end{enumerate}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 description 环境', () => {
      const text = `
        \\begin{description}
        \\item[Term 1] Description of term 1
        \\item[Term 2] Description of term 2
        \\end{description}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理嵌套列表', () => {
      const text = `
        \\begin{itemize}
        \\item Top level item
        \\begin{enumerate}
        \\item Nested numbered item
        \\item Another nested item
        \\end{enumerate}
        \\item Another top level item
        \\end{itemize}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的列表项内容', () => {
      const text = `
        \\begin{itemize}
        \\item Item with \\textbf{bold text} and \\textit{italic text}
        \\item Item with math: $x + y = z$
        \\item Item with line breaks\\\\and continuation
        \\end{itemize}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('表格环境处理', () => {
    test('应该处理 tabular 环境', () => {
      const text = `
        \\begin{tabular}{|l|c|r|}
        \\hline
        Left & Center & Right \\\\
        \\hline
        Data1 & Data2 & Data3 \\\\
        \\hline
        \\end{tabular}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查是否创建了表格相关的块或占位符
      const hasTabular = result.blocks.length > 0 || 
                         result.placeholders.some(p => p.type.includes('TABULAR') || p.content.includes('tabular'));
      expect(hasTabular).toBe(true);
    });
    
    test('应该处理 array 环境', () => {
      const text = `
        \\begin{array}{cc}
        a & b \\\\
        c & d
        \\end{array}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理复杂的表格格式', () => {
      const text = `
        \\begin{tabular}{|p{3cm}|c|r|}
        \\hline
        \\textbf{Column 1} & \\textbf{Column 2} & \\textbf{Column 3} \\\\
        \\hline
        Long text content & Center & 123.45 \\\\
        More text & $x^2$ & 67.89 \\\\
        \\hline
        \\end{tabular}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('图片环境处理', () => {
    test('应该处理 figure 环境', () => {
      const text = `
        \\begin{figure}[h]
        \\centering
        \\includegraphics[width=0.5\\textwidth]{image.png}
        \\caption{Test image caption}
        \\label{fig:test}
        \\end{figure}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      
      // 检查是否创建了图片相关的块或占位符
      const hasFigure = result.blocks.length > 0 || 
                       result.placeholders.some(p => p.type.includes('FIGURE') || p.content.includes('figure'));
      expect(hasFigure).toBe(true);
    });
    
    test('应该处理简单的图片环境', () => {
      const text = `
        \\begin{figure}
        \\includegraphics{simple.jpg}
        \\end{figure}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('引用环境处理', () => {
    test('应该处理 quote 环境', () => {
      const text = `
        \\begin{quote}
        This is a quoted text that should be formatted differently
        from the main text body.
        \\end{quote}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理 quotation 环境', () => {
      const text = `
        \\begin{quotation}
        This is a longer quotation that typically includes
        paragraph indentation and different formatting.
        \\end{quotation}
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
  
  describe('占位符机制', () => {
    test('应该创建符合格式的占位符', () => {
      const text = '\\begin{itemize}\\item test\\end{itemize}';
      const result = parser.process(text);
      
      // 检查占位符格式：__L{layer}_{TYPE}_{index}__
      result.placeholders.forEach(placeholder => {
        expect(placeholder).toHaveProperty('id');
        expect(placeholder).toHaveProperty('content');
        expect(placeholder).toHaveProperty('type');
        expect(placeholder).toHaveProperty('layer');
        
        // 验证 ID 格式
        expect(placeholder.id).toMatch(/^__L\d+_\w+_\d+__$/);
        expect(placeholder.layer).toBe(2); // BlockElementParser 是第2层
      });
    });
    
    test('应该保护现有占位符', () => {
      const existingPlaceholders = [
        {
          id: '__L1_SECTION_1__',
          content: '\\section{Test}',
          type: 'SECTION',
          layer: 1
        }
      ];
      
      const text = '__L1_SECTION_1__ \\begin{itemize}\\item test\\end{itemize}';
      const result = parser.process(text, existingPlaceholders);
      
      // 现有占位符应该被保护
      expect(result.text).toContain('__L1_SECTION_1__');
    });
  });
  
  describe('混合内容处理', () => {
    test('应该处理包含多种块级元素的文本', () => {
      const text = `
        Some introductory text.
        
        \\begin{itemize}
        \\item First point
        \\item Second point
        \\end{itemize}
        
        \\begin{tabular}{|c|c|}
        \\hline
        A & B \\\\
        \\hline
        C & D \\\\
        \\hline
        \\end{tabular}
        
        \\begin{figure}
        \\includegraphics{test.png}
        \\caption{Test figure}
        \\end{figure}
        
        Concluding text.
      `;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
      expect(result.blocks.length + result.placeholders.length).toBeGreaterThan(0);
    });
  });
  
  describe('错误处理', () => {
    test('应该处理未闭合的环境', () => {
      const text = '\\begin{itemize}\\item unclosed item';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 可能会有警告或错误，但应该继续处理
      expect(result.warnings.length + result.errors.length).toBeGreaterThanOrEqual(0);
    });
    
    test('应该处理不匹配的环境', () => {
      const text = '\\begin{itemize}\\item test\\end{enumerate}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种形式的错误或警告处理
    });
    
    test('应该处理空的环境', () => {
      const text = '\\begin{itemize}\\end{itemize}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理无效的表格格式', () => {
      const text = '\\begin{tabular}{invalid}content\\end{tabular}';
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      // 应该有某种处理或警告
    });
  });
  
  describe('边界条件', () => {
    test('应该处理空字符串', () => {
      const result = parser.process('');
      
      expect(result.text).toBe('');
      expect(result.blocks).toHaveLength(0);
      expect(result.placeholders).toHaveLength(0);
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理不包含块级元素的文本', () => {
      const text = 'This is just plain text without any block elements.';
      const result = parser.process(text);
      
      expect(result.text).toContain('plain text');
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理超长的列表项', () => {
      const longItem = 'Very '.repeat(200) + 'long item';
      const text = `\\begin{itemize}\\item ${longItem}\\end{itemize}`;
      
      const result = parser.process(text);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
    
    test('应该处理深度嵌套的列表', () => {
      const nestedText = `
        \\begin{itemize}
        \\item Level 1
        \\begin{itemize}
        \\item Level 2
        \\begin{itemize}
        \\item Level 3
        \\begin{itemize}
        \\item Level 4
        \\end{itemize}
        \\end{itemize}
        \\end{itemize}
        \\end{itemize}
      `;
      
      const result = parser.process(nestedText);
      
      expect(result.text).toBeTruthy();
      expect(result.errors).toHaveLength(0);
    });
  });
});