# Core 模块接口文档

## 模块概述

Core 模块是LaTeX Parser V3的核心解析引擎，实现五层流水线架构。该模块负责协调五个专门化的解析器，按序处理不同层级的LaTeX内容，通过占位符机制实现内容保护和层级隔离。

## 公共接口

### LaTeXParser (主控制器)

**位置**: `core/LaTeXParser.js`

**构造函数**:
```javascript
new LaTeXParser(config = {})
```

**配置参数**:
- `mode` {String} - 解析模式，'tolerant'(默认) 或 'strict'
- `enabledLayers` {Array} - 启用的解析层，默认 ['document', 'block', 'inline', 'math', 'special']
- `maxRecursionDepth` {Number} - 最大递归深度，默认 100

**核心方法**:
```javascript
parseComplete(content: String): ParseResult
```

**返回格式 (ParseResult)**:
```javascript
{
  success: Boolean,        // 解析是否成功
  processedText: String,   // 处理后的HTML文本
  blocks: Array,          // 结构化数据块
  metadata: Object,       // 处理元信息
  warnings: Array,        // 警告信息数组
  errors: Array          // 错误信息数组
}
```

### DocumentParser (第1层)

**位置**: `core/DocumentParser.js`

**职责**: 处理文档级结构命令（章节、标题、文档环境、数学环境预处理）

**核心方法**:
```javascript
process(text: String, existingPlaceholders: Array = []): LayerResult
```

**处理内容**:
- 章节命令：`\section`, `\subsection`, `\subsubsection`, `\chapter`
- 文档元数据：`\title`, `\author`, `\date`
- 文档环境：`\begin{document}...\end{document}`
- 数学环境保护：`$$...$$` 块保护

### BlockElementParser (第2层)

**位置**: `core/BlockElementParser.js`

**职责**: 处理块级结构元素（列表、表格、浮动环境）

**核心方法**:
```javascript
process(text: String, existingPlaceholders: Array = []): LayerResult
```

**处理内容**:
- 列表环境：`itemize`, `enumerate`, `description`
- 表格环境：`tabular`, `array`
- 图片环境：`figure`
- 引用环境：`quote`, `quotation`

### InlineElementParser (第3层)

**位置**: `core/InlineElementParser.js`

**职责**: 处理行内格式化和文本修饰

**核心方法**:
```javascript
process(text: String, existingPlaceholders: Array = []): LayerResult
```

**处理内容**:
- 文本格式：`\textbf`, `\textit`, `\emph`, `\underline`
- 字号命令：`\large`, `\small`, `\tiny`
- 特殊字符转义
- 换行命令：`\\`, `\newline`

### MathParser (第4层)

**位置**: `core/MathParser.js`

**职责**: 处理数学公式和数学环境

**核心方法**:
```javascript
process(text: String, existingPlaceholders: Array = []): LayerResult
```

**处理内容**:
- 行内数学：`$...$`, `\(...\)`
- 显示数学：`$$...$$`, `\[...\]`
- 数学环境：`equation`, `align`, `gather`, `matrix`
- 数学统计和验证

### SpecialParser (第5层)

**位置**: `core/SpecialParser.js`

**职责**: 处理特殊功能和扩展命令

**核心方法**:
```javascript
process(text: String, existingPlaceholders: Array = []): LayerResult
```

**处理内容**:
- 包引用：`\usepackage`
- 图片插入：`\includegraphics`
- 脚注系统：`\footnote`
- 交叉引用：`\cite`, `\ref`, `\label`
- 自定义命令

## 统一的层级接口

### LayerResult (层级输出格式)

所有解析层都遵循统一的输入输出接口：

```javascript
{
  text: String,              // 处理后的文本
  blocks: Array,            // 该层产生的结构化数据块
  placeholders: Array,      // 该层创建的占位符
  warnings: Array,          // 处理警告
  errors: Array            // 处理错误
}
```

### 占位符格式

```javascript
{
  id: String,               // 格式: "__L{layer}_{TYPE}_{index}__"
  content: String,          // 原始内容
  type: String,            // 占位符类型
  layer: Number            // 所属层级 (1-5)
}
```

## 数据流架构

### 处理流程

1. **输入** → DocumentParser (L1)
2. **L1输出** → BlockElementParser (L2)  
3. **L2输出** → InlineElementParser (L3)
4. **L3输出** → MathParser (L4)
5. **L4输出** → SpecialParser (L5)
6. **L5输出** → 占位符恢复 → **最终输出**

### 占位符保护机制

- 每层创建自己的占位符：`__L{layer}_{TYPE}_{index}__`
- 后续层自动保护前面层的占位符
- 最后统一恢复：L5→L4→L3→L2→L1

## 依赖关系

### 模块间依赖
- `LaTeXParser` 依赖所有五个解析器
- 各解析器独立，无横向依赖
- 所有解析器依赖 `utils/MathContentDetector`

### 外部依赖
- `config/latex-html-mapping.js` - HTML映射配置
- `utils/DollarDollarProtector.js` - 数学公式保护
- `utils/MathContentDetector.js` - 数学内容识别

## 错误处理约定

### 容错模式 (tolerant)
- 遇到错误生成警告，继续处理
- 未知命令保持原样输出
- 提供降级处理方案

### 严格模式 (strict)  
- 遇到错误立即停止处理
- 返回详细错误信息
- 适用于开发和调试场景

## 性能优化特性

- **层级控制**: 通过 `enabledLayers` 只启用需要的层
- **递归限制**: `maxRecursionDepth` 防止无限递归
- **占位符缓存**: 避免重复处理相同内容
- **并发安全**: 各层解析器无状态设计