/**
 * SpecialParser - 特殊功能解析器
 * 负责处理LaTeX特殊功能命令
 * 包括：包管理、图像处理、目录生成、脚注系统、自定义命令等
 * 
 * 当前为Green阶段占位符实现，后续将扩展为完整功能
 * TDD改进：添加$$保护机制
 */

// 新架构：不再需要DollarDollarProtector
// 占位符由LaTeXParser统一管理

class SpecialParser {
  constructor(config = {}) {
    this.config = {
      // 包管理配置
      supportedPackages: {
        'graphicx': { features: ['includegraphics'] },
        'amsmath': { features: ['math_environments'] },
        'hyperref': { features: ['hyperlinks'] },
        'geometry': { features: ['page_layout'] },
        'babel': { features: ['internationalization'] }
      },
      // 图像处理配置
      imageExtensions: ['png', 'jpg', 'jpeg', 'pdf', 'svg', 'eps'],
      // 脚注配置
      footnoteCounter: 0,
      ...config
    };
    // 新架构：不需要初始化DollarDollarProtector
  }

  /**
   * 处理特殊功能解析
   * @param {string} text - 输入文本
   * @param {Array} existingPlaceholders - 从其他层传递的占位符
   * @returns {object} 处理结果
   */
  process(text, existingPlaceholders = []) {
    // 不再使用DollarDollarProtector，因为$$已经在其他层被保护了
    let processedText = text;
    const blocks = [];
    const warnings = [];
    const errors = [];
    const placeholders = [...existingPlaceholders]; // 保留已有占位符

    try {
      // Green阶段：基础特殊功能处理
      // 后续将实现：
      // - 包管理系统
      // - 图像处理功能
      // - 目录生成
      // - 脚注系统
      // - 自定义命令
      // - 计数器系统

      // 当前实现的基础功能
      processedText = this.processPackages(processedText, blocks, warnings);
      processedText = this.processGraphics(processedText, blocks, warnings);
      processedText = this.processFootnotes(processedText, blocks);
      processedText = this.processTableOfContents(processedText, blocks);

    } catch (error) {
      errors.push({
        type: 'special_parsing_error',
        message: `特殊功能解析错误: ${error.message}`,
        position: { line: 1, column: 1 }
      });
    }

    // 不恢复占位符，保持它们直到最后

    return {
      text: processedText,
      blocks,
      warnings,
      errors,
      placeholders // 返回所有占位符
    };
  }

  /**
   * 处理包管理命令
   * @private
   */
  processPackages(text, blocks, warnings) {
    let processedText = text;

    // 处理\\usepackage命令
    processedText = processedText.replace(/\\usepackage(?:\[[^\]]*\])?\{([^}]+)\}/g, (match, packageName) => {
      const packages = packageName.split(',').map(p => p.trim());
      
      packages.forEach(pkg => {
        if (this.config.supportedPackages[pkg]) {
          blocks.push({
            type: 'package',
            packageName: pkg,
            features: this.config.supportedPackages[pkg].features,
            originalCommand: match,
            supported: true
          });
        } else {
          blocks.push({
            type: 'package',
            packageName: pkg,
            originalCommand: match,
            supported: false
          });

          warnings.push({
            type: 'unsupported_package',
            package: pkg,
            message: `包 ${pkg} 当前不支持，某些功能可能无法正常工作`
          });
        }
      });

      // 移除usepackage命令（已处理）
      return '';
    });

    return processedText;
  }

  /**
   * 处理图像命令
   * @private
   */
  processGraphics(text, blocks, warnings) {
    let processedText = text;

    // 处理\\includegraphics命令（包括选项）
    processedText = processedText.replace(/\\includegraphics(?:\[([^\]]*)\])?\{([^}]+)\}/g, (match, options, imagePath) => {
      const imageInfo = this.parseImagePath(imagePath);
      const imageOptions = this.parseImageOptions(options);
      
      blocks.push({
        type: 'includegraphics',
        imagePath: imageInfo.path,
        extension: imageInfo.extension,
        options: imageOptions,
        originalCommand: match,
        position: this.findPosition(text, match)
      });

      // 检查图像扩展名
      if (!this.config.imageExtensions.includes(imageInfo.extension)) {
        warnings.push({
          type: 'unsupported_image_format',
          imagePath: imagePath,
          extension: imageInfo.extension,
          message: `图像格式 .${imageInfo.extension} 可能不被支持`
        });
      }

      // 生成HTML图像标签，包含选项样式
      const altText = imageInfo.name;
      const styleAttr = this.generateImageStyle(imageOptions);
      return `<img src="${imagePath}" alt="${altText}" class="latex-image" ${styleAttr}/>`;
    });

    return processedText;
  }

  /**
   * 解析图像选项
   * @private
   */
  parseImageOptions(optionsStr) {
    const options = {};
    
    if (!optionsStr) return options;

    // 解析常见的图像选项
    const optionPairs = optionsStr.split(',').map(s => s.trim());
    
    optionPairs.forEach(pair => {
      // 处理 key=value 格式
      if (pair.includes('=')) {
        const [key, value] = pair.split('=').map(s => s.trim());
        
        // 处理常见选项
        switch(key) {
          case 'width':
            options.width = this.convertLatexDimension(value);
            break;
          case 'height':
            options.height = this.convertLatexDimension(value);
            break;
          case 'scale':
            options.scale = parseFloat(value);
            break;
          case 'angle':
            options.angle = parseFloat(value);
            break;
          default:
            options[key] = value;
        }
      } else {
        // 处理标志选项（如 keepaspectratio）
        options[pair] = true;
      }
    });

    return options;
  }

  /**
   * 转换LaTeX尺寸到CSS
   * @private
   */
  convertLatexDimension(dimension) {
    if (!dimension) return null;
    
    // 处理 \textwidth, \linewidth 等特殊值
    if (dimension === '\\textwidth' || dimension === '\\linewidth') {
      return '100%';
    }
    if (dimension === '\\columnwidth') {
      return '50%';
    }
    
    // 处理百分比形式（如 0.5\textwidth）
    const percentMatch = dimension.match(/^([\d.]+)\\(textwidth|linewidth|columnwidth)$/);
    if (percentMatch) {
      const percent = parseFloat(percentMatch[1]) * 100;
      return `${percent}%`;
    }
    
    // 处理带单位的值（pt, cm, mm, in, em, ex, px）
    const unitMatch = dimension.match(/^([\d.]+)(pt|cm|mm|in|em|ex|px)$/);
    if (unitMatch) {
      const value = parseFloat(unitMatch[1]);
      const unit = unitMatch[2];
      
      // 转换常见单位到CSS
      switch(unit) {
        case 'pt':
          return `${value * 1.333}px`; // 1pt = 1.333px
        case 'cm':
          return `${value * 37.795}px`; // 1cm = 37.795px
        case 'mm':
          return `${value * 3.7795}px`; // 1mm = 3.7795px
        case 'in':
          return `${value * 96}px`; // 1in = 96px
        default:
          return `${value}${unit}`;
      }
    }
    
    return dimension;
  }

  /**
   * 生成图像样式属性
   * @private
   */
  generateImageStyle(options) {
    const styles = [];
    
    if (options.width) {
      styles.push(`width: ${options.width}`);
    }
    if (options.height) {
      styles.push(`height: ${options.height}`);
    }
    if (options.scale) {
      styles.push(`transform: scale(${options.scale})`);
    }
    if (options.angle) {
      styles.push(`transform: rotate(${options.angle}deg)`);
    }
    if (options.keepaspectratio && !options.height) {
      styles.push('height: auto');
    }
    
    return styles.length > 0 ? `style="${styles.join('; ')}"` : '';
  }

  /**
   * 处理脚注系统
   * @private
   */
  processFootnotes(text, blocks) {
    let processedText = text;
    const footnotes = [];

    // 处理\\footnote{}命令
    processedText = processedText.replace(/\\footnote\{([^}]+)\}/g, (match, footnoteContent) => {
      this.config.footnoteCounter++;
      const footnoteId = `fn${this.config.footnoteCounter}`;
      const footnoteRefId = `fnref${this.config.footnoteCounter}`;
      
      // 收集脚注信息
      footnotes.push({
        id: footnoteId,
        refId: footnoteRefId,
        content: footnoteContent,
        number: this.config.footnoteCounter
      });
      
      blocks.push({
        type: 'footnote',
        id: footnoteId,
        refId: footnoteRefId,
        content: footnoteContent,
        number: this.config.footnoteCounter,
        originalCommand: match
      });

      // 生成脚注引用
      return `<sup id="${footnoteRefId}"><a href="#${footnoteId}" class="footnote-ref">[${this.config.footnoteCounter}]</a></sup>`;
    });

    // 如果有脚注，在文档末尾添加脚注内容区域
    if (footnotes.length > 0) {
      const footnotesHtml = this.renderFootnotes(footnotes);
      processedText += footnotesHtml;
    }

    return processedText;
  }

  /**
   * 渲染脚注内容区域
   * @private
   */
  renderFootnotes(footnotes) {
    let html = '\n<hr class="footnotes-separator">\n';
    html += '<section class="footnotes">\n';
    html += '<h2 class="footnotes-title">脚注</h2>\n';
    html += '<ol class="footnotes-list">\n';

    footnotes.forEach(footnote => {
      html += `  <li id="${footnote.id}" class="footnote-item">\n`;
      html += `    <p>${footnote.content} `;
      html += `<a href="#${footnote.refId}" class="footnote-backref">↩</a></p>\n`;
      html += `  </li>\n`;
    });

    html += '</ol>\n';
    html += '</section>\n';
    
    return html;
  }

  /**
   * 处理目录生成
   * @private
   */
  processTableOfContents(text, blocks) {
    let processedText = text;

    // 处理\\tableofcontents命令
    processedText = processedText.replace(/\\tableofcontents/g, (match) => {
      // 收集文档中的章节信息
      const sections = this.collectSections(text);
      
      blocks.push({
        type: 'table_of_contents',
        originalCommand: match,
        sections: sections,
        generated: true
      });

      // 生成目录HTML
      const tocHtml = this.renderTableOfContents(sections);
      return tocHtml;
    });

    return processedText;
  }

  /**
   * 收集文档中的章节信息
   * @private
   */
  collectSections(text) {
    const sections = [];
    const sectionCommands = {
      'chapter': 1,
      'section': 2,
      'subsection': 3,
      'subsubsection': 4
    };

    for (const [command, level] of Object.entries(sectionCommands)) {
      const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
      let match;
      while ((match = regex.exec(text)) !== null) {
        sections.push({
          level: level,
          title: match[1].trim(),
          command: command,
          position: text.indexOf(match[0])
        });
      }
    }

    // 按文档位置排序
    sections.sort((a, b) => a.position - b.position);
    
    // 生成锚点ID
    sections.forEach((section, index) => {
      section.id = `toc-${section.command}-${index}`;
    });

    return sections;
  }

  /**
   * 渲染目录HTML
   * @private
   */
  renderTableOfContents(sections) {
    if (sections.length === 0) {
      return '<nav class="table-of-contents"><p>No sections found</p></nav>';
    }

    let html = '<nav class="table-of-contents"><h2>目录</h2><ul class="toc-list">';
    let currentLevel = 0;

    sections.forEach(section => {
      // 调整嵌套级别
      while (currentLevel < section.level) {
        html += '<ul class="toc-sublist">';
        currentLevel++;
      }
      while (currentLevel > section.level) {
        html += '</ul>';
        currentLevel--;
      }

      // 添加目录项
      html += `<li class="toc-item toc-level-${section.level}">`;
      html += `<a href="#${section.id}">${section.title}</a>`;
      html += '</li>';
    });

    // 关闭所有未关闭的列表
    while (currentLevel > 0) {
      html += '</ul>';
      currentLevel--;
    }

    html += '</nav>';
    return html;
  }

  /**
   * 解析图像路径信息
   * @private
   */
  parseImagePath(imagePath) {
    const lastDot = imagePath.lastIndexOf('.');
    const lastSlash = Math.max(imagePath.lastIndexOf('/'), imagePath.lastIndexOf('\\'));
    
    let extension = '';
    let name = imagePath;
    let path = imagePath;

    if (lastDot > lastSlash) {
      extension = imagePath.substring(lastDot + 1).toLowerCase();
      name = imagePath.substring(lastSlash + 1, lastDot);
    } else {
      name = imagePath.substring(lastSlash + 1);
    }

    return {
      path,
      extension,
      name,
      directory: lastSlash > -1 ? imagePath.substring(0, lastSlash) : ''
    };
  }

  /**
   * 处理自定义命令（未来实现）
   * @private
   */
  processCustomCommands(text, blocks, warnings) {
    // TODO: 实现\\newcommand处理
    return text;
  }

  /**
   * 处理计数器系统（未来实现）
   * @private
   */
  processCounters(text, blocks) {
    // TODO: 实现计数器系统
    return text;
  }

  /**
   * 查找命令在文本中的位置
   * @private
   */
  findPosition(text, match) {
    const index = text.indexOf(match);
    if (index === -1) return { line: 1, column: 1 };

    const beforeMatch = text.substring(0, index);
    const lines = beforeMatch.split('\n');
    
    return {
      line: lines.length,
      column: lines[lines.length - 1].length + 1,
      start: index,
      end: index + match.length
    };
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = SpecialParser;
  console.log('[SpecialParser] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境（包括测试环境中的模拟浏览器）
  window.SpecialParser = SpecialParser;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.SpecialParser = SpecialParser;
  }
}
