/**
 * MathParser - 数学内容解析器
 * 负责处理LaTeX数学公式和环境
 * 包括：行内数学、显示数学、数学环境等
 * 
 * TDD改进：集成MathContentDetector实现统一数学内容识别
 * TDD改进2：添加$$保护机制避免重复处理
 */

// 浏览器和Node.js兼容的模块导入 - 内嵌简化版
const MathContentDetector_Math = (function() {
  if (typeof require !== 'undefined') {
    try { return require('../utils/MathContentDetector'); } catch (e) { /* fallback */ }
  } else if (typeof window !== 'undefined' && window.MathContentDetector) {
    return window.MathContentDetector;
  }
  
  return class MathContentDetector {
    constructor() {
      this.mathEnvironments = ['equation', 'equation*', 'align', 'align*', 'gather', 'gather*'];
    }
    detectMathEnvironments(text) {
      const blocks = [];
      this.mathEnvironments.forEach(env => {
        const regex = new RegExp(`\\\\begin\\{${env}\\}([\\s\\S]*?)\\\\end\\{${env}\\}`, 'g');
        let match;
        while ((match = regex.exec(text)) !== null) {
          blocks.push({
            type: 'math_environment', environment: env, content: match[1].trim(),
            fullMatch: match[0], start: match.index, end: match.index + match[0].length
          });
        }
      });
      return blocks;
    }
  };
})();

// 新架构：不再需要DollarDollarProtector
// 占位符由LaTeXParser统一管理

class MathParser {
  constructor(config = {}) {
    this.config = {
      // 数学环境映射
      mathEnvironments: {
        'equation': { display: true, numbered: true },
        'equation*': { display: true, numbered: false },
        'align': { display: true, numbered: true, multiline: true },
        'align*': { display: true, numbered: false, multiline: true },
        'alignat': { display: true, numbered: true, multiline: true, hasParameters: true },
        'alignat*': { display: true, numbered: false, multiline: true, hasParameters: true },
        'gather': { display: true, numbered: true, multiline: true },
        'gather*': { display: true, numbered: false, multiline: true }
      },
      // KaTeX配置
      katex: {
        displayMode: false,
        throwOnError: false,
        errorColor: '#cc0000',
        macros: {}
      },
      ...config
    };
    
    // TDD: 初始化数学内容检测器
    this.mathDetector = new MathContentDetector_Math();
    // 新架构：不需要初始化DollarDollarProtector
  }

  /**
   * 处理数学内容解析
   * @param {string} text - 输入文本
   * @param {Array} existingPlaceholders - 从其他层传递的占位符
   * @returns {object} 处理结果
   */
  process(text, existingPlaceholders = []) {
    // 不再使用DollarDollarProtector，因为$$已经在其他层被保护了
    let processedText = text;
    const blocks = [];
    const warnings = [];
    const errors = [];
    const placeholders = [...existingPlaceholders]; // 保留已有占位符

    try {
      // TDD Green阶段：所有数学环境处理
      
      // 处理所有支持的数学环境（保持原生LaTeX格式）
      processedText = this.processAllMathEnvironments(processedText, blocks);
      
      // 处理其他数学内容（行内、显示数学等）  
      this.processOtherMathContent(processedText, blocks, warnings);
      
      // 为剩余的数学环境添加KaTeX可识别的分隔符包装（DocumentParser已处理主要环境）
      processedText = this.wrapRemainingMathEnvironments(processedText);

    } catch (error) {
      console.error('[MathParser] 数学解析出错:', error);
      errors.push({
        type: 'math_parsing_error',
        message: `数学解析错误: ${error.message}`,
        position: { line: 1, column: 1 }
      });
    }

    // 不恢复占位符，保持它们直到最后

    return {
      text: processedText,
      blocks,
      warnings,
      errors,
      placeholders // 返回所有占位符
    };
  }


  /**
   * 处理所有数学环境
   * 保持原生LaTeX格式，确保KaTeX能正确识别和渲染
   * @param {string} text - 输入文本
   * @param {Array} blocks - 收集的数学块
   * @returns {string} 处理后的文本
   * @private
   */
  processAllMathEnvironments(text, blocks) {
    let processedText = text;
    
    
    // TDD: 首先识别DocumentParser已经包装的$$环境，不再重复处理
    const alreadyWrappedPattern = /\$\$\\begin\{[^}]+\}[\s\S]*?\\end\{[^}]+\}\$\$/g;
    const wrappedMatches = [...processedText.matchAll(alreadyWrappedPattern)];
    
    if (wrappedMatches.length > 0) {
      // 如果有已包装的环境，只处理统计，不修改内容
      wrappedMatches.forEach((match, index) => {
        const envMatch = match[0].match(/\\begin\{([^}]+)\}/);
        const envName = envMatch ? envMatch[1] : 'unknown';
        console.log(`[MathParser] TDD: 记录已包装的${envName}环境`);
        
        blocks.push({
          type: 'math_environment',
          environment: envName,
          content: match[0], // 完整的$$包装内容
          originalText: match[0],
          displayMode: true,
          alreadyWrapped: true,
          source: 'DocumentParser'
        });
      });
      
      return processedText; // 直接返回，保持$$格式
    }
    
    // 处理每种支持的数学环境（仅当没有已包装环境时）
    for (const [envName, envConfig] of Object.entries(this.config.mathEnvironments)) {
      // TDD: 修正正则表达式，使用正确的转义
      let regex;
      if (envConfig.hasParameters) {
        regex = new RegExp(`\\\\begin\\{${envName}\\}(?:\\{[^}]*\\})?([\\s\\S]*?)\\\\end\\{${envName}\\}`, 'g');
      } else {
        regex = new RegExp(`\\\\begin\\{${envName}\\}([\\s\\S]*?)\\\\end\\{${envName}\\}`, 'g');
      }
      
      const matches = [...processedText.matchAll(regex)];
      
      // console.log(`[MathParser] 找到 ${matches.length} 个 ${envName} 环境`);
      
      matches.forEach((match, index) => {
        const fullMatch = match[0];
        const content = match[1];
        
        // 为每个数学环境创建块信息
        const mathBlock = {
          type: 'math_environment',
          environment: envName,
          content: fullMatch,
          innerContent: content,
          originalCommand: fullMatch,
          displayMode: envConfig.display,
          numbered: envConfig.numbered,
          multiline: envConfig.multiline || false,
          processed: true,  // 标记为已处理
          position: this.findPosition(processedText, fullMatch),
          // V3 特有信息
          layer: 'math',
          metadata: {
            envType: envName,
            katexCompatible: true,
            preserveOriginal: true
          }
        };
        
        blocks.push(mathBlock);
        
      });
    }
    
    // 重要：保持原生LaTeX格式不变，让KaTeX auto-render处理
    return processedText;
  }

  /**
   * 处理align数学环境（保持向后兼容）
   * @param {string} text - 输入文本
   * @returns {string} 处理后的文本
   * @private
   */
  processAlignEnvironment(text) {
    // 这个方法保持向后兼容，但实际工作由processAllMathEnvironments完成
    return text;
  }
  
  /**
   * 转换align内容为KaTeX兼容格式
   * @private
   */
  convertAlignToKaTeX(alignContent) {
    // 移除多余的空白并规范换行符
    let content = alignContent.trim();
    
    // 将 \\ 转换为 \\ (align环境的换行符KaTeX也支持)
    content = content.replace(/\\\\\s*/g, '\\\\ ');
    
    return content;
  }

  /**
   * 处理其他数学内容（行内数学、显示数学等）
   * 注意：这个方法主要用于统计，实际处理由其他层完成
   * @private
   */
  processOtherMathContent(text, blocks, warnings) {
    // 检测行内数学（不包括已经被$$包装的）
    const inlineMathMatches = text.match(/(?<!\$)\$(?!\$)[^$]+\$(?!\$)/g) || [];
    inlineMathMatches.forEach(match => {
      blocks.push({
        type: 'inline_math',
        content: match,
        originalCommand: match,
        processed: true  // 行内数学保持原样，KaTeX可以直接识别
      });
    });

    // 检测显示数学
    const displayMathMatches = text.match(/\\\[([\s\S]*?)\\\]/g) || [];
    displayMathMatches.forEach(match => {
      blocks.push({
        type: 'display_math',
        content: match,
        originalCommand: match,
        processed: true  // 显示数学保持原样，KaTeX可以直接识别
      });
    });

    // 检测已经被$$包装的数学环境（说明已被DocumentParser处理）
    const wrappedMathEnvironments = text.match(/\$\$\\begin\{[^}]+\}[\s\S]*?\\end\{[^}]+\}\$\$/g) || [];
    wrappedMathEnvironments.forEach(match => {
      const envMatch = match.match(/\\begin\{([^}]+)\}/);
      const envName = envMatch ? envMatch[1] : 'unknown';
      blocks.push({
        type: 'math_environment',
        environment: envName,
        content: match,
        originalCommand: match,
        processed: true,  // 已经被$$包装，标记为已处理
        wrapped: true
      });
    });

    // 只在有真正未处理的内容时才添加警告
    const unprocessedBlocks = blocks.filter(b => !b.processed);
    if (unprocessedBlocks.length > 0) {
      warnings.push({
        type: 'math_content_unprocessed',
        message: `检测到${unprocessedBlocks.length}个未处理的数学内容`,
        count: unprocessedBlocks.length
      });
    }
  }

  /**
   * 处理行内数学公式（未来实现）
   * @private
   */
  processInlineMath(text, blocks, errors) {
    // TODO: 实现行内数学公式处理
    return text;
  }

  /**
   * 处理显示数学公式（未来实现）
   * @private
   */
  processDisplayMath(text, blocks, errors) {
    // TODO: 实现显示数学公式处理
    return text;
  }

  /**
   * 处理其他数学环境（未来实现）
   * 注意：align环境已通过processAlignEnvironment方法实现
   * @private
   */
  processMathEnvironments(text, blocks, errors) {
    // TODO: 实现其他数学环境处理（equation, gather等）
    // align环境已在processAlignEnvironment中实现
    return text;
  }

  /**
   * KaTeX渲染（未来实现）
   * @private
   */
  renderWithKaTeX(mathContent, displayMode = false) {
    // TODO: 集成KaTeX渲染
    return mathContent;
  }

  /**
   * 为剩余的数学环境添加KaTeX可识别的分隔符包装
   * DocumentParser已处理主要的equation、align等环境，这里处理其他环境
   * @param {string} text - 输入文本
   * @returns {string} 包装后的文本
   * @private
   */
  wrapRemainingMathEnvironments(text) {
    let processedText = text;
    
    
    // 为alignat环境添加$$包装（DocumentParser可能没有处理）
    processedText = processedText.replace(
      /\\begin\{alignat\*?\}(?:\{[^}]*\})?([\s\S]*?)\\end\{alignat\*?\}/g,
      (match, content) => {
        console.log('[MathParser] 为alignat环境添加$$包装:', match.substring(0, 50) + '...');
        return `$$\\begin{alignat}${content}\\end{alignat}$$`;
      }
    );
    
    // 处理独立的矩阵环境
    // 矩阵环境需要在数学模式中才能被KaTeX渲染
    const matrixEnvs = ['matrix', 'pmatrix', 'bmatrix', 'vmatrix', 'Vmatrix'];
    
    matrixEnvs.forEach(env => {
      // 简单的正则：匹配矩阵环境
      const regex = new RegExp(`\\\\begin\\{${env}\\}([\\s\\S]*?)\\\\end\\{${env}\\}`, 'g');
      
      // 查找所有匹配
      let matches = [];
      let match;
      while ((match = regex.exec(processedText)) !== null) {
        matches.push({
          fullMatch: match[0],
          content: match[1],
          index: match.index
        });
      }
      
      // 从后往前处理，避免索引偏移
      for (let i = matches.length - 1; i >= 0; i--) {
        const m = matches[i];
        
        // 检查前后文，判断是否需要包装
        const beforeIndex = Math.max(0, m.index - 100);
        const afterIndex = Math.min(processedText.length, m.index + m.fullMatch.length + 100);
        const before = processedText.substring(beforeIndex, m.index);
        const after = processedText.substring(m.index + m.fullMatch.length, afterIndex);
        
        // 更准确地检查是否已经在数学环境中
        // 检查整个文本，看矩阵环境是否在\[...\]、$$...$$或$...$内部
        const checkInMathDelimiters = () => {
          // 查找所有的\[...\]区域
          const bracketRegex = /\\\[([\s\S]*?)\\\]/g;
          let bracketMatch;
          while ((bracketMatch = bracketRegex.exec(processedText)) !== null) {
            const start = bracketMatch.index;
            const end = start + bracketMatch[0].length;
            // 检查矩阵环境是否在这个区域内
            if (m.index >= start && m.index + m.fullMatch.length <= end) {
              return true;
            }
          }

          // 查找所有的$$...$$区域
          const dollarRegex = /\$\$([\s\S]*?)\$\$/g;
          let dollarMatch;
          while ((dollarMatch = dollarRegex.exec(processedText)) !== null) {
            const start = dollarMatch.index;
            const end = start + dollarMatch[0].length;
            // 检查矩阵环境是否在这个区域内
            if (m.index >= start && m.index + m.fullMatch.length <= end) {
              return true;
            }
          }

          // 查找所有的$...$区域（行内数学）
          const inlineDollarRegex = /(?<!\$)\$(?!\$)([^$\n]*?)\$(?!\$)/g;
          let inlineDollarMatch;
          while ((inlineDollarMatch = inlineDollarRegex.exec(processedText)) !== null) {
            const start = inlineDollarMatch.index;
            const end = start + inlineDollarMatch[0].length;
            // 检查矩阵环境是否在这个区域内
            if (m.index >= start && m.index + m.fullMatch.length <= end) {
              return true;
            }
          }

          return false;
        };
        
        const inMathMode = checkInMathDelimiters();
        
        // 如果不在数学环境中，添加$$包装
        if (!inMathMode) {
          const wrapped = `$$${m.fullMatch}$$`;
          processedText = processedText.substring(0, m.index) + wrapped + 
                         processedText.substring(m.index + m.fullMatch.length);
        } else {
        }
      }
    });
    
    return processedText;
  }

  /**
   * 查找匹配内容在文本中的位置
   * @param {string} text - 原文本
   * @param {string} match - 匹配的内容
   * @returns {object} 位置信息
   * @private
   */
  findPosition(text, match) {
    const index = text.indexOf(match);
    if (index === -1) return { line: 1, column: 1, start: 0, end: 0 };

    const beforeMatch = text.substring(0, index);
    const lines = beforeMatch.split('\n');
    
    return {
      line: lines.length,
      column: lines[lines.length - 1].length + 1,
      start: index,
      end: index + match.length
    };
  }

  /**
   * TDD: 接收并正确包装数学内容的方法
   * @param {Object} mathInput - 包含text和mathBlocks的输入对象
   * @returns {Object} 处理结果
   */
  processWithContext(mathInput) {
    console.log('[MathParser] TDD: 处理带上下文的数学内容');
    
    let processedText = mathInput.text;
    
    if (mathInput.mathBlocks) {
      mathInput.mathBlocks.forEach(block => {
        if (block.type === 'math_environment' && block.environment) {
          console.log(`[MathParser] TDD: 包装${block.environment}环境为$$格式`);
          // 确保输出$$包装格式
          const originalPattern = `\\begin{${block.environment}}${block.content}\\end{${block.environment}}`;
          const wrappedPattern = `$$\\begin{${block.environment}}${block.content}\\end{${block.environment}}$$`;
          
          processedText = processedText.replace(originalPattern, wrappedPattern);
        }
      });
    }
    
    return {
      processedText,
      mathBlockCount: mathInput.mathBlocks ? mathInput.mathBlocks.length : 0
    };
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = MathParser;
  console.log('[MathParser] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境（包括测试环境中的模拟浏览器）
  window.MathParser = MathParser;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.MathParser = MathParser;
  }
}
