/**
 * LaTeX Parser V3 - 主解析器
 * 整合五层架构，提供统一的解析接口
 * 这是五层解析器架构的核心协调器
 */

// 动态获取各层解析器（支持Node.js和浏览器环境）
// 使用立即执行函数避免全局变量污染
const getParsers = () => {
  let parsers = {};
  
  if (typeof require !== 'undefined') {
    // Node.js环境
    try {
      parsers.DocumentParser = require('./DocumentParser');
      parsers.BlockElementParser = require('./BlockElementParser');
      parsers.InlineElementParser = require('./InlineElementParser');
      parsers.MathParser = require('./MathParser');
      parsers.SpecialParser = require('./SpecialParser');
    } catch (e) {
      console.warn('[LaTeXParser] 无法加载某些解析器模块:', e.message);
    }
  } else if (typeof window !== 'undefined') {
    // 浏览器环境 - 直接使用全局变量
    parsers.DocumentParser = window.DocumentParser;
    parsers.BlockElementParser = window.BlockElementParser;
    parsers.InlineElementParser = window.InlineElementParser;
    parsers.MathParser = window.MathParser;
    parsers.SpecialParser = window.SpecialParser;
  }
  
  return parsers;
};

// 获取解析器
const Parsers = getParsers();

class LaTeXParser {
  constructor(config = {}) {
    this.config = {
      mode: 'tolerant',
      enabledLayers: ['document', 'block', 'inline', 'math', 'special'],
      maxRecursionDepth: 100,
      ...config
    };
    
    // 初始化各层解析器
    this.initializeParsers();
    
  }
  
  /**
   * 初始化各层解析器
   */
  initializeParsers() {
    // 检查各层解析器是否可用
    if (Parsers.DocumentParser) {
      this.documentParser = new Parsers.DocumentParser(this.config.document || {});
    }
    
    if (Parsers.BlockElementParser) {
      this.blockParser = new Parsers.BlockElementParser(this.config.block || {});
    }
    
    if (Parsers.InlineElementParser) {
      this.inlineParser = new Parsers.InlineElementParser(this.config.inline || {});
    }
    
    if (Parsers.MathParser) {
      this.mathParser = new Parsers.MathParser(this.config.math || {});
    }
    
    if (Parsers.SpecialParser) {
      this.specialParser = new Parsers.SpecialParser(this.config.special || {});
    }
  }
  
  /**
   * 完整解析流程
   * @param {string} content - 输入的LaTeX内容
   * @returns {object} 解析结果
   */
  parseComplete(content) {
    const startTime = performance.now();
    
    // 空内容处理
    if (!content || content.trim() === '') {
      return this.createEmptyResult();
    }
    
    let processedText = content;
    const allBlocks = [];
    const allWarnings = [];
    const allErrors = [];
    const allPlaceholders = []; // 收集所有占位符
    
    try {
      // 第一层：文档结构解析 (Document Parser)
      if (this.documentParser && this.config.enabledLayers.includes('document')) {
        const docResult = this.documentParser.process(processedText, allPlaceholders);
        processedText = docResult.text;
        allBlocks.push(...(docResult.blocks || []));
        allWarnings.push(...(docResult.warnings || []));
        allErrors.push(...(docResult.errors || []));
        if (docResult.placeholders) {
          allPlaceholders.push(...docResult.placeholders);
        }
      }
      
      // 第二层：块级元素解析 (Block Element Parser)
      if (this.blockParser && this.config.enabledLayers.includes('block')) {
        const blockResult = this.blockParser.process(processedText, allPlaceholders);
        processedText = blockResult.text;
        allBlocks.push(...(blockResult.blocks || []));
        allWarnings.push(...(blockResult.warnings || []));
        allErrors.push(...(blockResult.errors || []));
        if (blockResult.placeholders) {
          allPlaceholders.push(...blockResult.placeholders);
        }
      }
      
      // 第三层：行内元素解析 (Inline Element Parser)
      if (this.inlineParser && this.config.enabledLayers.includes('inline')) {
        const inlineResult = this.inlineParser.process(processedText, allPlaceholders);
        processedText = inlineResult.text;
        allBlocks.push(...(inlineResult.blocks || []));
        allWarnings.push(...(inlineResult.warnings || []));
        allErrors.push(...(inlineResult.errors || []));
        if (inlineResult.placeholders) {
          allPlaceholders.push(...inlineResult.placeholders);
        }
      }
      
      // 第四层：数学公式解析 (Math Parser)
      if (this.mathParser && this.config.enabledLayers.includes('math')) {
        const mathResult = this.mathParser.process(processedText, allPlaceholders);
        processedText = mathResult.text;
        allBlocks.push(...(mathResult.blocks || []));
        allWarnings.push(...(mathResult.warnings || []));
        allErrors.push(...(mathResult.errors || []));
        if (mathResult.placeholders) {
          allPlaceholders.push(...mathResult.placeholders);
        }
      }
      
      // 第五层：特殊处理 (Special Parser)
      if (this.specialParser && this.config.enabledLayers.includes('special')) {
        const specialResult = this.specialParser.process(processedText, allPlaceholders);
        processedText = specialResult.text;
        allBlocks.push(...(specialResult.blocks || []));
        allWarnings.push(...(specialResult.warnings || []));
        allErrors.push(...(specialResult.errors || []));
        if (specialResult.placeholders) {
          allPlaceholders.push(...specialResult.placeholders);
        }
      }
      
    } catch (error) {
      console.error('[LaTeXParser] 解析错误:', error);
      allErrors.push({
        type: 'parse_error',
        message: error.message,
        layer: 'unknown'
      });
    }
    
    // 统一恢复所有占位符（从最后一层开始，逆序恢复）
    processedText = this.restoreAllPlaceholders(processedText, allPlaceholders);
    
    const endTime = performance.now();
    
    // 返回最终结果
    return {
      success: allErrors.length === 0,
      processedText: processedText,
      blocks: allBlocks,
      metadata: {
        parserVersion: '3.0.0',
        processingMethod: 'five-layer-tdd',
        architecture: 'five-layer-tdd',
        processingTime: endTime - startTime,
        nodeCount: allBlocks.length,
        complexityScore: this.calculateComplexity(allBlocks),
        enabledLayers: this.config.enabledLayers,
        layerStats: {
          document: allBlocks.filter(b => b.layer === 'document').length,
          block: allBlocks.filter(b => b.layer === 'block').length,
          inline: allBlocks.filter(b => b.layer === 'inline').length,
          math: allBlocks.filter(b => b.layer === 'math').length,
          special: allBlocks.filter(b => b.layer === 'special').length
        }
      },
      warnings: allWarnings,
      errors: allErrors
    };
  }
  
  /**
   * 创建空结果对象
   */
  createEmptyResult() {
    return {
      success: true,
      processedText: '',
      blocks: [],
      metadata: {
        parserVersion: '3.0.0',
        processingMethod: 'five-layer-tdd-empty',
        architecture: 'five-layer-tdd',
        processingTime: 0,
        nodeCount: 0,
        complexityScore: 0
      },
      warnings: [],
      errors: []
    };
  }
  
  /**
   * 计算内容复杂度
   */
  calculateComplexity(blocks) {
    return blocks.length * 10; // 简化计算
  }
  
  /**
   * 统一恢复所有占位符
   * @param {string} text - 包含占位符的文本
   * @param {Array} placeholders - 所有占位符数组
   * @returns {string} 恢复后的文本
   */
  restoreAllPlaceholders(text, placeholders) {
    let restoredText = text;
    
    // 按层级分组占位符
    const placeholdersByLayer = {};
    placeholders.forEach(p => {
      const layer = p.layer || 0;
      if (!placeholdersByLayer[layer]) {
        placeholdersByLayer[layer] = [];
      }
      placeholdersByLayer[layer].push(p);
    });
    
    // 从高层到低层逆序恢复（L5 -> L4 -> L3 -> L2 -> L1）
    const layers = Object.keys(placeholdersByLayer).sort((a, b) => b - a);
    
    layers.forEach(layer => {
      const layerPlaceholders = placeholdersByLayer[layer];
      layerPlaceholders.forEach(p => {
        // 使用函数返回值方式，避免$被解释为特殊字符
        restoredText = restoredText.replace(p.id, () => p.content);
      });
    });
    
    return restoredText;
  }
}

// 导出：Node.js 环境
if (typeof module !== 'undefined' && module.exports) {
  module.exports = LaTeXParser;
}

// 导出：浏览器环境
if (typeof window !== 'undefined') {
  window.LaTeXParser = LaTeXParser;
}