/**
 * InlineElementParser - 行内元素解析器
 * 负责处理LaTeX行内格式命令
 * 包括：文本格式、字号、引用、特殊字符等
 * 
 * TDD改进：集成MathContentDetector避免干扰数学内容
 */

// 浏览器和Node.js兼容的模块导入 - 内嵌简化版
const MathContentDetector_Inline = (function() {
  if (typeof require !== 'undefined') {
    try { return require('../utils/MathContentDetector'); } catch (e) { /* fallback */ }
  } else if (typeof window !== 'undefined' && window.MathContentDetector) {
    return window.MathContentDetector;
  }
  
  return class MathContentDetector {
    constructor() {
      this.mathEnvironments = ['equation', 'equation*', 'align', 'align*'];
    }
  };
})();

class InlineElementParser {
  constructor(config = {}) {
    this.config = {
      // 文本格式命令映射
      textCommands: {
        'textbf': { tag: 'strong', class: 'latex-bold' },
        'textit': { tag: 'em', class: null },
        'emph': { tag: 'em', class: 'emphasis' },
        'underline': { tag: 'u', class: null },
        'texttt': { tag: 'code', class: null },
        'textsc': { tag: 'span', class: 'small-caps' }
      },
      // 字号命令映射
      sizeCommands: {
        'tiny': { size: '0.5em', class: 'tiny' },
        'scriptsize': { size: '0.7em', class: 'scriptsize' },
        'footnotesize': { size: '0.8em', class: 'footnotesize' },
        'small': { size: '0.9em', class: 'small' },
        'normalsize': { size: '1em', class: 'normalsize' },
        'large': { size: '1.2em', class: 'large' },
        'Large': { size: '1.44em', class: 'Large' },
        'LARGE': { size: '1.73em', class: 'LARGE' },
        'huge': { size: '2.07em', class: 'huge' },
        'Huge': { size: '2.49em', class: 'Huge' }
      },
      // 特殊字符映射
      specialChars: {
        '\\&': '&amp;',
        '\\%': '%',
        '\\$': '$',
        '\\{': '{',
        '\\}': '}',
        '\\_': '_',
        '\\#': '#',
        '\\^{}': '^',
        '\\~{}': '~',
        '\\textbackslash': '\\'
      },
      // 布局控制命令映射
      layoutCommands: {
        // 换行命令
        '\\\\': { type: 'line-break', tag: 'br', className: null },
        'newline': { type: 'line-break', tag: 'br', className: 'newline' },
        'par': { type: 'paragraph-break', tag: 'p', className: 'paragraph-break' },
        // 分页命令
        'newpage': { type: 'page-break', tag: 'div', className: 'page-break' },
        'clearpage': { type: 'clear-page', tag: 'div', className: 'clear-page' },
        // 间距命令
        'hspace': { type: 'horizontal-space', tag: 'span', className: 'hspace' },
        'vspace': { type: 'vertical-space', tag: 'div', className: 'vspace' },
        'hfill': { type: 'horizontal-fill', tag: 'span', className: 'hfill' }
      },
      ...config
    };
    
    // TDD: 初始化数学内容检测器
    this.mathDetector = new MathContentDetector_Inline();
  }

  /**
   * 处理行内元素解析
   * @param {string} text - 输入文本
   * @param {Array} existingPlaceholders - 从其他层传递的占位符
   * @returns {object} 处理结果
   */
  process(text, existingPlaceholders = []) {
    let processedText = text;
    const blocks = [];
    const warnings = [];
    const errors = [];
    const cssRules = new Set();
    const placeholders = [...existingPlaceholders]; // 保留已有占位符

    try {
      // 处理布局控制命令（优先处理，避免与其他命令冲突）
      const layoutResult = this.processLayoutCommands(processedText, blocks, warnings, cssRules, placeholders);
      if (layoutResult && layoutResult.text !== undefined) {
        processedText = layoutResult.text;
        if (layoutResult.placeholders) {
          placeholders.push(...layoutResult.placeholders);
        }
      }

      // 处理字号命令（需要在文本格式之前处理以避免冲突）
      processedText = this.processSizeCommands(processedText, blocks, cssRules);

      // 处理文本格式命令
      processedText = this.processTextCommands(processedText, blocks, warnings, cssRules);

      // 处理引用系统
      processedText = this.processReferenceCommands(processedText, blocks, warnings);

      // 处理特殊字符
      processedText = this.processSpecialCharacters(processedText, blocks, cssRules);

      // 检测未知命令
      this.detectUnknownCommands(processedText, warnings);

    } catch (error) {
      errors.push({
        type: 'inline_parsing_error',
        message: `行内解析错误: ${error.message}`,
        position: { line: 1, column: 1 }
      });
    }

    return {
      text: processedText,
      blocks,
      warnings,
      errors,
      css: this.generateCSS(cssRules),
      placeholders // 返回所有占位符
    };
  }

  /**
   * 处理文本格式命令
   * @private
   */
  processTextCommands(text, blocks, warnings, cssRules) {
    let processedText = text;

    for (const [command, config] of Object.entries(this.config.textCommands)) {
      // 使用智能括号匹配算法处理嵌套的格式化命令
      processedText = this.processCommandWithBraceMatching(
        processedText, command, config, blocks, warnings, cssRules
      );

      // 处理错误的命令格式（缺少大括号）
      const errorRegex = new RegExp(`\\\\${command}(?!\\{)(?![a-zA-Z])`, 'g');
      processedText = processedText.replace(errorRegex, (match) => {
        warnings.push({
          type: 'malformed_command',
          command: match,
          message: `格式化命令缺少大括号: ${match}`,
          suggestion: `建议使用 ${match}{内容}`
        });
        return match; // 保持原样
      });
    }

    return processedText;
  }

  /**
   * 处理字号命令
   * @private
   */
  processSizeCommands(text, blocks, cssRules) {
    let processedText = text;

    for (const [command, config] of Object.entries(this.config.sizeCommands)) {
      // 处理LaTeX标准格式：{\tiny content}（大括号包围的内容）
      const standardRegex = new RegExp(`\\{\\\\${command}\\s+([^}]*)\\}`, 'g');
      processedText = processedText.replace(standardRegex, (match, content) => {
        const className = `latex-${command}`;
        blocks.push({
          type: `${command}_grouped`,
          content: content,
          fontSize: config.size,
          cssClass: className,
          originalCommand: match
        });

        // 添加CSS规则
        cssRules.add(`
          .${className} {
            font-size: ${config.size};
          }
        `);

        // 处理嵌套格式化
        const processedContent = this.processNestedFormatting(content, blocks, [], cssRules);
        return `<span class="${className}">${processedContent}</span>`;
      });

      // 处理带内容的字号命令：\large{content}
      const withContentRegex = new RegExp(`\\\\${command}\\{([^}]*)\\}`, 'g');
      processedText = processedText.replace(withContentRegex, (match, content) => {
        const className = `latex-${command}`;
        blocks.push({
          type: `${command}_with_content`,
          content: content,
          fontSize: config.size,
          cssClass: className,
          originalCommand: match
        });

        // 添加CSS规则
        cssRules.add(`
          .${className} {
            font-size: ${config.size};
          }
        `);

        // 处理嵌套格式化
        const processedContent = this.processNestedFormatting(content, blocks, [], cssRules);
        return `<span class="${className}">${processedContent}</span>`;
      });
    }

    return processedText;
  }

  /**
   * 处理引用系统命令
   * @private
   */
  processReferenceCommands(text, blocks, warnings) {
    let processedText = text;

    // 先保护数学环境内的 \label 和 \ref 命令
    // 重要修复：优先匹配$$包装的数学环境，避免内部的\label被错误处理
    // 匹配顺序：1. $$包装的环境  2. 单独的$$...$$  3. $...$  4. 未包装的数学环境
    const mathEnvPattern = /\$\$\\begin\{(?:equation|align|gather|alignat|multline|eqnarray|pmatrix|bmatrix|vmatrix|Vmatrix|matrix)\*?\}[\s\S]*?\\end\{(?:equation|align|gather|alignat|multline|eqnarray|pmatrix|bmatrix|vmatrix|Vmatrix|matrix)\*?\}\$\$|\$\$[\s\S]*?\$\$|\$[^$]+\$|\\begin\{(equation|align|gather|alignat|multline|eqnarray|pmatrix|bmatrix|vmatrix|Vmatrix|matrix)\*?\}[\s\S]*?\\end\{\1\*?\}/g;
    const mathProtections = [];
    let mathIndex = 0;
    
    // 保护数学环境
    processedText = processedText.replace(mathEnvPattern, (match) => {
      const placeholder = `__MATH_PROTECTION_${mathIndex++}__`;
      mathProtections.push({ placeholder, content: match });
      return placeholder;
    });

    // 处理非数学环境中的 \\label{} 命令
    processedText = processedText.replace(/\\label\{([^}]+)\}/g, (match, labelKey) => {
      blocks.push({
        type: 'label',
        labelKey: labelKey.trim(),
        originalCommand: match,
        position: this.findPosition(text, match)
      });

      return `<span id="${labelKey.trim()}" class="latex-label"></span>`;
    });

    // 处理非数学环境中的 \\ref{} 命令
    processedText = processedText.replace(/\\ref\{([^}]+)\}/g, (match, refKey) => {
      blocks.push({
        type: 'ref',
        refKey: refKey.trim(),
        originalCommand: match,
        position: this.findPosition(text, match)
      });

      // 简单的引用链接（实际应该查找对应的label）
      return `<a href="#${refKey.trim()}" class="latex-ref">[ref]</a>`;
    });

    // 处理\\cite{}命令
    processedText = processedText.replace(/\\cite\{([^}]+)\}/g, (match, citeKey) => {
      blocks.push({
        type: 'cite',
        citeKey: citeKey.trim(),
        originalCommand: match,
        position: this.findPosition(text, match)
      });

      return `<cite class="latex-cite">[${citeKey.trim()}]</cite>`;
    });

    // 恢复被保护的数学环境
    // 重要修复：使用函数返回值避免$被JavaScript的replace解释为特殊字符
    mathProtections.forEach(protection => {
      processedText = processedText.replace(protection.placeholder, () => protection.content);
    });

    return processedText;
  }

  /**
   * 处理特殊字符转义
   * @private
   */
  processSpecialCharacters(text, blocks, cssRules) {
    let processedText = text;

    // 首先处理转义的特殊字符（带反斜杠的）
    // 对于会产生~的转义，使用占位符避免二次处理
    const tildePlaceholder = '___TILDE_PLACEHOLDER___';
    
    for (const [latexChar, htmlChar] of Object.entries(this.config.specialChars)) {
      const regex = new RegExp(this.escapeRegExp(latexChar), 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: 'special_char',
          latexChar: latexChar,
          htmlChar: htmlChar,
          originalCommand: match
        });

        // 如果转换结果是~，使用占位符
        if (htmlChar === '~') {
          return tildePlaceholder;
        }
        return htmlChar;
      });
    }

    // 然后处理不断行空格 ~ (不带反斜杠的)
    processedText = processedText.replace(/~/g, (match) => {
      blocks.push({
        type: 'non_breaking_space',
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(`
        .non-breaking-space {
          white-space: nowrap;
        }
      `);

      return '<span class="non-breaking-space"> </span>';
    });

    // 最后恢复占位符为真正的~字符
    processedText = processedText.replace(new RegExp(this.escapeRegExp(tildePlaceholder), 'g'), '~');

    return processedText;
  }

  /**
   * 检测未知命令并发出警告
   * @private
   */
  detectUnknownCommands(text, warnings) {
    // 查找所有LaTeX命令
    const commandRegex = /\\([a-zA-Z]+)(?:\{[^}]*\})?/g;
    const foundCommands = [];
    let match;

    while ((match = commandRegex.exec(text)) !== null) {
      foundCommands.push(match[1]);
    }

    // 已知命令列表
    const knownCommands = [
      // 文本格式命令
      ...Object.keys(this.config.textCommands),
      // 字号命令
      ...Object.keys(this.config.sizeCommands),
      // 引用命令
      'label', 'ref', 'cite',
      // 文档结构命令（从DocumentParser）
      'section', 'subsection', 'subsubsection', 'paragraph',
      'title', 'author', 'date', 'maketitle',
      // 环境命令（从BlockElementParser）
      'begin', 'end', 'item',
      // 数学相关（暂时列为已知）
      'left', 'right', 'frac', 'sqrt', 'sum', 'int'
    ];

    // 检查未知命令
    const unknownCommands = [...new Set(foundCommands)].filter(cmd => 
      !knownCommands.includes(cmd) && 
      !this.isLikelyMathCommand(cmd)
    );

    unknownCommands.forEach(command => {
      warnings.push({
        type: 'unknown_command',
        command: command,
        message: `未知命令: \\${command}`,
        suggestion: this.suggestAlternative(command)
      });
    });
  }

  /**
   * 检查是否可能是数学命令
   * @private
   */
  isLikelyMathCommand(command) {
    const mathPrefixes = ['math', 'text', 'cal', 'bf', 'rm', 'sf', 'tt', 'it'];
    const greekLetters = [
      'alpha', 'beta', 'gamma', 'delta', 'epsilon', 'zeta', 'eta', 'theta',
      'iota', 'kappa', 'lambda', 'mu', 'nu', 'xi', 'omicron', 'pi',
      'rho', 'sigma', 'tau', 'upsilon', 'phi', 'chi', 'psi', 'omega'
    ];
    
    // 添加常见的数学符号和运算符
    const mathSymbols = [
      // 无穷和极限
      'infty', 'lim', 'limsup', 'liminf', 'to', 'rightarrow', 'leftarrow',
      // 运算符
      'sum', 'prod', 'int', 'oint', 'partial', 'nabla',
      // 关系符号
      'leq', 'geq', 'neq', 'approx', 'equiv', 'sim', 'simeq',
      // 集合符号
      'in', 'notin', 'subset', 'subseteq', 'cup', 'cap', 'emptyset',
      // 逻辑符号
      'forall', 'exists', 'neg', 'land', 'lor',
      // 其他常见符号
      'cdot', 'times', 'div', 'pm', 'mp', 'ast', 'star', 'circ',
      'bullet', 'ldots', 'cdots', 'vdots', 'ddots',
      // 分数和根号
      'frac', 'sqrt', 'root',
      // 三角函数
      'sin', 'cos', 'tan', 'cot', 'sec', 'csc',
      'arcsin', 'arccos', 'arctan', 'sinh', 'cosh', 'tanh',
      // 对数和指数
      'log', 'ln', 'exp',
      // 括号
      'left', 'right', 'big', 'Big', 'bigg', 'Bigg',
      // 矩阵
      'begin', 'end', 'pmatrix', 'bmatrix', 'vmatrix', 'Vmatrix',
      // 其他
      'det', 'dim', 'ker', 'deg', 'gcd', 'lcm', 'min', 'max', 'sup', 'inf'
    ];

    return mathPrefixes.some(prefix => command.startsWith(prefix)) ||
           greekLetters.includes(command.toLowerCase()) ||
           mathSymbols.includes(command);
  }

  /**
   * 建议替代命令
   * @private
   */
  suggestAlternative(command) {
    const suggestions = {
      'bold': 'textbf',
      'italic': 'textit',
      'emphasize': 'emph',
      'code': 'texttt'
    };

    return suggestions[command.toLowerCase()] || null;
  }

  /**
   * 转义正则表达式特殊字符
   * @private
   */
  escapeRegExp(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
  }

  /**
   * 处理布局控制命令
   * @private
   */
  processLayoutCommands(text, blocks, warnings, cssRules, placeholders = []) {
    let processedText = text;
    const newPlaceholders = [];

    // 处理双反斜杠换行命令（包括带可选参数的）
    const lineBreaksResult = this.processLineBreaks(processedText, blocks, warnings, cssRules, placeholders);
    if (lineBreaksResult && lineBreaksResult.text !== undefined) {
      processedText = lineBreaksResult.text;
      if (lineBreaksResult.placeholders) {
        newPlaceholders.push(...lineBreaksResult.placeholders);
      }
    }

    // 处理其他单个布局命令（不需要参数的）
    for (const [command, config] of Object.entries(this.config.layoutCommands)) {
      if (command === '\\\\') continue; // 已经在processLineBreaks中处理
      if (['hspace', 'vspace', 'hfill'].includes(command)) continue; // 在processSpaceCommands中处理
      
      processedText = this.processSingleLayoutCommand(
        processedText, command, config, blocks, warnings, cssRules
      );
    }

    // 处理间距命令（需要参数）
    processedText = this.processSpaceCommands(processedText, blocks, warnings, cssRules);

    // 最后处理段落标记，将整个文本包装在段落标签中
    processedText = this.wrapInParagraphs(processedText, cssRules);

    return processedText;
  }

  /**
   * 处理换行命令（包括带可选参数的\\[length]）
   * @private
   */
  processLineBreaks(text, blocks, warnings, cssRules, placeholders = []) {
    let processedText = text;
    let newPlaceholders = [];

    // 处理带可选参数的双反斜杠 \\[length]
    processedText = processedText.replace(/\\\\\[([^\]]*)\]/g, (match, spacing) => {
      blocks.push({
        type: 'line-break-with-space',
        spacing: spacing,
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(`
        .line-break-with-space {
          display: block;
        }
      `);

      return `<br class="line-break-with-space" style="margin-bottom: ${spacing}">`;
    });

    // 处理\newline命令
    processedText = processedText.replace(/\\newline/g, (match) => {
      blocks.push({
        type: 'newline',
        originalCommand: match
      });
      return '<br class="latex-newline">';
    });

    // 处理普通双反斜杠 \\ (但保护数学公式内容)
    const lineBreakResult = this.processLineBreaksWithMathProtection(processedText, blocks, placeholders);
    processedText = lineBreakResult.text;
    if (lineBreakResult.placeholders) {
      newPlaceholders.push(...lineBreakResult.placeholders);
    }

    return {
      text: processedText,
      placeholders: newPlaceholders
    };
  }

  /**
   * 处理双反斜杠换行，但保护数学公式内容
   * @param {string} text - 输入文本
   * @param {Array} blocks - 块数组
   * @returns {string} 处理后的文本
   * @private
   */
  processLineBreaksWithMathProtection(text, blocks, existingPlaceholders = []) {
    // 先找到所有数学公式内容，用占位符保护
    const mathProtections = [];
    const newPlaceholders = [];
    let protectedText = text;
    
    // 首先检查并跳过其他层产生的占位符（L1, L2, L4, L5）
    const layerPlaceholderPattern = /__L[1245]_[A-Z_]+\d+__/g;
    const hasLayerPlaceholders = layerPlaceholderPattern.test(protectedText);
    if (hasLayerPlaceholders) {
      // 保护这些占位符不被破坏
      protectedText = protectedText.replace(/__L[1245]_[A-Z_]+\d+__/g, (match) => {
        const placeholder = `__L3_KEEP_${mathProtections.length}__`;
        mathProtections.push({
          placeholder,
          content: match,
          type: 'keep_layer_placeholder',
          doNotModify: true
        });
        return placeholder;
      });
      
      // 现在安全地处理双反斜杠（跳过后续的数学保护）
      protectedText = protectedText.replace(/\\\\/g, (match) => {
        blocks.push({
          type: 'line-break',
          originalCommand: match
        });
        return '<br>';
      });
      
      // 暂时恢复占位符用于返回
      mathProtections.forEach((protection) => {
        protectedText = protectedText.replace(protection.placeholder, () => protection.content);
      });
      
      return {
        text: protectedText,
        placeholders: newPlaceholders
      };
    }
    
    // TDD: 保护LaTeX数学环境 - 但跳过已被DocumentParser处理的$$包装环境
    const mathEnvironments = ['align', 'align\\*', 'equation', 'equation\\*', 'gather', 'gather\\*', 'alignat', 'alignat\\*'];
    
    // 首先保护已经包装了$$的数学环境（DocumentParser处理的结果）
    protectedText = protectedText.replace(/\$\$\\begin\{[^}]+\}[\s\S]*?\\end\{[^}]+\}\$\$/g, (match) => {
      const placeholder = `__MATH_WRAPPED_${mathProtections.length}__`;
      mathProtections.push({
        placeholder,
        content: match,
        type: 'wrapped_environment',
        alreadyWrapped: true  // 标记已包装，直接保护不再处理
      });
      return placeholder;
    });
    
    // 然后保护其他未包装的数学环境（较少见的情况）
    mathEnvironments.forEach(env => {
      const regex = new RegExp(`\\\\begin\\{${env}\\}[\\s\\S]*?\\\\end\\{${env}\\}`, 'g');
      protectedText = protectedText.replace(regex, (match) => {
        // 检查是否已经被上面的规则保护了
        if (!match.startsWith('$$') || !match.endsWith('$$')) {
          const placeholder = `__MATH_ENV_${mathProtections.length}__`;
          mathProtections.push({
            placeholder,
            content: match,
            type: 'environment',
            environment: env
          });
          return placeholder;
        }
        return match; // 已经被包装的，不处理
      });
    });
    
    // 重要修复：必须先处理$$...$$，再处理$...$，避免正则冲突
    // TDD: 保护显示数学公式 $$...$$ (修复正则表达式支持多行)
    protectedText = protectedText.replace(/\$\$[\s\S]+?\$\$/g, (match, offset) => {
      const placeholder = `__MATH_DISPLAY_${mathProtections.length}__`;
      mathProtections.push({
        placeholder,
        content: match,
        type: 'display'
      });
      return placeholder;
    });
    
    // 保护行内数学公式 $...$ - 必须在$$...$$之后处理
    protectedText = protectedText.replace(/\$[^$]+\$/g, (match, offset) => {
      const placeholder = `__MATH_INLINE_${mathProtections.length}__`;
      mathProtections.push({
        placeholder,
        content: match,
        type: 'inline'
      });
      return placeholder;
    });
    
    // 现在安全地处理双反斜杠
    protectedText = protectedText.replace(/\\\\/g, (match) => {
      blocks.push({
        type: 'line-break',
        originalCommand: match
      });
      return '<br>';
    });
    
    // 不恢复数学公式内容，保持占位符
    // 将这些占位符转换为L3层的占位符
    mathProtections.forEach((protection, index) => {
      const placeholderExists = protectedText.includes(protection.placeholder);
      
      if (placeholderExists) {
        // 如果是其他层的占位符，保持原样
        if (protection.type === 'keep_layer_placeholder') {
          // 已经保护了，不需要额外处理
        } else {
          // 将临时占位符转换为L3层占位符
          const l3Placeholder = `__L3_MATH_${newPlaceholders.length}__`;
          protectedText = protectedText.replace(protection.placeholder, l3Placeholder);
          newPlaceholders.push({
            id: l3Placeholder,
            content: protection.content,
            type: protection.type,
            layer: 3
          });
        }
      } else {
        console.log(`[InlineElementParser] TDD: ⚠️ 占位符 "${protection.placeholder}" 未找到！可能被其他处理破坏了`);
      }
    });
    
    return {
      text: protectedText,
      placeholders: newPlaceholders
    };
  }

  /**
   * 处理单个布局命令（不需要参数的）
   * @private
   */
  processSingleLayoutCommand(text, command, config, blocks, warnings, cssRules) {
    let processedText = text;
    const commandPattern = `\\${command}`;

    processedText = processedText.replace(new RegExp(`\\\\${command}(?![a-zA-Z])`, 'g'), (match) => {
      blocks.push({
        type: config.type,
        command: command,
        originalCommand: match
      });

      // 添加CSS规则
      if (config.className) {
        cssRules.add(this.getLayoutCommandCSS(config));
      }

      // 特殊处理段落命令 - 这里只做标记，后续会包装整个文档
      if (config.type === 'paragraph-break') {
        return '§PARAGRAPH_BREAK§';
      }

      // 生成HTML
      const className = config.className ? ` class="${config.className}"` : '';
      if (config.tag === 'br') {
        return `<${config.tag}${className}>`;
      } else {
        return `<${config.tag}${className}></${config.tag}>`;
      }
    });

    return processedText;
  }

  /**
   * 处理间距命令（需要参数的）
   * @private
   */
  processSpaceCommands(text, blocks, warnings, cssRules) {
    let processedText = text;

    // 处理 \hspace{length}
    processedText = processedText.replace(/\\hspace\{([^}]*)\}/g, (match, length) => {
      if (!length.trim()) {
        warnings.push({
          type: 'empty_space_command',
          command: 'hspace',
          message: 'hspace命令缺少长度参数'
        });
        return match;
      }

      if (!this.isValidSpaceUnit(length)) {
        warnings.push({
          type: 'invalid_space_unit',
          command: 'hspace',
          unit: length,
          message: `hspace命令使用了无效的单位: ${length}`
        });
      }

      blocks.push({
        type: 'horizontal-space',
        length: length,
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(`
        .hspace {
          display: inline-block;
        }
      `);

      return `<span class="hspace" style="display: inline-block; width: ${length};"></span>`;
    });

    // 处理 \vspace{length}
    processedText = processedText.replace(/\\vspace\{([^}]*)\}/g, (match, length) => {
      if (!length.trim()) {
        warnings.push({
          type: 'empty_space_command',
          command: 'vspace',
          message: 'vspace命令缺少长度参数'
        });
        return match;
      }

      if (!this.isValidSpaceUnit(length)) {
        warnings.push({
          type: 'invalid_space_unit',
          command: 'vspace',
          unit: length,
          message: `vspace命令使用了无效的单位: ${length}`
        });
      }

      blocks.push({
        type: 'vertical-space',
        length: length,
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(`
        .vspace {
          display: block;
        }
      `);

      return `<div class="vspace" style="height: ${length};"></div>`;
    });

    // 处理 \hfill
    processedText = processedText.replace(/\\hfill(?![a-zA-Z])/g, (match) => {
      blocks.push({
        type: 'horizontal-fill',
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(`
        .hfill {
          flex: 1;
          display: inline-block;
        }
      `);

      return '<span class="hfill" style="flex: 1; display: inline-block;"></span>';
    });

    return processedText;
  }

  /**
   * 将文本包装在段落标签中，处理段落分隔符
   * @private
   */
  wrapInParagraphs(text, cssRules) {
    // 如果没有段落分隔符，直接返回
    if (!text.includes('§PARAGRAPH_BREAK§')) {
      return text;
    }

    // 添加段落相关的CSS
    cssRules.add(`
      .paragraph-break {
        margin: 1em 0;
        text-indent: 0;
      }
    `);

    // 分割文本并包装在段落标签中
    const paragraphs = text.split('§PARAGRAPH_BREAK§');
    return paragraphs
      .map(p => p.trim())
      .filter(p => p.length > 0)
      .map(p => `<p class="paragraph-break">${p}</p>`)
      .join('');
  }

  /**
   * 检查空间单位是否有效
   * @private
   */
  isValidSpaceUnit(unit) {
    const validUnits = /(em|ex|in|cm|mm|pt|pc|px|%|\d+(\.\d+)?)$/;
    return validUnits.test(unit.trim());
  }

  /**
   * 获取布局命令的CSS样式
   * @private
   */
  getLayoutCommandCSS(config) {
    const className = config.className;
    
    switch (config.type) {
      case 'line-break':
        return `
          .${className} {
            display: block;
            margin: 0;
          }
        `;
      case 'paragraph-break':
        return `
          .${className} {
            margin: 1em 0;
            text-indent: 0;
          }
        `;
      case 'page-break':
        return `
          .${className} {
            page-break-before: always;
            height: 0;
            margin: 0;
          }
        `;
      case 'clear-page':
        return `
          .${className} {
            page-break-before: always;
            clear: both;
            height: 0;
            margin: 0;
          }
        `;
      default:
        return '';
    }
  }

  /**
   * 使用智能括号匹配处理命令
   * @private
   */
  processCommandWithBraceMatching(text, command, config, blocks, warnings, cssRules) {
    let processedText = text;
    const commandPattern = `\\${command}{`;
    
    let searchIndex = 0;
    while (true) {
      const commandIndex = processedText.indexOf(commandPattern, searchIndex);
      if (commandIndex === -1) break;
      
      const braceStart = commandIndex + commandPattern.length - 1; // 位置在 '{'
      const content = this.extractBalancedBraces(processedText, braceStart);
      
      if (content === null) {
        // 无法找到匹配的大括号，生成警告
        warnings.push({
          type: 'unmatched_brace',
          command: commandPattern.slice(0, -1), // 移除最后的 '{'
          message: `命令缺少匹配的大括号: ${commandPattern.slice(0, -1)}`,
          position: { line: 1, column: commandIndex + 1 }
        });
        searchIndex = commandIndex + commandPattern.length;
        continue;
      }
      
      const fullMatch = processedText.substring(commandIndex, braceStart + content.length + 2);
      
      blocks.push({
        type: command,
        content: content,
        htmlTag: config.tag,
        cssClass: config.class,
        originalCommand: fullMatch,
        position: this.findPosition(text, fullMatch)
      });

      // 添加CSS规则
      if (config.class) {
        cssRules.add(this.getCSSRuleForClass(config.class));
      }

      // 递归处理嵌套内容
      const processedContent = this.processNestedFormatting(content, blocks, warnings, cssRules);

      // 生成HTML
      const className = config.class ? ` class="${config.class}"` : '';
      const replacement = `<${config.tag}${className}>${processedContent}</${config.tag}>`;
      
      // 替换文本
      processedText = processedText.substring(0, commandIndex) + 
                    replacement + 
                    processedText.substring(braceStart + content.length + 2);
      
      // 更新搜索位置
      searchIndex = commandIndex + replacement.length;
    }
    
    return processedText;
  }

  /**
   * 提取平衡的大括号内容
   * @private
   */
  extractBalancedBraces(text, startIndex) {
    if (text[startIndex] !== '{') return null;
    
    let braceCount = 1;
    let index = startIndex + 1;
    let content = '';
    
    while (index < text.length && braceCount > 0) {
      const char = text[index];
      
      if (char === '{') {
        braceCount++;
      } else if (char === '}') {
        braceCount--;
      } else if (char === '\\' && index + 1 < text.length) {
        // 处理转义字符
        content += char + text[index + 1];
        index += 2;
        continue;
      }
      
      if (braceCount > 0) {
        content += char;
      }
      
      index++;
    }
    
    return braceCount === 0 ? content : null;
  }

  /**
   * 处理嵌套格式化
   * @private
   */
  processNestedFormatting(content, blocks, warnings, cssRules) {
    // 递归处理内容中的格式化命令，使用智能括号匹配
    let processedContent = content;
    
    // 处理文本格式命令
    for (const [command, config] of Object.entries(this.config.textCommands)) {
      processedContent = this.processCommandWithBraceMatching(
        processedContent, command, config, blocks, warnings, cssRules
      );
    }

    return processedContent;
  }

  /**
   * 获取CSS类的规则
   * @private
   */
  getCSSRuleForClass(className) {
    const cssRules = {
      'small-caps': `
        .small-caps {
          font-variant: small-caps;
        }
      `,
      'emphasis': `
        .emphasis {
          font-style: italic;
          font-weight: normal;
        }
      `,
      'emph': `
        .emph {
          font-style: italic;
          font-weight: normal;
        }
      `,
      'texttt': `
        .texttt {
          font-family: 'Courier New', 'Liberation Mono', monospace;
          background-color: #f5f5f5;
          padding: 0.2em 0.4em;
          border-radius: 3px;
        }
      `
    };

    return cssRules[className] || '';
  }

  /**
   * 生成CSS样式
   * @private
   */
  generateCSS(cssRules) {
    if (cssRules.size === 0) return undefined;
    
    return Array.from(cssRules)
      .filter(rule => rule.trim())
      .join('\n');
  }

  /**
   * 查找命令在文本中的位置
   * @private
   */
  findPosition(text, match) {
    const index = text.indexOf(match);
    if (index === -1) return { line: 1, column: 1 };

    const beforeMatch = text.substring(0, index);
    const lines = beforeMatch.split('\n');
    
    return {
      line: lines.length,
      column: lines[lines.length - 1].length + 1,
      start: index,
      end: index + match.length
    };
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = InlineElementParser;
  console.log('[InlineElementParser] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境（包括测试环境中的模拟浏览器）
  window.InlineElementParser = InlineElementParser;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.InlineElementParser = InlineElementParser;
  }
}
