/**
 * DocumentParser - 文档结构解析器
 * 负责处理LaTeX文档级别的结构命令
 * 包括：章节、标题、文档环境等
 * 
 * TDD改进：集成MathContentDetector实现统一数学内容识别
 */

// 浏览器和Node.js兼容的模块导入 - 内嵌简化版数学检测器
const MathContentDetector_Doc = (function() {
    if (typeof require !== 'undefined') {
      try {
        return require('../utils/MathContentDetector');
      } catch (e) {
        // Fallback to embedded version if file doesn't exist
      }
    } else if (typeof window !== 'undefined' && window.MathContentDetector) {
      return window.MathContentDetector;
    }
    
    // 内嵌简化版数学内容检测器
    return class MathContentDetector {
      constructor() {
        // 分别列出带星号和不带星号的环境（排除矩阵环境）
        this.mathEnvironments = ['equation*', 'equation', 'align*', 'align', 'gather*', 'gather'];
      }
      
      detectMathEnvironments(text) {
        // 为了向后兼容，调用extractMathBlocks并转换格式
        const mathBlocks = this.extractMathBlocks(text);
        return mathBlocks.map(block => ({
          type: 'math_environment',
          environment: block.environment,
          content: block.content.replace(/^\\begin\{[^}]+\}/, '').replace(/\\end\{[^}]+\}$/, '').trim(),
          fullMatch: block.content,
          start: block.startIndex,
          end: block.endIndex
        }));
      }
      
      // 添加extractMathBlocks方法以兼容DocumentParser
      extractMathBlocks(text) {
        const blocks = [];
        
        // 扩展环境列表，包含所有数学环境（排除矩阵环境，由KaTeX处理）
        const allMathEnvs = [
          'equation*', 'equation',
          'align*', 'align',
          'alignat*', 'alignat',
          'gather*', 'gather',
          'multline*', 'multline',
          'split', 'array'
          // 移除矩阵环境：'matrix', 'pmatrix', 'bmatrix', 'vmatrix', 'Vmatrix', 'Bmatrix'
          // 这些环境由KaTeX的宏定义处理，不应创建占位符
        ];
        
        // 为每个环境创建独立的正则表达式
        allMathEnvs.forEach(env => {
          // 转义环境名中的特殊字符（如 * ）
          const escapedEnv = env.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
          const regex = new RegExp(
            `\\\\begin\\{${escapedEnv}\\}([\\s\\S]*?)\\\\end\\{${escapedEnv}\\}`,
            'g'
          );
          
          let match;
          while ((match = regex.exec(text)) !== null) {
            // 检查是否与已有块重叠
            let isOverlapping = false;
            for (const block of blocks) {
              // 检查完全重叠或部分重叠
              if ((match.index >= block.startIndex && match.index < block.endIndex) ||
                  (block.startIndex >= match.index && block.startIndex < match.index + match[0].length)) {
                isOverlapping = true;
                break;
              }
            }
            
            if (!isOverlapping) {
              blocks.push({
                type: 'display_environment',
                environment: env,
                content: match[0],
                startIndex: match.index,
                endIndex: match.index + match[0].length
              });
            }
          }
        });
        
        // 按起始位置排序
        blocks.sort((a, b) => a.startIndex - b.startIndex);
        
        
        return blocks;
      }
    };
  })();

  class DocumentParser {
  constructor(config = {}) {
    this.config = {
      // 默认章节映射
      sectionMapping: {
        'part': { level: 1, tag: 'h1', class: 'part' },
        'chapter': { level: 1, tag: 'h1', class: 'chapter' },
        'section': { level: 2, tag: 'h2', class: 'section' },
        'subsection': { level: 3, tag: 'h3', class: 'subsection' },
        'subsubsection': { level: 4, tag: 'h4', class: 'subsubsection' },
        'paragraph': { level: 5, tag: 'h5', class: 'paragraph' }
      },
      ...config
    };
    
    // TDD: 初始化数学内容检测器
    this.mathDetector = new MathContentDetector_Doc();
  }

  /**
   * 处理文档结构解析
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  process(text) {
    let processedText = text;
    const blocks = [];
    const placeholders = []; // 新增：存储占位符
    const warnings = [];
    const errors = [];

    try {
      // 关键修复：首先处理数学环境，添加$$包装
      // 这必须在处理其他命令之前，避免\label等命令被替换
      const mathResult = this.processMathEnvironments(processedText, blocks);
      processedText = mathResult.text;
      placeholders.push(...(mathResult.placeholders || []));
      
      // 处理各级章节命令
      for (const [command, config] of Object.entries(this.config.sectionMapping)) {
        processedText = this.processSectionCommand(
          processedText, 
          command, 
          config, 
          blocks
        );
      }

      // 处理文档环境
      processedText = this.processDocumentEnvironment(processedText, blocks, warnings);

      // 处理文档元数据
      processedText = this.processDocumentMetadata(processedText, blocks, warnings);

    } catch (error) {
      errors.push({
        type: 'document_parsing_error',
        message: `文档解析错误: ${error.message}`,
        position: { line: 1, column: 1 }
      });
    }

    return {
      text: processedText,
      blocks,
      placeholders, // 新增：返回占位符数组
      warnings,
      errors
    };
  }

  /**
   * 处理章节命令
   * @private
   */
  processSectionCommand(text, command, config, blocks) {
    const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
    
    return text.replace(regex, (match, title) => {
      blocks.push({
        type: command,
        level: config.level,
        title: title.trim(),
        htmlTag: config.tag,
        cssClass: config.class,
        originalCommand: match,
        position: this.findPosition(text, match)
      });

      // 生成HTML输出 - 保持向后兼容，不添加class属性
      return `<${config.tag}>${title.trim()}</${config.tag}>`;
    });
  }

  /**
   * 处理文档环境 \\begin{document}...\\end{document}
   * @private
   */
  processDocumentEnvironment(text, blocks, warnings) {
    const docEnvRegex = /\\begin\{document\}([\s\S]*?)\\end\{document\}/;
    const match = text.match(docEnvRegex);
    
    if (match) {
      blocks.push({
        type: 'document_environment',
        content: match[1].trim(),
        originalCommand: match[0]
      });

      return text.replace(docEnvRegex, (fullMatch, content) => {
        return `<main class="document">${content.trim()}</main>`;
      });
    }

    // 检查是否有未匹配的document环境
    if (text.includes('\\begin{document}') && !text.includes('\\end{document}')) {
      warnings.push({
        type: 'unmatched_environment',
        environment: 'document',
        message: '文档环境未正确关闭'
      });
    }

    return text;
  }

  /**
   * 处理文档元数据（标题、作者、日期等）
   * @private
   */
  processDocumentMetadata(text, blocks, warnings) {
    let processedText = text;
    
    // 关键修复：先保护已经被$$包装的数学环境，避免内部的\label被替换
    const mathProtections = [];
    processedText = processedText.replace(/\$\$[\s\S]*?\$\$/g, (match) => {
      const placeholder = `__PROTECTED_MATH_${mathProtections.length}__`;
      mathProtections.push(match);
      return placeholder;
    });

    // 处理\\title{}命令
    processedText = processedText.replace(/\\title\{([^}]+)\}/g, (match, title) => {
      blocks.push({
        type: 'document_title',
        title: title.trim(),
        originalCommand: match
      });
      return `<h1 class="document-title">${title.trim()}</h1>`;
    });

    // 处理\\author{}命令
    processedText = processedText.replace(/\\author\{([^}]+)\}/g, (match, author) => {
      blocks.push({
        type: 'document_author',
        author: author.trim(),
        originalCommand: match
      });
      return `<div class="document-author">${author.trim()}</div>`;
    });

    // 处理\\date{}命令
    processedText = processedText.replace(/\\date\{([^}]+)\}/g, (match, date) => {
      blocks.push({
        type: 'document_date',
        date: date.trim(),
        originalCommand: match
      });
      return `<div class="document-date">${date.trim()}</div>`;
    });

    // 处理\\maketitle命令
    processedText = processedText.replace(/\\maketitle/g, (match) => {
      blocks.push({
        type: 'make_title',
        originalCommand: match
      });
      return '<header class="document-header"></header>';
    });

    // 处理配置命令 - 这些命令应该被忽略，不在HTML中显示
    const configCommands = [
      'geometry',
      'usepackage', 
      'documentclass',
      'pagestyle',
      'pagenumbering',
      'setlength',
      'renewcommand',
      'newcommand'
    ];
    
    configCommands.forEach(command => {
      // 处理带参数的命令，如 \geometry{...}
      const regex = new RegExp(`\\\\${command}\\{[^}]*\\}`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return ''; // 返回空字符串，完全移除
      });
      
      // 处理可选参数的命令，如 \documentclass[options]{class}
      const regexWithOptions = new RegExp(`\\\\${command}\\[[^\\]]*\\]\\{[^}]*\\}`, 'g');
      processedText = processedText.replace(regexWithOptions, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return ''; // 返回空字符串，完全移除
      });
    });

    // 处理特殊命令
    // 处理 \today 命令
    processedText = processedText.replace(/\\today/g, (match) => {
      blocks.push({
        type: 'today',
        originalCommand: match,
        ignored: true
      });
      return ''; // 返回空字符串，完全移除
    });

    // 处理文本格式化命令
    const formattingCommands = {
      'textbf': { tag: 'strong', class: 'latex-bold' },
      'textit': { tag: 'em', class: 'latex-italic' }, 
      'texttt': { tag: 'code', class: 'latex-monospace' },
      'underline': { tag: 'u', class: 'latex-underline' },
      'emph': { tag: 'em', class: 'latex-emphasis' },
      'textsc': { tag: 'span', class: 'latex-smallcaps' }
    };

    Object.entries(formattingCommands).forEach(([command, config]) => {
      const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
      processedText = processedText.replace(regex, (match, content) => {
        blocks.push({
          type: `formatting_${command}`,
          content: content.trim(),
          originalCommand: match
        });
        return `<${config.tag} class="${config.class}">${content.trim()}</${config.tag}>`;
      });
    });

    // 处理环境命令
    // 处理 center 环境
    processedText = processedText.replace(/\\begin\{center\}([\s\S]*?)\\end\{center\}/g, (match, content) => {
      blocks.push({
        type: 'center_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<div class="latex-center">${content.trim()}</div>`;
    });

    // 处理 quote 环境
    processedText = processedText.replace(/\\begin\{quote\}([\s\S]*?)\\end\{quote\}/g, (match, content) => {
      blocks.push({
        type: 'quote_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<blockquote class="latex-quote">${content.trim()}</blockquote>`;
    });

    // 处理 quotation 环境
    processedText = processedText.replace(/\\begin\{quotation\}([\s\S]*?)\\end\{quotation\}/g, (match, content) => {
      blocks.push({
        type: 'quotation_environment', 
        content: content.trim(),
        originalCommand: match
      });
      return `<blockquote class="latex-quotation">${content.trim()}</blockquote>`;
    });

    // 处理列表环境
    // 处理 itemize 环境（无序列表）
    processedText = processedText.replace(/\\begin\{itemize\}([\s\S]*?)\\end\{itemize\}/g, (match, content) => {
      const items = content.split('\\item').filter(item => item.trim()).map(item => 
        `<li>${item.trim()}</li>`
      ).join('');
      
      blocks.push({
        type: 'itemize_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<ul class="latex-itemize">${items}</ul>`;
    });

    // 处理 enumerate 环境（有序列表）
    processedText = processedText.replace(/\\begin\{enumerate\}([\s\S]*?)\\end\{enumerate\}/g, (match, content) => {
      const items = content.split('\\item').filter(item => item.trim()).map(item => 
        `<li>${item.trim()}</li>`
      ).join('');
      
      blocks.push({
        type: 'enumerate_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<ol class="latex-enumerate">${items}</ol>`;
    });

    // 处理 description 环境（描述列表）
    processedText = processedText.replace(/\\begin\{description\}([\s\S]*?)\\end\{description\}/g, (match, content) => {
      const items = content.replace(/\\item\[([^\]]+)\]/g, '<dt>$1</dt><dd>')
                          .replace(/\\item/g, '<dd>')
                          .replace(/<dd>([^<]*?)(?=<dt>|$)/g, '<dd>$1</dd>');
      
      blocks.push({
        type: 'description_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<dl class="latex-description">${items}</dl>`;
    });

    // 处理页面控制命令
    processedText = processedText.replace(/\\newpage/g, (match) => {
      blocks.push({
        type: 'newpage',
        originalCommand: match
      });
      return '<div class="latex-pagebreak"></div>';
    });

    processedText = processedText.replace(/\\clearpage/g, (match) => {
      blocks.push({
        type: 'clearpage', 
        originalCommand: match
      });
      return '<div class="latex-clearpage"></div>';
    });

    // 处理表格结构命令（基础支持）
    processedText = processedText.replace(/\\tableofcontents/g, (match) => {
      blocks.push({
        type: 'tableofcontents',
        originalCommand: match
      });
      return '<nav class="latex-toc"><p>目录将自动生成</p></nav>';
    });

    // 处理章节标记命令
    const chapterCommands = {
      'part': { tag: 'h1', class: 'latex-part' },
      'chapter': { tag: 'h1', class: 'latex-chapter' }
    };

    Object.entries(chapterCommands).forEach(([command, config]) => {
      const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
      processedText = processedText.replace(regex, (match, title) => {
        blocks.push({
          type: command,
          title: title.trim(),
          originalCommand: match
        });
        return `<${config.tag} class="${config.class}">${title.trim()}</${config.tag}>`;
      });
    });

    // 处理URL和超链接命令
    processedText = processedText.replace(/\\url\{([^}]+)\}/g, (match, url) => {
      blocks.push({
        type: 'url',
        url: url.trim(),
        originalCommand: match
      });
      return `<a href="${url.trim()}" class="latex-url" target="_blank">${url.trim()}</a>`;
    });

    processedText = processedText.replace(/\\href\{([^}]+)\}\{([^}]+)\}/g, (match, url, text) => {
      blocks.push({
        type: 'href',
        url: url.trim(),
        text: text.trim(),
        originalCommand: match
      });
      return `<a href="${url.trim()}" class="latex-href" target="_blank">${text.trim()}</a>`;
    });

    // 处理特殊字符和符号命令
    const symbolCommands = {
      'LaTeX': 'LaTeX',
      'TeX': 'TeX',
      'ldots': '…',
      'textbackslash': '\\',
      'textasciicircum': '^',
      'textasciitilde': '~',
      'textbar': '|',
      'textless': '<',
      'textgreater': '>',
      'textdollar': '$',
      'textpercent': '%',
      'textampersand': '&',
      'texthash': '#',
      'textunderscore': '_',
      'textregistered': '®',
      'texttrademark': '™',
      'textcopyright': '©'
    };

    Object.entries(symbolCommands).forEach(([command, symbol]) => {
      const regex = new RegExp(`\\\\${command}(?!\\w)`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: `symbol_${command}`,
          symbol: symbol,
          originalCommand: match
        });
        return `<span class="latex-symbol">${symbol}</span>`;
      });
    });

    // 处理间距命令
    const spacingCommands = {
      'quad': '<span class="latex-quad"> </span>',
      'qquad': '<span class="latex-qquad">  </span>',
      'hspace': '', // 需要参数处理
      'vspace': '', // 需要参数处理
      'smallskip': '<div class="latex-smallskip"></div>',
      'medskip': '<div class="latex-medskip"></div>',
      'bigskip': '<div class="latex-bigskip"></div>'
    };

    Object.entries(spacingCommands).forEach(([command, html]) => {
      if (html) {
        const regex = new RegExp(`\\\\${command}(?!\\w)`, 'g');
        processedText = processedText.replace(regex, (match) => {
          blocks.push({
            type: `spacing_${command}`,
            originalCommand: match
          });
          return html;
        });
      }
    });

    // 处理带参数的间距命令
    processedText = processedText.replace(/\\hspace\{([^}]+)\}/g, (match, space) => {
      blocks.push({
        type: 'hspace',
        space: space.trim(),
        originalCommand: match
      });
      return `<span class="latex-hspace" style="display:inline-block;width:${space.trim()}"></span>`;
    });

    processedText = processedText.replace(/\\vspace\{([^}]+)\}/g, (match, space) => {
      blocks.push({
        type: 'vspace',
        space: space.trim(),
        originalCommand: match
      });
      return `<div class="latex-vspace" style="height:${space.trim()}"></div>`;
    });

    // 处理脚注命令
    processedText = processedText.replace(/\\footnote\{([^}]+)\}/g, (match, content) => {
      const footnoteId = `footnote-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      blocks.push({
        type: 'footnote',
        id: footnoteId,
        content: content.trim(),
        originalCommand: match
      });
      return `<sup><a href="#${footnoteId}" class="latex-footnote" title="${content.trim()}">[*]</a></sup>`;
    });

    // 处理引用命令
    const refCommands = {
      'ref': 'latex-ref',
      'pageref': 'latex-pageref',
      'cite': 'latex-cite',
      'label': 'latex-label'
    };

    Object.entries(refCommands).forEach(([command, className]) => {
      const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
      processedText = processedText.replace(regex, (match, refKey) => {
        blocks.push({
          type: command,
          refKey: refKey.trim(),
          originalCommand: match
        });
        if (command === 'label') {
          return `<span id="${refKey.trim()}" class="${className}"></span>`;
        } else {
          return `<a href="#${refKey.trim()}" class="${className}">[${refKey.trim()}]</a>`;
        }
      });
    });

    // TDD: 数学环境已经在process方法开始时处理了，这里不再重复调用
    // processedText = this.processMathEnvironments(processedText, blocks);

    // 处理表格环境（基础支持）
    processedText = processedText.replace(/\\begin\{tabular\}(?:\[[^\]]*\])?\{([^}]+)\}([\s\S]*?)\\end\{tabular\}/g, (match, colSpec, content) => {
      blocks.push({
        type: 'tabular_environment',
        columnSpec: colSpec.trim(),
        content: content.trim(),
        originalCommand: match
      });
      
      // 简单的表格处理
      const rows = content.trim().split('\\\\').filter(row => row.trim());
      const tableRows = rows.map(row => {
        const cells = row.split('&').map(cell => cell.trim());
        return '<tr>' + cells.map(cell => `<td>${cell}</td>`).join('') + '</tr>';
      }).join('');
      
      return `<table class="latex-tabular">${tableRows}</table>`;
    });

    // 处理图像环境
    processedText = processedText.replace(/\\begin\{figure\}(?:\[[^\]]*\])?([\s\S]*?)\\end\{figure\}/g, (match, content) => {
      blocks.push({
        type: 'figure_environment',
        content: content.trim(),
        originalCommand: match
      });
      return `<figure class="latex-figure">${content.trim()}</figure>`;
    });

    // 处理标题命令（caption）
    processedText = processedText.replace(/\\caption\{([^}]+)\}/g, (match, caption) => {
      blocks.push({
        type: 'caption',
        caption: caption.trim(),
        originalCommand: match
      });
      return `<figcaption class="latex-caption">${caption.trim()}</figcaption>`;
    });

    // 处理文档级配置命令（直接移除）
    const documentConfigCommands = [
      'pagestyle', 'pagenumbering', 'setlength', 'addtolength',
      'linespread', 'baselinestretch', 'parindent', 'parskip',
      'textwidth', 'textheight', 'oddsidemargin', 'evensidemargin',
      'topmargin', 'headheight', 'headsep', 'footskip'
    ];

    documentConfigCommands.forEach(command => {
      const regex = new RegExp(`\\\\${command}\\{[^}]*\\}`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return '';
      });
    });

    // 处理更多文本格式命令
    const additionalFormattingCommands = {
      'textrm': { tag: 'span', class: 'latex-roman' },
      'textsf': { tag: 'span', class: 'latex-sansserif' },
      'textmd': { tag: 'span', class: 'latex-medium' },
      'textup': { tag: 'span', class: 'latex-upright' },
      'textsl': { tag: 'span', class: 'latex-slanted' }
    };

    Object.entries(additionalFormattingCommands).forEach(([command, config]) => {
      const regex = new RegExp(`\\\\${command}\\{([^}]+)\\}`, 'g');
      processedText = processedText.replace(regex, (match, content) => {
        blocks.push({
          type: `formatting_${command}`,
          content: content.trim(),
          originalCommand: match
        });
        return `<${config.tag} class="${config.class}">${content.trim()}</${config.tag}>`;
      });
    });

    // 处理命令行和代码环境
    const codeEnvironments = ['verbatim', 'lstlisting', 'minted'];
    codeEnvironments.forEach(env => {
      const regex = new RegExp(`\\\\begin\\{${env}\\}([\\s\\S]*?)\\\\end\\{${env}\\}`, 'g');
      processedText = processedText.replace(regex, (match, content) => {
        blocks.push({
          type: `${env}_environment`,
          content: content,
          originalCommand: match
        });
        return `<pre class="latex-${env}"><code>${content}</code></pre>`;
      });
    });

    // 处理边距和布局命令（移除）
    const layoutCommands = [
      'marginpar', 'marginparwidth', 'marginparsep',
      'columnsep', 'columnwidth', 'linewidth'
    ];

    layoutCommands.forEach(command => {
      const regex = new RegExp(`\\\\${command}\\{[^}]*\\}`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return '';
      });
    });

    // 处理字体大小命令
    const fontSizeCommands = {
      'tiny': 'latex-tiny',
      'scriptsize': 'latex-scriptsize',
      'footnotesize': 'latex-footnotesize',
      'small': 'latex-small',
      'normalsize': 'latex-normalsize',
      'large': 'latex-large',
      'Large': 'latex-Large',
      'LARGE': 'latex-LARGE',
      'huge': 'latex-huge',
      'Huge': 'latex-Huge'
    };

    Object.entries(fontSizeCommands).forEach(([command, className]) => {
      // 处理环境形式: {\small text}
      const envRegex = new RegExp(`\\{\\\\${command}\\s+([^}]+)\\}`, 'g');
      processedText = processedText.replace(envRegex, (match, content) => {
        blocks.push({
          type: `fontsize_${command}`,
          content: content.trim(),
          originalCommand: match
        });
        return `<span class="${className}">${content.trim()}</span>`;
      });

      // 处理命令形式: \small
      const cmdRegex = new RegExp(`\\\\${command}(?!\\w)`, 'g');
      processedText = processedText.replace(cmdRegex, (match) => {
        blocks.push({
          type: `fontsize_${command}`,
          originalCommand: match
        });
        return `<span class="${className}">`;
      });
    });

    // 处理分栏命令
    processedText = processedText.replace(/\\twocolumn/g, (match) => {
      blocks.push({
        type: 'twocolumn',
        originalCommand: match
      });
      return '<div class="latex-twocolumn">';
    });

    processedText = processedText.replace(/\\onecolumn/g, (match) => {
      blocks.push({
        type: 'onecolumn',
        originalCommand: match
      });
      return '</div><div class="latex-onecolumn">';
    });

    // 处理附录命令
    processedText = processedText.replace(/\\appendix/g, (match) => {
      blocks.push({
        type: 'appendix',
        originalCommand: match
      });
      return '<div class="latex-appendix-marker"></div>';
    });

    // 处理索引命令（移除）
    const indexCommands = ['index', 'printindex', 'makeindex'];
    indexCommands.forEach(command => {
      const regex = new RegExp(`\\\\${command}(?:\\{[^}]*\\})?`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return '';
      });
    });

    // 处理词汇表命令（移除）
    const glossaryCommands = ['glossary', 'printglossary', 'makeglossary'];
    glossaryCommands.forEach(command => {
      const regex = new RegExp(`\\\\${command}(?:\\{[^}]*\\})?`, 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: command,
          originalCommand: match,
          ignored: true
        });
        return '';
      });
    });

    // 处理特殊符号转义
    const escapeChars = {
      '\\{': '{',
      '\\}': '}',
      '\\$': '$',
      '\\&': '&',
      '\\%': '%',
      '\\#': '#',
      '\\_': '_'
    };

    Object.entries(escapeChars).forEach(([escaped, char]) => {
      const regex = new RegExp(escaped.replace(/[\\{}$&%#_]/g, '\\$&'), 'g');
      processedText = processedText.replace(regex, (match) => {
        blocks.push({
          type: 'escape_char',
          char: char,
          originalCommand: match
        });
        return `<span class="latex-escaped">${char}</span>`;
      });
    });

    // 恢复被保护的数学环境
    mathProtections.forEach((content, index) => {
      const placeholder = `__PROTECTED_MATH_${index}__`;
      processedText = processedText.replace(placeholder, () => content);
    });
    
    return processedText;
  }

  /**
   * 查找命令在文本中的位置
   * @private
   */
  findPosition(text, match) {
    const index = text.indexOf(match);
    if (index === -1) return { line: 1, column: 1 };

    const beforeMatch = text.substring(0, index);
    const lines = beforeMatch.split('\n');
    
    return {
      line: lines.length,
      column: lines[lines.length - 1].length + 1,
      start: index,
      end: index + match.length
    };
  }

  /**
   * TDD: 使用统一的数学内容检测器处理数学环境
   * @param {string} text - 输入文本
   * @param {Array} blocks - 输出块数组
   * @returns {string} 处理后的文本
   */
  processMathEnvironments(text, blocks) {
    let processedText = text;
    const placeholders = []; // 存储所有占位符
    
    // 首先保护 LaTeX 代码块，防止其内容被当作数学环境处理
    let codeBlockIndex = 0;
    processedText = processedText.replace(/```(?:latex|LaTeX|tex|TeX)\s*\n([\s\S]*?)\n```/g, (match, content) => {
      const placeholder = `__L1_LATEX_CODEBLOCK_${codeBlockIndex++}__`;
      
      // 检查内容是否已经有数学分隔符
      const trimmedContent = content.trim();
      const hasWrappingDelimiters = 
        (trimmedContent.startsWith('\\[') && trimmedContent.endsWith('\\]')) ||
        (trimmedContent.startsWith('$$') && trimmedContent.endsWith('$$'));
      
      let processedContent;
      
      if (hasWrappingDelimiters) {
        // 如果整体已有分隔符，保持原样
        processedContent = trimmedContent;
      } else {
        // 整个代码块作为一个数学公式处理，不要分行
        processedContent = `$$\n${trimmedContent}\n$$`;
      }
      
      placeholders.push({
        id: placeholder,
        content: processedContent,
        type: 'latex_codeblock',
        layer: 1
      });
      return placeholder;
    });
    
    // 首先处理独立的 $$ 块，使用占位符保护它们
    let displayMathIndex = 0;
    
    // 匹配所有的 $$ 块（包括独立的和包装数学环境的）
    processedText = processedText.replace(/\$\$([\s\S]*?)\$\$/g, (match, content) => {
      // 所有 $$ 块都需要保护，不管是否包含数学环境
      const placeholder = `__L1_DOC_MATH_${displayMathIndex}__`;
      placeholders.push({
        id: placeholder,
        content: match,
        type: 'display_math',
        layer: 1
      });
      
      // 检查是否是数学环境
      const isEnvironment = content.trim().startsWith('\\begin{') && content.trim().includes('\\end{');
      
      blocks.push({
        type: isEnvironment ? 'display_math_environment' : 'display_math',
        content: match,
        originalCommand: match,
        katexReady: true,
        protected: true,
        hasEnvironment: isEnvironment,
        index: displayMathIndex
      });
      
      console.log(`[DocumentParser] 保护$$块 #${displayMathIndex} (${isEnvironment ? '包含环境' : '纯公式'}): ${match.substring(0, 50)}...`);
      displayMathIndex++;
      
      return placeholder;
    });
    
    // 使用MathContentDetector提取所有数学块（\begin{equation}等环境）
    const mathBlocks = this.mathDetector.extractMathBlocks(processedText);
    
    // 去重：如果有多个块在相同位置，只保留第一个
    const uniqueBlocks = [];
    const seenPositions = new Set();
    
    for (const block of mathBlocks) {
      const posKey = `${block.startIndex}-${block.endIndex}`;
      if (!seenPositions.has(posKey)) {
        seenPositions.add(posKey);
        uniqueBlocks.push(block);
      } else {
        console.log(`[DocumentParser] TDD: 跳过重复块 - 环境:${block.environment}, 位置:${posKey}`);
      }
    }
    
    // 从后往前处理，避免索引变化
    for (let i = uniqueBlocks.length - 1; i >= 0; i--) {
      const mathBlock = uniqueBlocks[i];
      
      // 只处理display_environment类型，跳过inline_math
      if (mathBlock.type !== 'display_environment') {
        continue;
      }
      
      if (mathBlock.type === 'display_environment') {
        console.log(`[DocumentParser] TDD: 处理${mathBlock.environment}环境 -> 创建占位符`);
        
        // 创建占位符
        const placeholder = `__L1_DOC_ENV_${placeholders.length}__`;
        
        // 添加到占位符数组
        placeholders.push({
          id: placeholder,
          content: `$$${mathBlock.content}$$`, // 包装后的内容
          type: 'math_environment',
          environment: mathBlock.environment,
          layer: 1
        });
        
        // 添加到块列表（用于TDD测试验证）
        blocks.push({
          type: `${mathBlock.environment}_environment`,
          content: mathBlock.content,
          originalCommand: mathBlock.content,
          katexWrapped: true,
          environment: mathBlock.environment,
          placeholder: placeholder
        });
        
        // 替换为占位符
        const beforeContent = processedText.substring(0, mathBlock.startIndex);
        const afterContent = processedText.substring(mathBlock.endIndex);
        processedText = beforeContent + placeholder + afterContent;
      }
    }
    
    // 不再恢复占位符，返回处理结果和占位符数组
    return {
      text: processedText,
      placeholders: placeholders
    };
  }

  /**
   * 处理多行环境块（如cases, align等）
   * @param {Array} processedLines - 已处理的行数组
   * @returns {string} 合并后的内容
   */
  processEnvironmentBlocks(processedLines) {
    const result = [];
    let i = 0;
    
    while (i < processedLines.length) {
      const line = processedLines[i];
      
      // 检查是否是环境的开始
      const beginMatch = line.match(/^\$\$\\begin\{([^}]+)\}/);
      if (beginMatch) {
        const envName = beginMatch[1];
        let envContent = [line.replace(/^\$\$/, '').replace(/\$\$$/, '')]; // 去掉$$包装
        i++;
        
        // 收集环境内容直到找到对应的\end
        let foundEnd = false;
        while (i < processedLines.length && !foundEnd) {
          const currentLine = processedLines[i];
          envContent.push(currentLine.replace(/^\$\$/, '').replace(/\$\$$/, ''));
          
          if (currentLine.includes(`\\end{${envName}}`)) {
            foundEnd = true;
          }
          i++;
        }
        
        // 如果找到了完整的环境，合并为一个$$块
        if (foundEnd) {
          result.push(`$$${envContent.join('\n')}$$`);
        } else {
          // 如果没有找到结束标记，恢复原来的处理方式
          result.push(...envContent.map(content => content.trim() ? `$$${content}$$` : content));
        }
      } else {
        // 普通行，直接添加
        result.push(line);
        i++;
      }
    }
    
    return result.join('\n');
  }

  /**
   * TDD: 添加必需的接口方法供测试使用
   * 模拟DocumentParser单独处理并返回mathBlocks
   */
  processForTesting(text) {
    const mathBlocks = this.mathDetector.extractMathBlocks(text);
    return {
      mathBlocks: mathBlocks.map(block => ({
        type: block.type === 'display_environment' ? 'math_environment' : block.type,
        environment: block.environment || null,
        content: block.content
      }))
    };
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = DocumentParser;
  console.log('[DocumentParser] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境（包括测试环境中的模拟浏览器）
  window.DocumentParser = DocumentParser;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.DocumentParser = DocumentParser;
  }
}
