/**
 * BlockElementParser - 块级元素解析器
 * 负责处理LaTeX块级结构命令
 * 包括：列表、表格、浮动环境、定理环境等
 * 
 * TDD改进：添加$$数学公式保护机制
 */

// 新架构：不再需要DollarDollarProtector
// 占位符由LaTeXParser统一管理

class BlockElementParser {
  constructor(config = {}) {
    this.config = {
      // 列表环境映射
      listEnvironments: {
        'itemize': { type: 'ul', ordered: false },
        'enumerate': { type: 'ol', ordered: true },
        'description': { type: 'dl', ordered: false }
      },
      // 表格环境映射
      tableEnvironments: {
        'tabular': { type: 'table', floating: false },
        'table': { type: 'figure', floating: true, figureType: 'table' }
      },
      // 浮动环境映射
      floatEnvironments: {
        'figure': { type: 'figure', figureType: 'figure' },
        'table': { type: 'figure', figureType: 'table' }
      },
      // 对齐环境映射
      alignmentEnvironments: {
        'center': { type: 'div', className: 'center', alignment: 'center' },
        'flushleft': { type: 'div', className: 'flushleft', alignment: 'left' },
        'flushright': { type: 'div', className: 'flushright', alignment: 'right' }
      },
      // 引用环境映射
      quoteEnvironments: {
        'quote': { type: 'blockquote', className: 'quote', quoteType: 'short' },
        'quotation': { type: 'blockquote', className: 'quotation', quoteType: 'long' }
      },
      // 定理类环境映射
      theoremEnvironments: {
        'theorem': { type: 'div', className: 'latex-theorem', displayName: '定理' },
        'lemma': { type: 'div', className: 'latex-lemma', displayName: 'Lemma' },
        'proposition': { type: 'div', className: 'latex-proposition', displayName: '命题' },
        'corollary': { type: 'div', className: 'latex-corollary', displayName: '推论' },
        'definition': { type: 'div', className: 'latex-definition', displayName: '定义' },
        'example': { type: 'div', className: 'latex-example', displayName: '例' },
        'remark': { type: 'div', className: 'latex-remark', displayName: '注' },
        'proof': { type: 'div', className: 'latex-proof', displayName: '证明' }
      },
      ...config
    };
    // 新架构：不需要初始化DollarDollarProtector
  }

  /**
   * 处理块级元素解析
   * @param {string} text - 输入文本
   * @param {Array} existingPlaceholders - 从其他层传递的占位符
   * @returns {object} 处理结果
   */
  process(text, existingPlaceholders = []) {
    const blocks = [];
    const warnings = [];
    const errors = [];
    const cssRules = new Set();
    const placeholders = [...existingPlaceholders]; // 保留已有占位符
    
    // 先处理代码块（```latex格式）
    let processedText = this.processCodeBlocks(text, blocks, warnings, placeholders);
    
    // 不再使用DollarDollarProtector，因为$$已经在其他层被保护了
    // 检查并跳过其他层的占位符

    try {
      // 处理文档类声明 (必须最先处理)
      const docClassResult = this.processDocumentClass(processedText);
      if (docClassResult.success) {
        processedText = docClassResult.processedText;
        if (docClassResult.documentClass) {
          blocks.push({
            type: 'document_class',
            documentClass: docClassResult.documentClass
          });
        }
        warnings.push(...docClassResult.warnings);
      } else {
        warnings.push(...docClassResult.errors);
      }
      
      // 处理对齐环境
      processedText = this.processAlignmentEnvironments(processedText, blocks, warnings, cssRules);

      // 处理引用块环境
      processedText = this.processQuoteEnvironments(processedText, blocks, cssRules);

      // 处理定理类环境
      processedText = this.processTheoremEnvironments(processedText, blocks, warnings, cssRules);

      // 处理列表环境
      processedText = this.processListEnvironments(processedText, blocks, warnings);

      // 处理表格环境
      processedText = this.processTableEnvironments(processedText, blocks, warnings);

      // 处理浮动环境
      processedText = this.processFloatEnvironments(processedText, blocks, warnings);

      // 处理代码环境
      processedText = this.processVerbatimEnvironments(processedText, blocks);

      // 检查未匹配的环境和错误匹配的环境
      this.checkUnmatchedEnvironments(processedText, warnings);
      this.checkMismatchedEnvironments(processedText, warnings);
      this.checkUnknownEnvironments(processedText, warnings);

    } catch (error) {
      errors.push({
        type: 'block_parsing_error',
        message: `块级解析错误: ${error.message}`,
        position: { line: 1, column: 1 }
      });
    }

    // 不恢复占位符，保持它们直到最后

    return {
      text: processedText,
      blocks,
      warnings,
      errors,
      css: this.generateCSS(cssRules),
      placeholders // 返回所有占位符
    };
  }

  /**
   * 处理对齐环境
   * @private
   */
  processAlignmentEnvironments(text, blocks, warnings, cssRules) {
    let processedText = text;

    for (const [envName, config] of Object.entries(this.config.alignmentEnvironments)) {
      processedText = this.processEnvironment(
        processedText, envName, config, blocks, warnings, cssRules
      );
    }

    return processedText;
  }

  /**
   * 通用环境处理方法
   * @private
   */
  processEnvironment(text, envName, config, blocks, warnings, cssRules) {
    let processedText = text;
    const beginPattern = `\\\\begin\\{${envName}\\}`;
    const endPattern = `\\\\end\\{${envName}\\}`;
    const regex = new RegExp(`${beginPattern}([\\s\\S]*?)${endPattern}`, 'g');

    processedText = processedText.replace(regex, (match, content) => {
      blocks.push({
        type: envName,
        content: content.trim(),
        config: config,
        originalCommand: match
      });

      // 添加CSS规则
      cssRules.add(this.getEnvironmentCSS(config));

      // 生成HTML
      const className = config.className ? ` class="${config.className}"` : '';
      return `<${config.type}${className}>${content.trim()}</${config.type}>`;
    });

    // 检查不匹配的环境
    this.checkMismatchedEnvironment(processedText, envName, warnings);

    return processedText;
  }

  /**
   * 检查不匹配的环境
   * @private
   */
  checkMismatchedEnvironment(text, envName, warnings) {
    const beginPattern = `\\\\begin\\{${envName}\\}`;
    const endPattern = `\\\\end\\{${envName}\\}`;
    
    const beginMatches = (text.match(new RegExp(beginPattern, 'g')) || []).length;
    const endMatches = (text.match(new RegExp(endPattern, 'g')) || []).length;
    
    if (beginMatches > endMatches) {
      warnings.push({
        type: 'unclosed_environment',
        environment: envName,
        message: `环境未闭合: ${envName}`,
        count: beginMatches - endMatches
      });
    } else if (endMatches > beginMatches) {
      warnings.push({
        type: 'extra_end_environment',
        environment: envName,
        message: `多余的环境结束标签: ${envName}`,
        count: endMatches - beginMatches
      });
    }
  }

  /**
   * 获取环境CSS样式
   * @private
   */
  getEnvironmentCSS(config) {
    const className = config.className;
    
    if (config.alignment) {
      return `
        .${className} {
          text-align: ${config.alignment};
          margin: 1em 0;
        }
      `;
    }
    
    if (config.quoteType) {
      if (config.quoteType === 'short') {
        return `
          .${className} {
            margin: 1em 2em;
            padding: 0.5em 1em;
            border-left: 4px solid #ddd;
            background-color: #f9f9f9;
            font-style: italic;
          }
        `;
      } else {
        return `
          .${className} {
            margin: 1.5em 2em;
            padding: 1em;
            border-left: 4px solid #ccc;
            background-color: #f5f5f5;
            text-indent: 1em;
          }
        `;
      }
    }
    
    return '';
  }

  /**
   * 生成CSS样式
   * @private
   */
  generateCSS(cssRules) {
    if (cssRules.size === 0) return undefined;
    
    return Array.from(cssRules)
      .filter(rule => rule.trim())
      .join('\\n');
  }

  /**
   * 处理列表环境
   * @private
   */
  processListEnvironments(text, blocks, warnings) {
    let processedText = text;

    for (const [envName, config] of Object.entries(this.config.listEnvironments)) {
      const regex = new RegExp(`\\\\begin\\{${envName}\\}([\\s\\S]*?)\\\\end\\{${envName}\\}`, 'g');
      
      processedText = processedText.replace(regex, (match, content) => {
        const items = this.parseListItems(content, 0);
        
        if (items.length === 0) {
          warnings.push({
            type: 'empty_list',
            environment: envName,
            message: `空的${envName}环境`
          });
          return '';
        }

        blocks.push({
          type: envName,
          itemCount: items.length,
          items: items,
          htmlTag: config.type,
          originalCommand: match,
          position: this.findPosition(text, match)
        });

        return this.renderList(items, config.type, 0);
      });
    }

    return processedText;
  }

  /**
   * 解析列表项
   * @private
   */
  parseListItems(content, depth = 0) {
    const items = [];
    const itemRegex = /\\item\s+((?:[^\\]|\\(?!item))*)/g;
    let match;

    while ((match = itemRegex.exec(content)) !== null) {
      let itemContent = match[1].trim();
      
      // 递归处理嵌套列表
      const nestedListTypes = ['itemize', 'enumerate', 'description'];
      for (const listType of nestedListTypes) {
        const nestedRegex = new RegExp(`\\\\begin\\{${listType}\\}([\\s\\S]*?)\\\\end\\{${listType}\\}`, 'g');
        itemContent = itemContent.replace(nestedRegex, (nestedMatch, nestedContent) => {
          const nestedItems = this.parseListItems(nestedContent, depth + 1);
          const nestedListType = listType === 'enumerate' ? 'ol' : 
                                listType === 'description' ? 'dl' : 'ul';
          return this.renderList(nestedItems, nestedListType, depth + 1);
        });
      }
      
      if (itemContent) {
        items.push({
          content: itemContent,
          raw: match[0]
        });
      }
    }

    return items;
  }

  /**
   * 渲染列表HTML
   * @private
   */
  renderList(items, listType, depth = 0) {
    const depthClass = depth > 0 ? ` latex-nested-${depth}` : '';
    const listItems = items.map(item => `<li>${item.content}</li>`).join('');
    return `<${listType} class="latex-list${depthClass}">${listItems}</${listType}>`;
  }

  /**
   * 处理表格环境（完整实现）
   * @private
   */
  processTableEnvironments(text, blocks, warnings) {
    let processedText = text;

    // 处理tabular环境
    processedText = this.processTabularEnvironment(processedText, blocks, warnings, text);

    // 处理table浮动环境
    processedText = this.processTableFloatEnvironment(processedText, blocks, warnings, text);

    return processedText;
  }

  /**
   * 处理tabular环境
   * @private
   */
  processTabularEnvironment(text, blocks, warnings, originalText) {
    const tabularRegex = /\\begin\{tabular\}(?:\[[^\]]*\])?\{([^}]+)\}([\s\S]*?)\\end\{tabular\}/g;
    
    return text.replace(tabularRegex, (match, colspec, content) => {
      try {
        // 解析列规范
        const columnInfo = this.parseColumnSpec(colspec);
        if (!columnInfo.isValid) {
          warnings.push({
            type: 'invalid_column_spec',
            columnSpec: colspec,
            message: `无效的列规范: ${colspec}`
          });
        }

        // 解析表格内容
        const tableData = this.parseTableContent(content, columnInfo.columns);
        
        // 检查列数匹配
        this.validateColumnCount(tableData.rows, columnInfo.columns, warnings);

        // 创建表格块信息
        blocks.push({
          type: 'tabular',
          columnSpec: colspec,
          columnAlignment: columnInfo.alignment,
          columns: columnInfo.columns,
          rows: tableData.rows,
          hasHeader: tableData.hasHeader,
          hasBorders: columnInfo.hasBorders,
          originalCommand: match,
          position: this.findPosition(originalText, match)
        });

        // 生成HTML表格
        return this.renderTable(tableData, columnInfo);

      } catch (error) {
        warnings.push({
          type: 'table_parsing_error',
          message: `表格解析错误: ${error.message}`,
          originalCommand: match
        });
        return match; // 返回原始内容
      }
    });
  }

  /**
   * 处理table浮动环境
   * @private
   */
  processTableFloatEnvironment(text, blocks, warnings, originalText) {
    const tableFloatRegex = /\\begin\{table\}(?:\[([^\]]*)\])?([\s\S]*?)\\end\{table\}/g;
    
    return text.replace(tableFloatRegex, (match, position, content) => {
      try {
        // 查找标题和标签
        const captionMatch = content.match(/\\caption\{([^}]+)\}/);
        const labelMatch = content.match(/\\label\{([^}]+)\}/);

        // 移除caption和label，保留其他内容
        let innerContent = content
          .replace(/\\caption\{[^}]+\}/g, '')
          .replace(/\\label\{[^}]+\}/g, '')
          .replace(/\\centering\s*/g, '')
          .trim();

        blocks.push({
          type: 'table',
          figureType: 'table',
          caption: captionMatch ? captionMatch[1] : null,
          label: labelMatch ? labelMatch[1] : null,
          position: position || null,
          content: innerContent,
          originalCommand: match,
          location: this.findPosition(originalText, match)  // 避免命名冲突
        });

        // 生成浮动表格HTML
        const caption = captionMatch ? `<figcaption>${captionMatch[1]}</figcaption>` : '';
        const id = labelMatch ? ` id="${labelMatch[1]}"` : '';
        
        return `<figure class="latex-table"${id}>
          ${innerContent}
          ${caption}
        </figure>`;

      } catch (error) {
        warnings.push({
          type: 'table_float_error',
          message: `浮动表格解析错误: ${error.message}`,
          originalCommand: match
        });
        return match;
      }
    });
  }

  /**
   * 解析列规范
   * @private
   */
  parseColumnSpec(colspec) {
    const alignment = [];
    let columns = 0;
    let hasBorders = false;
    let isValid = true;

    try {
      for (let i = 0; i < colspec.length; i++) {
        const char = colspec[i];
        switch (char) {
          case 'l':
            alignment.push('left');
            columns++;
            break;
          case 'c':
            alignment.push('center');
            columns++;
            break;
          case 'r':
            alignment.push('right');
            columns++;
            break;
          case '|':
            hasBorders = true;
            break;
          case 'p':
            // 处理 p{width} 格式
            if (i + 1 < colspec.length && colspec[i + 1] === '{') {
              const endBrace = colspec.indexOf('}', i + 2);
              if (endBrace !== -1) {
                alignment.push('left'); // p列默认左对齐
                columns++;
                i = endBrace; // 跳过整个p{width}
              } else {
                isValid = false;
              }
            } else {
              isValid = false;
            }
            break;
          case ' ':
            // 忽略空格
            break;
          default:
            // 未知字符
            isValid = false;
            break;
        }
      }
    } catch (error) {
      isValid = false;
    }

    return {
      alignment,
      columns,
      hasBorders,
      isValid
    };
  }

  /**
   * 解析表格内容
   * @private
   */
  parseTableContent(content, expectedColumns) {
    const rows = [];
    let hasHeader = false;

    // 预处理：移除所有\\hline并检测表头
    let cleanContent = content;
    const hlinePattern = /\\hline\s*/g;
    const hlineMatches = content.match(hlinePattern) || [];
    
    // 如果开头就有hline，说明有表头
    if (content.trim().startsWith('\\hline')) {
      hasHeader = true;
    }
    
    // 移除所有hline
    cleanContent = cleanContent.replace(hlinePattern, '');

    // 分割行，处理\\
    const lines = cleanContent.split(/\\\\/);
    let isFirstContentRow = true;

    for (const line of lines) {
      const trimmedLine = line.trim();
      if (!trimmedLine) continue;

      // 分割列，处理&，并处理特殊字符
      const cells = trimmedLine.split('&').map(cell => {
        let processedCell = cell.trim();
        // 处理特殊字符转义
        processedCell = this.processSpecialCharsInCell(processedCell);
        return processedCell;
      });
      
      if (cells.length > 0 && !(cells.length === 1 && cells[0] === '')) {
        rows.push({
          cells: cells,
          isHeader: hasHeader && isFirstContentRow
        });
        isFirstContentRow = false;
      }
    }

    return {
      rows,
      hasHeader: rows.some(row => row.isHeader)
    };
  }

  /**
   * 处理单元格中的特殊字符
   * @private
   */
  processSpecialCharsInCell(cellContent) {
    const specialChars = {
      '\\&': '&amp;',
      '\\%': '%',
      '\\$': '$',
      '\\{': '{',
      '\\}': '}',
      '\\_': '_',
      '\\#': '#'
    };

    let processed = cellContent;
    for (const [latexChar, htmlChar] of Object.entries(specialChars)) {
      const regex = new RegExp(this.escapeRegExp(latexChar), 'g');
      processed = processed.replace(regex, htmlChar);
    }

    return processed;
  }

  /**
   * 转义正则表达式特殊字符
   * @private
   */
  escapeRegExp(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
  }

  /**
   * 验证列数
   * @private
   */
  validateColumnCount(rows, expectedColumns, warnings) {
    for (const row of rows) {
      if (row.cells.length !== expectedColumns) {
        warnings.push({
          type: 'column_mismatch',
          expected: expectedColumns,
          actual: row.cells.length,
          message: `行的列数 (${row.cells.length}) 与列规范不匹配 (${expectedColumns})`
        });
      }
    }
  }

  /**
   * 渲染表格HTML
   * @private
   */
  renderTable(tableData, columnInfo) {
    let html = '<table class="latex-table';
    if (columnInfo.hasBorders) {
      html += ' bordered-table';
    }
    html += '">';

    // 渲染表头
    const headerRows = tableData.rows.filter(row => row.isHeader);
    if (headerRows.length > 0) {
      html += '<thead>';
      for (const row of headerRows) {
        html += '<tr>';
        for (let i = 0; i < row.cells.length; i++) {
          const cell = row.cells[i];
          const alignment = columnInfo.alignment[i] || 'left';
          html += `<th class="text-align-${alignment}">${cell || ''}</th>`;
        }
        html += '</tr>';
      }
      html += '</thead>';
    }

    // 渲染表体
    const bodyRows = tableData.rows.filter(row => !row.isHeader);
    if (bodyRows.length > 0) {
      html += '<tbody>';
      for (const row of bodyRows) {
        html += '<tr>';
        for (let i = 0; i < row.cells.length; i++) {
          const cell = row.cells[i];
          const alignment = columnInfo.alignment[i] || 'left';
          html += `<td class="text-align-${alignment}">${cell || ''}</td>`;
        }
        html += '</tr>';
      }
      html += '</tbody>';
    }

    html += '</table>';
    return html;
  }

  /**
   * 处理浮动环境
   * @private
   */
  processFloatEnvironments(text, blocks, warnings) {
    let processedText = text;

    for (const [envName, config] of Object.entries(this.config.floatEnvironments)) {
      const regex = new RegExp(`\\\\begin\\{${envName}\\}(?:\\[[^\\]]*\\])?([\\s\\S]*?)\\\\end\\{${envName}\\}`, 'g');
      
      processedText = processedText.replace(regex, (match, content) => {
        // 查找标题
        const captionMatch = content.match(/\\caption\{([^}]+)\}/);
        const labelMatch = content.match(/\\label\{([^}]+)\}/);

        blocks.push({
          type: envName,
          figureType: config.figureType,
          caption: captionMatch ? captionMatch[1] : null,
          label: labelMatch ? labelMatch[1] : null,
          content: content.trim(),
          originalCommand: match,
          position: this.findPosition(text, match)
        });

        // 基础浮动环境渲染
        const caption = captionMatch ? `<figcaption>${captionMatch[1]}</figcaption>` : '';
        const id = labelMatch ? ` id="${labelMatch[1]}"` : '';
        
        return `<figure class="latex-${envName}"${id}>
          ${content.replace(/\\caption\{[^}]+\}/g, '').replace(/\\label\{[^}]+\}/g, '').trim()}
          ${caption}
        </figure>`;
      });
    }

    return processedText;
  }

  /**
   * 处理引用块环境
   * @private
   */
  processQuoteEnvironments(text, blocks, cssRules) {
    let processedText = text;

    for (const [envName, config] of Object.entries(this.config.quoteEnvironments)) {
      processedText = this.processEnvironment(
        processedText, envName, config, blocks, [], cssRules
      );
    }

    return processedText;
  }

  /**
   * 处理定理类环境
   * @private
   */
  processTheoremEnvironments(text, blocks, warnings, cssRules) {
    let processedText = text;

    for (const [envName, config] of Object.entries(this.config.theoremEnvironments)) {
      // 匹配带可选参数的定理环境，如 \begin{theorem}[标题]
      const regex = new RegExp(`\\\\begin\\{${envName}\\}(?:\\[([^\\]]*)\\])?([\\s\\S]*?)\\\\end\\{${envName}\\}`, 'g');
      
      processedText = processedText.replace(regex, (match, title, content) => {
        const blockId = `theorem-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
        
        blocks.push({
          type: 'theorem_environment',
          environment: envName,
          title: title ? title.trim() : null,
          content: content.trim(),
          config: config,
          id: blockId
        });

        // 添加CSS规则
        cssRules.add(this.getTheoremCSS(config));

        // 创建占位符以保护theorem结构
        const placeholderId = `THEOREM_ENV_${blockId}`;
        const contentId = `THEOREM_CONTENT_${blockId}`;
        
        // 将整个theorem结构存储为占位符
        const theoremStructure = {
          id: placeholderId,
          type: 'theorem_structure',
          envName: envName,
          title: title,
          content: content.trim(),
          config: config,
          blockId: blockId
        };
        
        // 添加到占位符列表（如果有的话）
        if (!warnings.placeholders) {
          warnings.placeholders = [];
        }
        warnings.placeholders.push(theoremStructure);
        
        // 生成最终的HTML结构
        // 注意：内容部分暂时保留原始LaTeX，让后续的MathParser等处理
        let html = `<div class="${config.className}" id="${blockId}">`;
        
        // 添加环境标题
        if (envName === 'proof') {
          html += `<div class="theorem-header"><strong>${config.displayName}</strong></div>`;
        } else {
          html += `<div class="theorem-header"><strong>${config.displayName}`;
          if (title) {
            html += ` (${title})`;
          }
          html += `</strong></div>`;
        }
        
        // 内容部分保留原始LaTeX语法，让MathParser处理数学公式
        html += `<div class="theorem-content">${content.trim()}</div>`;
        
        // proof环境在末尾添加QED符号
        if (envName === 'proof') {
          html += `<div class="qed-symbol">□</div>`;
        }
        
        html += `</div>`;
        
        return html;
      });
    }

    // 检查不匹配的定理环境
    for (const envName of Object.keys(this.config.theoremEnvironments)) {
      this.checkMismatchedEnvironment(processedText, envName, warnings);
    }

    return processedText;
  }

  /**
   * 生成定理环境的CSS
   * @private
   */
  getTheoremCSS(config) {
    return `.${config.className} {
      margin: 1em 0;
      padding: 0.5em 1em;
      border-left: 3px solid #666;
      background-color: #f9f9f9;
    }
    .${config.className} .theorem-header {
      margin-bottom: 0.5em;
      font-weight: bold;
    }
    .${config.className} .theorem-content {
      line-height: 1.6;
    }
    .${config.className} .qed-symbol {
      text-align: right;
      margin-top: 0.5em;
    }`;
  }

  /**
   * 处理代码环境
   * @private
   */
  processVerbatimEnvironments(text, blocks) {
    let processedText = text;

    // 处理verbatim环境
    processedText = processedText.replace(/\\begin\{verbatim\}([\s\S]*?)\\end\{verbatim\}/g, (match, content) => {
      blocks.push({
        type: 'verbatim',
        content: content,
        originalCommand: match
      });

      return `<pre><code class="latex-verbatim">${this.escapeHtml(content)}</code></pre>`;
    });

    return processedText;
  }

  /**
   * 检查错误匹配的环境
   * @private
   */
  checkMismatchedEnvironments(text, warnings) {
    // 查找所有begin和end标签对
    const beginRegex = /\\begin\{([^}]+)\}/g;
    const endRegex = /\\end\{([^}]+)\}/g;
    
    const beginTags = [];
    const endTags = [];
    
    let match;
    while ((match = beginRegex.exec(text)) !== null) {
      beginTags.push({
        env: match[1],
        position: match.index
      });
    }
    
    while ((match = endRegex.exec(text)) !== null) {
      endTags.push({
        env: match[1],
        position: match.index
      });
    }
    
    // 检查是否有匹配错误的环境
    const stack = [];
    const allTags = [...beginTags, ...endTags].sort((a, b) => a.position - b.position);
    
    for (const tag of allTags) {
      if (beginTags.some(b => b === tag)) {
        // 开始标签
        stack.push(tag);
      } else {
        // 结束标签
        if (stack.length === 0) {
          warnings.push({
            type: 'orphan_end_environment',
            environment: tag.env,
            message: `孤立的环境结束标签: ${tag.env}`
          });
        } else {
          const lastBegin = stack.pop();
          if (lastBegin.env !== tag.env) {
            warnings.push({
              type: 'mismatched_environment',
              environment: lastBegin.env,
              expectedEnd: lastBegin.env,
              actualEnd: tag.env,
              message: `环境标签不匹配: 期望 \\end{${lastBegin.env}}，实际 \\end{${tag.env}}`
            });
          }
        }
      }
    }
    
    // 检查未闭合的环境
    for (const unclosedTag of stack) {
      warnings.push({
        type: 'mismatched_environment',
        environment: unclosedTag.env,
        message: `环境未闭合: ${unclosedTag.env}`
      });
    }
  }

  /**
   * 检查未知环境
   * @private
   */
  checkUnknownEnvironments(text, warnings) {
    const allKnownEnvironments = new Set([
      ...Object.keys(this.config.alignmentEnvironments),
      ...Object.keys(this.config.quoteEnvironments),
      ...Object.keys(this.config.listEnvironments),
      ...Object.keys(this.config.tableEnvironments),
      ...Object.keys(this.config.floatEnvironments),
      ...Object.keys(this.config.theoremEnvironments),
      // 添加其他已知环境
      'verbatim', 'lstlisting', 'equation', 'align', 'gather'
    ]);
    
    const beginRegex = /\\begin\{([^}]+)\}/g;
    let match;
    
    while ((match = beginRegex.exec(text)) !== null) {
      const envName = match[1];
      if (!allKnownEnvironments.has(envName)) {
        warnings.push({
          type: 'unknown_environment',
          environment: envName,
          message: `未知环境: ${envName}`
        });
      }
    }
  }

  /**
   * 检查未匹配的环境
   * @private
   */
  checkUnmatchedEnvironments(text, warnings) {
    // 查找所有\\begin命令
    const beginMatches = text.match(/\\begin\{(\w+)\}/g) || [];
    const endMatches = text.match(/\\end\{(\w+)\}/g) || [];

    const beginEnvs = beginMatches.map(match => match.match(/\\begin\{(\w+)\}/)[1]);
    const endEnvs = endMatches.map(match => match.match(/\\end\{(\w+)\}/)[1]);

    // 检查每个begin是否有对应的end
    beginEnvs.forEach(env => {
      const beginCount = beginEnvs.filter(e => e === env).length;
      const endCount = endEnvs.filter(e => e === env).length;

      if (beginCount > endCount) {
        warnings.push({
          type: 'unmatched_environment',
          environment: env,
          message: `环境 ${env} 未正确关闭`
        });
      }
    });
  }

  /**
   * HTML转义
   * @private
   */
  escapeHtml(text) {
    const htmlEscapes = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };

    return text.replace(/[&<>"']/g, match => htmlEscapes[match]);
  }

  /**
   * 查找命令在文本中的位置
   * @private
   */
  findPosition(text, match) {
    const index = text.indexOf(match);
    if (index === -1) return { line: 1, column: 1 };

    const beforeMatch = text.substring(0, index);
    const lines = beforeMatch.split('\n');
    
    return {
      line: lines.length,
      column: lines[lines.length - 1].length + 1,
      start: index,
      end: index + match.length
    };
  }

  // ========== 数学公式处理方法 ==========

  /**
   * 处理行内数学公式 $...$
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processInlineMath(text) {
    const mathBlocks = [];
    const errors = [];
    let processedText = text;

    try {
      // 检查未闭合的行内公式
      const unclosedDollars = (text.match(/\$/g) || []).length;
      if (unclosedDollars % 2 !== 0) {
        errors.push('未闭合的行内数学公式：$符号数量不匹配');
        return { success: false, processedText: text, mathBlocks: [], errors };
      }

      // 匹配 $...$ 格式的行内公式
      const inlineMathRegex = /\$([^$\n]+?)\$/g;
      let match;

      while ((match = inlineMathRegex.exec(text)) !== null) {
        const formula = match[1].trim();
        const mathId = `inline-math-${mathBlocks.length}`;
        
        mathBlocks.push({
          type: 'inline-math',
          formula: formula,
          id: mathId,
          position: this.findPosition(text, match[0])
        });

        // 替换为HTML
        processedText = processedText.replace(match[0], 
          `<span class="latex-inline-math" id="${mathId}">${this.escapeHtml(formula)}</span>`);
      }

      return {
        success: true,
        processedText,
        mathBlocks,
        errors
      };
    } catch (error) {
      errors.push(`行内数学公式处理错误: ${error.message}`);
      return { success: false, processedText: text, mathBlocks: [], errors };
    }
  }

  /**
   * 处理块级数学公式 $$...$$
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processBlockMath(text) {
    const mathBlocks = [];
    const errors = [];
    let processedText = text;

    try {
      // 检查未闭合的块级公式
      const doubleDollars = (text.match(/\$\$/g) || []).length;
      if (doubleDollars % 2 !== 0) {
        errors.push('未闭合的块级数学公式：$$符号数量不匹配');
        return { success: false, processedText: text, mathBlocks: [], errors };
      }

      // 匹配 $$...$$ 格式的块级公式
      const blockMathRegex = /\$\$([\s\S]*?)\$\$/g;
      let match;

      while ((match = blockMathRegex.exec(text)) !== null) {
        const formula = match[1].trim();
        const mathId = `display-math-${mathBlocks.length}`;
        
        mathBlocks.push({
          type: 'display-math',
          formula: formula,
          id: mathId,
          position: this.findPosition(text, match[0])
        });

        // 替换为HTML
        processedText = processedText.replace(match[0], 
          `<div class="latex-display-math" id="${mathId}">${this.escapeHtml(formula)}</div>`);
      }

      return {
        success: true,
        processedText,
        mathBlocks,
        errors
      };
    } catch (error) {
      errors.push(`块级数学公式处理错误: ${error.message}`);
      return { success: false, processedText: text, mathBlocks: [], errors };
    }
  }

  /**
   * 处理数学环境 (equation, align等)
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processMathEnvironments(text) {
    const mathBlocks = [];
    const errors = [];
    let processedText = text;

    try {
      // 检查环境不匹配
      const beginMatches = text.match(/\\begin\{(equation\*?|align\*?|gather\*?|multline\*?)\}/g) || [];
      const endMatches = text.match(/\\end\{(equation\*?|align\*?|gather\*?|multline\*?)\}/g) || [];
      
      if (beginMatches.length !== endMatches.length) {
        errors.push('数学环境不匹配：\\begin和\\end数量不一致');
        return { success: false, processedText: text, mathBlocks: [], errors };
      }

      // 检查特定的环境不匹配情况
      const envMismatch = /\\begin\{(equation\*?|align\*?|gather\*?|multline\*?)\}[\s\S]*?\\end\{(?!\1\})/;
      if (envMismatch.test(text)) {
        errors.push('数学环境不匹配：\\begin{env} 与 \\end{env} 环境名不一致');
        return { success: false, processedText: text, mathBlocks: [], errors };
      }

      // 匹配数学环境
      const mathEnvRegex = /\\begin\{(equation\*?|align\*?|gather\*?|multline\*?)\}([\s\S]*?)\\end\{\1\}/g;
      let match;

      while ((match = mathEnvRegex.exec(text)) !== null) {
        const envType = match[1];
        const content = match[2].trim();
        const mathId = `math-env-${mathBlocks.length}`;
        
        // 检查是否有标签
        const labelMatch = content.match(/\\label\{([^}]+)\}/);
        const label = labelMatch ? labelMatch[1] : null;
        const formula = labelMatch ? content.replace(/\\label\{[^}]+\}/, '').trim() : content;

        mathBlocks.push({
          type: envType,
          formula: formula,
          label: label,
          id: mathId,
          numbered: !envType.endsWith('*'),
          position: this.findPosition(text, match[0])
        });

        // 生成HTML
        let htmlClass = `latex-${envType.replace('*', '')}`;
        let idAttr = label ? `id="${label}"` : `id="${mathId}"`;
        
        processedText = processedText.replace(match[0], 
          `<div class="${htmlClass}" ${idAttr}>${this.escapeHtml(formula)}</div>`);
      }

      return {
        success: true,
        processedText,
        mathBlocks,
        errors
      };
    } catch (error) {
      errors.push(`数学环境处理错误: ${error.message}`);
      return { success: false, processedText: text, mathBlocks: [], errors };
    }
  }

  /**
   * 处理所有数学公式
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processAllMath(text) {
    let processedText = text;
    const allMathBlocks = [];
    const allErrors = [];

    // 按正确优先级处理：先处理环境，再处理块级，最后处理行内
    // 这样确保高优先级的数学结构不会被低优先级的误识别
    const envResult = this.processMathEnvironments(processedText);
    processedText = envResult.processedText;
    allMathBlocks.push(...envResult.mathBlocks);
    allErrors.push(...envResult.errors);

    const blockResult = this.processBlockMath(processedText);
    processedText = blockResult.processedText;
    allMathBlocks.push(...blockResult.mathBlocks);
    allErrors.push(...blockResult.errors);

    const inlineResult = this.processInlineMath(processedText);
    processedText = inlineResult.processedText;
    allMathBlocks.push(...inlineResult.mathBlocks);
    allErrors.push(...inlineResult.errors);

    return {
      success: allErrors.length === 0,
      processedText,
      mathBlocks: allMathBlocks,
      errors: allErrors
    };
  }

  // ========== 图形和浮动环境处理方法 ==========

  /**
   * 处理figure环境
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processFigureEnvironments(text) {
    const figureBlocks = [];
    const errors = [];
    const warnings = [];
    let processedText = text;

    try {
      // 检查环境匹配
      const beginMatches = text.match(/\\begin\{(figure|wrapfigure|subfigure)\}/g) || [];
      const endMatches = text.match(/\\end\{(figure|wrapfigure|subfigure)\}/g) || [];
      
      if (beginMatches.length !== endMatches.length) {
        errors.push('图形环境不匹配：\\begin和\\end数量不一致');
        return { success: false, processedText: text, figureBlocks: [], errors, warnings };
      }

      // 匹配figure环境
      const figureRegex = /\\begin\{(figure|wrapfigure)\}(?:\[([^\]]*)\])?(?:\{([^}]*)\})?(?:\{([^}]*)\})?([\s\S]*?)\\end\{\1\}/g;
      let match;

      while ((match = figureRegex.exec(text)) !== null) {
        const envType = match[1];
        const positionParam = match[2] || '';
        const wrapPos = match[3] || '';  // wrapfigure位置参数
        const wrapWidth = match[4] || '';  // wrapfigure宽度参数
        const content = match[5].trim();
        
        const figureData = this.parseFigureContent(content, envType, positionParam, wrapPos, wrapWidth);
        figureData.id = `figure-${figureBlocks.length}`;
        figureData.textPosition = this.findPosition(text, match[0]);
        
        figureBlocks.push(figureData);
        
        // 生成HTML
        const FigureProcessor = require('../processors/FigureProcessor');
        const processor = new FigureProcessor();
        const figureHtml = processor.renderFigure(figureData);
        
        processedText = processedText.replace(match[0], figureHtml);
      }

      return {
        success: true,
        processedText,
        figureBlocks,
        errors,
        warnings
      };
    } catch (error) {
      errors.push(`图形环境处理错误: ${error.message}`);
      return { success: false, processedText: text, figureBlocks: [], errors, warnings };
    }
  }

  /**
   * 解析figure环境内容
   * @param {string} content - figure内容
   * @param {string} envType - 环境类型
   * @param {string} positionParam - 位置参数
   * @param {string} wrapPos - wrapfigure位置参数
   * @param {string} wrapWidth - wrapfigure宽度参数
   * @returns {object}
   */
  parseFigureContent(content, envType, positionParam, wrapPos = '', wrapWidth = '') {
    const figureData = {
      type: envType,
      position: envType === 'wrapfigure' ? wrapPos : positionParam,
      width: wrapWidth,
      images: [],
      caption: null,
      label: null,
      centered: false,
      subfigures: [],
      layout: 'normal'
    };

    // 检查是否居中
    if (/\\centering/.test(content)) {
      figureData.centered = true;
      content = content.replace(/\\centering\s*/, '');
    }

    // 提取标题
    const captionMatch = content.match(/\\caption\{([^}]+)\}/);
    if (captionMatch) {
      figureData.caption = captionMatch[1];
    }

    // 提取标签
    const labelMatch = content.match(/\\label\{([^}]+)\}/);
    if (labelMatch) {
      figureData.label = labelMatch[1];
    }

    // 解析子图环境
    const subfigureRegex = /\\begin\{subfigure\}\{([^}]+)\}([\s\S]*?)\\end\{subfigure\}/g;
    let subfigMatch;
    while ((subfigMatch = subfigureRegex.exec(content)) !== null) {
      const width = subfigMatch[1];
      const subfigContent = subfigMatch[2];
      
      const subfigure = this.parseSubfigure(subfigContent, width);
      figureData.subfigures.push(subfigure);
    }

    // 如果没有子图，解析普通图片
    if (figureData.subfigures.length === 0) {
      const images = this.parseIncludeGraphics(content);
      figureData.images = images;
      
      // 检测布局
      if (images.length > 1 && /\\hfill/.test(content)) {
        figureData.layout = 'side-by-side';
      }
    }

    return figureData;
  }

  /**
   * 解析子图
   * @param {string} content - 子图内容
   * @param {string} width - 子图宽度
   * @returns {object}
   */
  parseSubfigure(content, width) {
    const images = this.parseIncludeGraphics(content);
    const captionMatch = content.match(/\\caption\{([^}]+)\}/);
    
    return {
      width: width,
      image: images[0] || null,
      caption: captionMatch ? captionMatch[1] : null
    };
  }

  /**
   * 解析includegraphics命令
   * @param {string} text - 包含图片命令的文本
   * @returns {array} 图片数组
   */
  parseIncludeGraphics(text) {
    const images = [];
    const graphicsRegex = /\\includegraphics(?:\[([^\]]*)\])?\{([^}]+)\}/g;
    let match;

    while ((match = graphicsRegex.exec(text)) !== null) {
      const params = match[1] || '';
      const src = match[2];
      
      if (!src) continue;
      
      const imageData = {
        src: src,
        alt: src.split('/').pop().split('.')[0]
      };

      // 解析参数
      if (params) {
        const paramPairs = params.split(',');
        paramPairs.forEach(pair => {
          const [key, value] = pair.split('=').map(s => s.trim());
          if (key && value) {
            imageData[key] = value;
          }
        });
      }

      images.push(imageData);
    }

    return images;
  }

  /**
   * 处理行内图片
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processInlineGraphics(text) {
    const imageBlocks = [];
    const errors = [];
    const warnings = [];
    let processedText = text;

    try {
      const graphicsRegex = /\\includegraphics(?:\[([^\]]*)\])?\{([^}]*)\}/g;
      let match;

      while ((match = graphicsRegex.exec(text)) !== null) {
        const params = match[1] || '';
        const src = match[2];
        
        // 验证图片路径
        if (!src || !src.trim()) {
          errors.push('图形文件路径不能为空');
          // 仍然替换原文本，避免重复匹配
          processedText = processedText.replace(match[0], '');
          continue;
        }

        const imageData = {
          type: 'inline-image',
          src: src,
          id: `inline-image-${imageBlocks.length}`,
          position: this.findPosition(text, match[0])
        };

        // 解析参数
        if (params) {
          const paramResult = this.parseGraphicsParams(params);
          Object.assign(imageData, paramResult.params);
          warnings.push(...paramResult.warnings);
        }

        imageBlocks.push(imageData);

        // 生成HTML
        const FigureProcessor = require('../processors/FigureProcessor');
        const processor = new FigureProcessor();
        const imageHtml = processor.renderImage(imageData);
        
        processedText = processedText.replace(match[0], imageHtml);
      }

      return {
        success: errors.length === 0,
        processedText,
        imageBlocks,
        errors,
        warnings
      };
    } catch (error) {
      errors.push(`行内图形处理错误: ${error.message}`);
      return { success: false, processedText: text, imageBlocks: [], errors, warnings };
    }
  }

  /**
   * 解析图形参数
   * @param {string} params - 参数字符串
   * @returns {object}
   */
  parseGraphicsParams(params) {
    const result = { params: {}, warnings: [] };
    const validParams = ['width', 'height', 'scale', 'angle'];
    
    const paramPairs = params.split(',');
    paramPairs.forEach(pair => {
      const [key, value] = pair.split('=').map(s => s.trim());
      if (key && value) {
        if (validParams.includes(key)) {
          result.params[key] = value;
        } else {
          result.warnings.push(`无效的图形参数: ${key}`);
        }
      }
    });

    return result;
  }

  // ========== 引用系统处理方法 ==========

  /**
   * 处理标签定义收集
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processLabels(text) {
    const labels = {};
    const errors = [];
    const warnings = [];
    const counters = {
      section: 0,
      subsection: 0,
      subsubsection: 0,
      figure: 0,
      table: 0,
      equation: 0
    };

    try {
      // 收集章节标签
      this.collectSectionLabels(text, labels, counters, warnings);
      
      // 收集图形标签
      this.collectFigureLabels(text, labels, counters, warnings);
      
      // 收集表格标签
      this.collectTableLabels(text, labels, counters, warnings);
      
      // 收集数学公式标签
      this.collectEquationLabels(text, labels, counters, warnings);

      return {
        success: errors.length === 0,
        labels,
        counters,
        errors,
        warnings
      };
    } catch (error) {
      errors.push(`标签处理错误: ${error.message}`);
      return { success: false, labels: {}, counters, errors, warnings };
    }
  }

  /**
   * 收集章节标签
   * @private
   */
  collectSectionLabels(text, labels, counters, warnings) {
    const sectionRegex = /\\(section|subsection|subsubsection)\{([^}]+)\}[\s]*\\label\{([^}]*)\}/g;
    let match;

    while ((match = sectionRegex.exec(text)) !== null) {
      const level = match[1];
      const title = match[2];
      const labelId = match[3];

      if (!labelId.trim()) {
        warnings.push('空的标签定义');
        continue;
      }

      if (labels[labelId]) {
        warnings.push(`重复的标签定义: ${labelId}`);
        continue;
      }

      // 更新计数器
      if (level === 'section') {
        counters.section++;
        counters.subsection = 0;
        counters.subsubsection = 0;
      } else if (level === 'subsection') {
        counters.subsection++;
        counters.subsubsection = 0;
      } else if (level === 'subsubsection') {
        counters.subsubsection++;
      }

      // 生成编号
      let number;
      if (level === 'section') {
        number = counters.section.toString();
      } else if (level === 'subsection') {
        number = `${counters.section}.${counters.subsection}`;
      } else {
        number = `${counters.section}.${counters.subsection}.${counters.subsubsection}`;
      }

      labels[labelId] = {
        id: labelId,
        type: level,
        title: title,
        number: number,
        element: level === 'section' ? 'h2' : level === 'subsection' ? 'h3' : 'h4'
      };
    }
  }

  /**
   * 收集图形标签
   * @private
   */
  collectFigureLabels(text, labels, counters, warnings) {
    const figureRegex = /\\begin\{figure\}[\s\S]*?\\caption\{([^}]+)\}[\s\S]*?\\label\{([^}]+)\}[\s\S]*?\\end\{figure\}/g;
    let match;

    while ((match = figureRegex.exec(text)) !== null) {
      const caption = match[1];
      const labelId = match[2];

      if (!labelId.trim()) {
        warnings.push('空的标签定义');
        continue;
      }

      if (labels[labelId]) {
        warnings.push(`重复的标签定义: ${labelId}`);
        continue;
      }

      counters.figure++;

      labels[labelId] = {
        id: labelId,
        type: 'figure',
        title: caption,
        number: counters.figure.toString(),
        element: 'figure'
      };
    }
  }

  /**
   * 收集表格标签
   * @private
   */
  collectTableLabels(text, labels, counters, warnings) {
    const tableRegex = /\\begin\{table\}[\s\S]*?\\caption\{([^}]+)\}[\s\S]*?\\label\{([^}]+)\}[\s\S]*?\\end\{table\}/g;
    let match;

    while ((match = tableRegex.exec(text)) !== null) {
      const caption = match[1];
      const labelId = match[2];

      if (!labelId.trim()) {
        warnings.push('空的标签定义');
        continue;
      }

      if (labels[labelId]) {
        warnings.push(`重复的标签定义: ${labelId}`);
        continue;
      }

      counters.table++;

      labels[labelId] = {
        id: labelId,
        type: 'table',
        title: caption,
        number: counters.table.toString(),
        element: 'figure'
      };
    }
  }

  /**
   * 收集数学公式标签
   * @private
   */
  collectEquationLabels(text, labels, counters, warnings) {
    const equationRegex = /\\begin\{equation\}[\s]*\\label\{([^}]+)\}([\s\S]*?)\\end\{equation\}/g;
    let match;

    while ((match = equationRegex.exec(text)) !== null) {
      const labelId = match[1];
      const formula = match[2].trim();

      if (!labelId.trim()) {
        warnings.push('空的标签定义');
        continue;
      }

      if (labels[labelId]) {
        warnings.push(`重复的标签定义: ${labelId}`);
        continue;
      }

      counters.equation++;

      labels[labelId] = {
        id: labelId,
        type: 'equation',
        title: formula,
        number: counters.equation.toString(),
        element: 'div'
      };
    }
  }

  /**
   * 处理引用链接
   * @param {string} text - 输入文本
   * @param {object} labels - 标签映射
   * @returns {object} 处理结果
   */
  processReferences(text, labels) {
    const references = [];
    const errors = [];
    const warnings = [];
    let processedText = text;

    try {
      // 处理\ref{}引用
      processedText = this.processRefReferences(processedText, labels, references, errors, warnings);
      
      // 处理\pageref{}引用
      processedText = this.processPagerefReferences(processedText, labels, references, errors, warnings);

      return {
        success: errors.length === 0,
        processedText,
        references,
        errors,
        warnings
      };
    } catch (error) {
      errors.push(`引用处理错误: ${error.message}`);
      return { success: false, processedText: text, references: [], errors, warnings };
    }
  }

  /**
   * 处理\ref{}引用
   * @private
   */
  processRefReferences(text, labels, references, errors, warnings) {
    const refRegex = /\\ref\{([^}]+)\}/g;
    let processedText = text;
    let match;

    while ((match = refRegex.exec(text)) !== null) {
      const refId = match[1];
      
      if (!refId.trim()) {
        warnings.push('空的引用ID');
        continue;
      }

      const targetLabel = labels[refId];
      
      if (!targetLabel) {
        errors.push(`未定义的引用: ${refId}`);
        
        // 替换为错误标记
        const ReferenceProcessor = require('../processors/ReferenceProcessor');
        const processor = new ReferenceProcessor();
        const errorHtml = processor.renderReferenceError(refId);
        
        processedText = processedText.replace(match[0], errorHtml);
      } else {
        // 检查自引用
        if (this.isWithinLabel(text, match.index, refId)) {
          warnings.push(`自引用检测: ${refId}`);
        }

        const refData = {
          refId: refId,
          type: 'ref',
          targetType: targetLabel.type,
          targetNumber: targetLabel.number,
          targetTitle: targetLabel.title
        };
        
        references.push(refData);
        
        // 生成HTML
        const ReferenceProcessor = require('../processors/ReferenceProcessor');
        const processor = new ReferenceProcessor();
        const refHtml = processor.renderReference(refData);
        
        processedText = processedText.replace(match[0], refHtml);
      }
    }

    return processedText;
  }

  /**
   * 处理\pageref{}引用
   * @private
   */
  processPagerefReferences(text, labels, references, errors, warnings) {
    const pagerefRegex = /\\pageref\{([^}]+)\}/g;
    let processedText = text;
    let match;

    while ((match = pagerefRegex.exec(text)) !== null) {
      const refId = match[1];
      const targetLabel = labels[refId];
      
      if (!targetLabel) {
        errors.push(`未定义的引用: ${refId}`);
        
        // 替换为错误标记
        const ReferenceProcessor = require('../processors/ReferenceProcessor');
        const processor = new ReferenceProcessor();
        const errorHtml = processor.renderReferenceError(refId);
        
        processedText = processedText.replace(match[0], errorHtml);
      } else {
        const refData = {
          refId: refId,
          type: 'pageref',
          targetType: targetLabel.type,
          targetNumber: targetLabel.number
        };
        
        references.push(refData);
        
        // 生成HTML
        const ReferenceProcessor = require('../processors/ReferenceProcessor');
        const processor = new ReferenceProcessor();
        const refHtml = processor.renderPageReference(refData);
        
        processedText = processedText.replace(match[0], refHtml);
      }
    }

    return processedText;
  }

  /**
   * 处理脚注
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processFootnotes(text) {
    const footnotes = [];
    const warnings = [];
    const errors = [];
    let processedText = text;

    try {
      // 处理基本脚注 \footnote{content}
      const footnoteRegex = /\\footnote\{([^{}]*(?:\{[^{}]*\}[^{}]*)*)\}/g;
      let match;
      let footnoteNumber = 1;

      // 检测未闭合的脚注
      const unmatched = /\\footnote\{[^}]*$/g;
      if (unmatched.test(text)) {
        errors.push('未闭合的脚注');
        return {
          success: false,
          processedText: text,
          footnotes: [],
          warnings,
          errors
        };
      }

      // 检测嵌套脚注
      const nestedPattern = /\\footnote\{[^{}]*\\footnote\{[^}]*\}/g;
      if (nestedPattern.test(text)) {
        warnings.push('检测到嵌套脚注，只处理最外层脚注');
      }

      while ((match = footnoteRegex.exec(text)) !== null) {
        const content = match[1].trim();
        
        // 处理空脚注
        if (!content) {
          warnings.push(`空的脚注内容 (脚注 ${footnoteNumber})`);
          // 移除空脚注
          processedText = processedText.replace(match[0], '');
          continue;
        }

        // 创建脚注数据
        const footnoteData = {
          id: `footnote-${footnoteNumber}`,
          number: footnoteNumber.toString(),
          content: content,
          sourcePosition: {
            start: match.index,
            end: match.index + match[0].length,
            originalText: match[0]
          }
        };

        // 检查是否包含格式命令
        if (/\\[a-zA-Z]+\{/.test(content)) {
          footnoteData.hasFormatting = true;
        }

        // 检查是否包含引用
        if (/\\ref\{/.test(content)) {
          footnoteData.hasReferences = true;
        }

        // 检查是否为长脚注
        if (content.length > 200) {
          footnoteData.isLong = true;
        }

        footnotes.push(footnoteData);

        // 生成脚注链接HTML
        const FootnoteProcessor = require('../processors/FootnoteProcessor');
        const processor = new FootnoteProcessor();
        const linkHtml = processor.renderFootnoteLink(footnoteData);

        // 替换原脚注为链接
        processedText = processedText.replace(match[0], linkHtml);
        footnoteNumber++;
      }

      // 使用FootnoteProcessor为脚注分配正确编号
      const FootnoteProcessor = require('../processors/FootnoteProcessor');
      const processor = new FootnoteProcessor();
      const numberedFootnotes = processor.assignNumbers(footnotes);

      // 验证脚注
      const validation = processor.validateFootnotes(numberedFootnotes);
      warnings.push(...validation.warnings);
      errors.push(...validation.errors);

      return {
        success: errors.length === 0,
        processedText,
        footnotes: numberedFootnotes,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`脚注处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        footnotes: [],
        warnings,
        errors
      };
    }
  }

  /**
   * 处理文档元数据
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processMetadata(text) {
    const metadata = {};
    const warnings = [];
    const errors = [];
    let processedText = text;
    let hasMaketitle = false;

    try {
      // 处理标题命令
      const titleResult = this.processTitleCommand(processedText, warnings, errors);
      processedText = titleResult.processedText;
      if (titleResult.metadata) {
        metadata.title = titleResult.metadata;
      }

      // 处理作者命令
      const authorResult = this.processAuthorCommand(processedText, warnings, errors);
      processedText = authorResult.processedText;
      if (authorResult.metadata) {
        metadata.author = authorResult.metadata;
      }

      // 处理日期命令
      const dateResult = this.processDateCommand(processedText, warnings, errors);
      processedText = dateResult.processedText;
      if (dateResult.metadata) {
        metadata.date = dateResult.metadata;
      }

      // 处理maketitle命令
      const maketitleResult = this.processMaketitleCommand(processedText, metadata, warnings, errors);
      processedText = maketitleResult.processedText;
      hasMaketitle = maketitleResult.hasMaketitle;

      return {
        success: errors.length === 0,
        processedText,
        metadata,
        hasMaketitle,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`元数据处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        metadata: {},
        hasMaketitle: false,
        warnings,
        errors
      };
    }
  }

  /**
   * 处理标题命令
   * @private
   */
  processTitleCommand(text, warnings, errors) {
    const titleRegex = /\\title\{([^{}]*(?:\{[^{}]*\}[^{}]*)*)\}/g;
    let processedText = text;
    let metadata = null;
    let titleCount = 0;

    // 检测未闭合的标题
    const unmatched = /\\title\{[^}]*$/g;
    if (unmatched.test(text)) {
      errors.push('未闭合的title命令');
      return { processedText: text, metadata: null };
    }

    // 检测嵌套的标题
    const nestedPattern = /\\title\{[^{}]*\\title\{[^}]*\}/g;
    if (nestedPattern.test(text)) {
      warnings.push('检测到嵌套的title命令');
    }

    let match;
    while ((match = titleRegex.exec(text)) !== null) {
      titleCount++;
      const content = match[1].trim();
      
      // 检查重复定义
      if (titleCount > 1) {
        warnings.push('重复定义的title命令，使用最后一个定义');
      }

      // 处理空标题
      if (!content) {
        warnings.push('空的title命令');
      }

      // 创建元数据
      metadata = {
        content: content,
        command: match[0],
        position: {
          start: match.index,
          end: match.index + match[0].length,
          originalText: match[0]
        }
      };

      // 检查是否包含格式命令
      if (/\\[a-zA-Z]+\{/.test(content)) {
        metadata.hasFormatting = true;
      }

      // 移除title命令
      processedText = processedText.replace(match[0], '');
    }

    return { processedText, metadata };
  }

  /**
   * 处理作者命令
   * @private
   */
  processAuthorCommand(text, warnings, errors) {
    const authorRegex = /\\author\{([^{}]*(?:\{[^{}]*\}[^{}]*)*)\}/g;
    let processedText = text;
    let metadata = null;
    let authorCount = 0;

    // 检测未闭合的作者
    const unmatched = /\\author\{[^}]*$/g;
    if (unmatched.test(text)) {
      errors.push('未闭合的author命令');
      return { processedText: text, metadata: null };
    }

    let match;
    while ((match = authorRegex.exec(text)) !== null) {
      authorCount++;
      const content = match[1].trim();

      // 检查重复定义
      if (authorCount > 1) {
        warnings.push('重复定义的author命令，使用最后一个定义');
      }

      // 处理空作者
      if (!content) {
        warnings.push('空的author命令');
      }

      // 使用MetadataProcessor解析作者信息
      const MetadataProcessor = require('../processors/MetadataProcessor');
      const processor = new MetadataProcessor();
      const authorInfo = processor.parseAuthorContent(content);

      // 创建元数据
      metadata = {
        content: content,
        authors: authorInfo.authors,
        hasThanks: authorInfo.hasThanks,
        hasAffiliation: authorInfo.hasAffiliation,
        command: match[0],
        position: {
          start: match.index,
          end: match.index + match[0].length,
          originalText: match[0]
        }
      };

      // 移除author命令
      processedText = processedText.replace(match[0], '');
    }

    return { processedText, metadata };
  }

  /**
   * 处理日期命令
   * @private
   */
  processDateCommand(text, warnings, errors) {
    const dateRegex = /\\date\{([^{}]*)\}/g;
    let processedText = text;
    let metadata = null;
    let dateCount = 0;

    let match;
    while ((match = dateRegex.exec(text)) !== null) {
      dateCount++;
      let content = match[1].trim();

      // 检查重复定义
      if (dateCount > 1) {
        warnings.push('重复定义的date命令，使用最后一个定义');
      }

      // 处理\today特殊情况
      let isToday = false;
      if (content === '\\today') {
        const MetadataProcessor = require('../processors/MetadataProcessor');
        const processor = new MetadataProcessor();
        content = processor.formatTodayDate();
        isToday = true;
      }

      // 创建元数据
      metadata = {
        content: content,
        isEmpty: content === '',
        isToday: isToday,
        command: match[0],
        position: {
          start: match.index,
          end: match.index + match[0].length,
          originalText: match[0]
        }
      };

      // 移除date命令
      processedText = processedText.replace(match[0], '');
    }

    return { processedText, metadata };
  }

  /**
   * 处理maketitle命令
   * @private
   */
  processMaketitleCommand(text, metadata, warnings, errors) {
    const maketitleRegex = /\\maketitle/g;
    let processedText = text;
    let hasMaketitle = false;

    const match = maketitleRegex.exec(text);
    if (match) {
      hasMaketitle = true;

      // 检查是否有元数据
      if (!metadata.title && !metadata.author && !metadata.date) {
        warnings.push('使用maketitle命令但没有找到标题元数据');
      }

      // 生成标题页HTML
      const MetadataProcessor = require('../processors/MetadataProcessor');
      const processor = new MetadataProcessor();
      const titlePageHtml = processor.generateTitlePage(metadata);

      // 替换maketitle命令为标题页HTML
      processedText = processedText.replace(match[0], titlePageHtml);
    }

    return { processedText, hasMaketitle };
  }

  /**
   * 处理文档类声明
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processDocumentClass(text) {
    const warnings = [];
    const errors = [];
    let processedText = text;
    let documentClass = null;

    try {
      // 文档类正则：支持可选参数和必需参数
      const docClassRegex = /\\documentclass(?:\[([^\]]*)\])?\{([^}]+)\}/g;
      let classCount = 0;
      let match;

      // 检测未闭合的文档类 - 检测缺少右括号的情况
      const unmatchedBracket = /\\documentclass\[[^\]]*\{/g;
      const unmatchedBrace = /\\documentclass\{[^}]*$/g;
      
      // 检查是否有未闭合的方括号
      let hasUnmatchedBracket = false;
      const matches = text.match(/\\documentclass\[[^\]]*$/g);
      if (matches) {
        hasUnmatchedBracket = true;
      }
      
      // 检查是否有未闭合的花括号
      let hasUnmatchedBrace = false;
      if (unmatchedBrace.test(text)) {
        hasUnmatchedBrace = true;
      }
      
      if (hasUnmatchedBracket || hasUnmatchedBrace) {
        errors.push('未闭合的documentclass命令');
        return {
          success: false,
          processedText: text,
          documentClass: null,
          warnings,
          errors
        };
      }

      while ((match = docClassRegex.exec(text)) !== null) {
        classCount++;
        const optionsString = match[1] || '';
        const docType = match[2].trim();

        // 检查重复定义
        if (classCount > 1) {
          warnings.push('重复的documentclass定义，使用最后一个定义');
        }

        // 解析选项
        const options = optionsString ? 
          optionsString.split(',').map(opt => opt.trim()).filter(opt => opt) : 
          [];

        // 创建文档类信息
        documentClass = {
          type: docType,
          options: options,
          command: match[0],
          position: {
            start: match.index,
            end: match.index + match[0].length,
            originalText: match[0]
          }
        };

        // 检查特殊文档类型
        if (docType === 'beamer') {
          documentClass.isPresentation = true;
        }

        // 检查是否为已知文档类型 (浏览器兼容版本)
        const knownDocumentClasses = ['article', 'report', 'book', 'letter', 'beamer', 'memoir', 'amsart', 'amsbook'];
        if (!knownDocumentClasses.includes(docType)) {
          warnings.push(`未知的文档类型: ${docType}`);
        }

        // 移除文档类命令
        processedText = processedText.replace(match[0], '');
      }

      return {
        success: errors.length === 0,
        processedText,
        documentClass,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`文档类处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        documentClass: null,
        warnings,
        errors
      };
    }
  }

  /**
   * 处理包声明
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processPackages(text) {
    const warnings = [];
    const errors = [];
    const packages = [];
    let processedText = text;

    try {
      // 包管理正则：支持可选参数和必需参数
      const packageRegex = /\\usepackage(?:\[([^\]]*)\])?\{([^}]+)\}/g;
      const packageNames = new Set(); // 用于检测重复包
      let match;

      // 检测未闭合的包声明
      const unmatched = /\\usepackage(?:\[[^\]]*\])?\{[^}]*$/g;
      if (unmatched.test(text)) {
        errors.push('未闭合的usepackage命令');
        return {
          success: false,
          processedText: text,
          packages: [],
          warnings,
          errors
        };
      }

      while ((match = packageRegex.exec(text)) !== null) {
        const optionsString = match[1] || '';
        const packageName = match[2].trim();

        // 解析选项
        const options = optionsString ? 
          optionsString.split(',').map(opt => opt.trim()).filter(opt => opt) : 
          [];

        // 创建包信息
        const packageInfo = {
          name: packageName,
          options: options,
          command: match[0],
          position: {
            start: match.index,
            end: match.index + match[0].length,
            originalText: match[0]
          }
        };

        // 解析包名列表（如果包含逗号分隔的多个包）
        if (packageName.includes(',')) {
          const packageList = packageName.split(',').map(name => name.trim()).filter(name => name);
          packageInfo.packageList = packageList;
        }

        // 检查复杂选项
        if (options.some(opt => opt.includes('='))) {
          packageInfo.hasComplexOptions = true;
        }

        // 检查重复包声明
        const mainPackageName = packageInfo.packageList ? 
          packageInfo.packageList[0] : 
          packageName;
        
        if (packageNames.has(mainPackageName)) {
          warnings.push(`重复包声明: ${mainPackageName}`);
        }
        packageNames.add(mainPackageName);

        packages.push(packageInfo);

        // 移除包声明命令
        processedText = processedText.replace(match[0], '');
      }

      return {
        success: errors.length === 0,
        processedText,
        packages,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`包管理处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        packages: [],
        warnings,
        errors
      };
    }
  }

  /**
   * 检查引用是否在其定义的标签范围内（自引用检测）
   * @private
   */
  isWithinLabel(text, refPosition, labelId) {
    // 简化的自引用检测逻辑
    const labelPattern = new RegExp(`\\\\label\\{${labelId}\\}`, 'g');
    let labelMatch;
    
    while ((labelMatch = labelPattern.exec(text)) !== null) {
      // 如果引用位置在标签定义之后的合理范围内，可能是自引用
      if (refPosition > labelMatch.index && refPosition < labelMatch.index + 500) {
        return true;
      }
    }
    
    return false;
  }

  // ========== 目录生成系统处理方法 ==========

  /**
   * 处理目录命令 - 完整目录生成流程
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processTableOfContents(text) {
    const warnings = [];
    const errors = [];

    try {
      // 1. 检测tableofcontents命令
      const tocMatches = text.match(/\\tableofcontents/g) || [];
      const hasTOC = tocMatches.length > 0;

      if (tocMatches.length > 1) {
        warnings.push('检测到多个tableofcontents命令，只处理第一个');
      }

      // 2. 始终收集所有章节信息（无论是否有tableofcontents命令）
      const sections = this.collectSections(text, warnings);

      // 3. 生成章节编号
      const TOCProcessor = require('../processors/TOCProcessor');
      const tocProcessor = new TOCProcessor();
      const numberedSections = tocProcessor.formatSectionNumbers(sections);

      // 4. 处理目录警告和HTML生成
      let processedText = text;
      
      if (hasTOC) {
        // 只在有tableofcontents命令时处理目录
        if (sections.length === 0) {
          warnings.push('找到tableofcontents命令但没有找到章节内容，未找到章节内容');
        }

        // 生成目录HTML
        const tocHtml = tocProcessor.generateTOCHTML(numberedSections);

        // 替换tableofcontents命令为目录HTML
        processedText = text.replace(/\\tableofcontents/, tocHtml);
      }

      return {
        success: true,
        processedText,
        hasTOC,
        sections: numberedSections,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`目录处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        hasTOC: false,
        sections: [],
        warnings,
        errors
      };
    }
  }

  /**
   * 收集文档中的所有章节信息
   * @private
   */
  collectSections(text, warnings = []) {
    const sections = [];
    const sectionTypes = ['section', 'subsection', 'subsubsection', 'paragraph', 'subparagraph'];
    
    // 统一的章节匹配正则 - 支持嵌套大括号
    const sectionRegex = /\\(section|subsection|subsubsection|paragraph|subparagraph)\{((?:[^{}]|\{[^}]*\})*)\}(?:\\label\{([^}]*)\})?/g;
    
    let match;
    let sectionIndex = 0;

    while ((match = sectionRegex.exec(text)) !== null) {
      const element = match[1];
      const title = match[2].trim();
      const labelId = match[3];
      
      // 计算层级
      const level = this.getSectionLevel(element);
      
      // 检查格式化标题
      const hasFormatting = this.hasLatexFormatting(title);
      
      // 生成ID
      const TOCProcessor = require('../processors/TOCProcessor');
      const tocProcessor = new TOCProcessor();
      const id = labelId || tocProcessor.generateAutoId(title, sectionIndex);
      
      sections.push({
        level,
        title,
        element,
        id,
        hasFormatting,
        position: {
          start: match.index,
          end: match.index + match[0].length,
          originalText: match[0]
        }
      });
      
      sectionIndex++;
    }

    return sections;
  }

  /**
   * 获取章节层级
   * @private
   */
  getSectionLevel(element) {
    const levels = {
      'section': 1,
      'subsection': 2,
      'subsubsection': 3,
      'paragraph': 4,
      'subparagraph': 5
    };
    return levels[element] || 1;
  }

  /**
   * 检查标题是否包含LaTeX格式化命令
   * @private
   */
  hasLatexFormatting(title) {
    return /\\(textbf|textit|emph|textsc|texttt)\{/.test(title);
  }

  // ========== 高级环境系统处理方法 ==========

  /**
   * 处理高级环境 - 完整高级环境处理流程
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processAdvancedEnvironments(text) {
    const warnings = [];
    const errors = [];

    try {
      // 1. 初始化处理器
      const AdvancedEnvironmentProcessor = require('../processors/AdvancedEnvironmentProcessor');
      const processor = new AdvancedEnvironmentProcessor();

      // 2. 检查环境匹配性
      const matchingResult = processor.checkEnvironmentMatching(text);
      if (!matchingResult.valid) {
        errors.push(...matchingResult.errors);
        return {
          success: false,
          processedText: text,
          environments: [],
          warnings,
          errors
        };
      }

      // 3. 解析所有高级环境
      const environments = this.parseAdvancedEnvironments(text, processor, warnings);

      // 4. 验证环境结构
      const validation = processor.validateEnvironmentStructure(environments);
      warnings.push(...validation.warnings);

      // 5. 生成HTML并替换
      let processedText = text;
      
      // 按位置倒序排列，避免替换时位置偏移
      const sortedEnvs = [...environments].sort((a, b) => b.position.start - a.position.start);
      
      sortedEnvs.forEach(env => {
        const envHtml = processor.generateEnvironmentHTML(env);
        processedText = processedText.substring(0, env.position.start) + 
                      envHtml + 
                      processedText.substring(env.position.end);
      });

      return {
        success: true,
        processedText,
        environments,
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`高级环境处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        environments: [],
        warnings,
        errors
      };
    }
  }

  /**
   * 解析所有高级环境
   * @private
   */
  parseAdvancedEnvironments(text, processor, warnings = []) {
    const environments = [];
    
    // 支持的环境类型
    const envTypes = ['theorem', 'proof', 'abstract', 'definition', 'lemma', 'corollary', 'proposition', 'remark', 'example'];
    
    // 为每种环境类型解析
    envTypes.forEach(envType => {
      const envMatches = this.extractEnvironments(text, envType, processor);
      environments.push(...envMatches);
    });

    // 按位置排序
    environments.sort((a, b) => a.position.start - b.position.start);

    // 生成编号
    processor.resetCounters();
    environments.forEach(env => {
      env.number = processor.generateNumber(env.type);
    });

    return environments;
  }

  /**
   * 提取特定类型的环境
   * @private
   */
  extractEnvironments(text, envType, processor) {
    const environments = [];
    
    // 匹配环境的正则表达式
    const envRegex = new RegExp(
      `\\\\begin\\{${envType}\\}(?:\\[([^\\]]*)\\])?(?:\\s*\\\\label\\{([^}]*)\\})?([\\s\\S]*?)\\\\end\\{${envType}\\}`,
      'g'
    );
    
    let match;
    let envIndex = 0;

    while ((match = envRegex.exec(text)) !== null) {
      const title = processor.parseTitle(match[1] || '');
      const labelId = match[2];
      const content = match[3].trim();
      
      // 检查空内容
      if (!content) {
        // 只警告，不阻止处理
        // warnings会在调用方添加
      }
      
      // 生成ID
      const id = labelId || processor.generateAutoId(envType, envIndex);
      
      environments.push({
        type: envType,
        title,
        content,
        id,
        number: '', // 将在后续步骤中生成
        hasLabel: !!labelId,
        position: {
          start: match.index,
          end: match.index + match[0].length,
          originalText: match[0]
        }
      });
      
      envIndex++;
    }

    return environments;
  }

  // ========== 文献引用系统处理方法 ==========

  /**
   * 处理文献引用 - 完整文献引用处理流程
   * @param {string} text - 输入文本
   * @returns {object} 处理结果
   */
  processBibliography(text) {
    const warnings = [];
    const errors = [];

    try {
      // 1. 初始化处理器
      const BibliographyProcessor = require('../processors/BibliographyProcessor');
      const processor = new BibliographyProcessor();
      processor.resetCounters();

      // 2. 解析所有引用
      const allCitations = this.extractCitations(text, processor, warnings);

      // 3. 解析文献条目
      const bibliographyData = this.extractBibliography(text, warnings, errors);

      // 4. 去重引用（只保留唯一的key，用于返回结果）
      const uniqueCitations = [];
      const seenKeys = new Set();
      allCitations.forEach(citation => {
        if (!seenKeys.has(citation.key)) {
          seenKeys.add(citation.key);
          uniqueCitations.push(citation);
        }
      });

      // 5. 验证引用完整性
      const validation = processor.validateCitations(uniqueCitations, bibliographyData.items || []);
      warnings.push(...validation.warnings);

      // 6. 生成HTML并替换
      let processedText = text;

      // 6a. 替换引用为编号链接（使用所有引用，包括重复的）
      processedText = this.replaceCitations(processedText, allCitations, bibliographyData.items || [], processor);

      // 5b. 替换文献环境为HTML
      if (bibliographyData.hasThebibliography) {
        const bibHtml = processor.generateBibliographyHTML(bibliographyData.items);
        processedText = processedText.replace(bibliographyData.fullMatch, bibHtml);
      } else if (bibliographyData.hasBibliographyFile) {
        const noticeHtml = processor.generateBibliographyNoticeHTML(bibliographyData.file);
        processedText = processedText.replace(bibliographyData.fileMatch, noticeHtml);
      }

      return {
        success: errors.length === 0,
        processedText,
        citations: uniqueCitations,
        bibliographyItems: bibliographyData.items || [],
        hasBibliography: bibliographyData.hasBibliographyFile,
        bibliographyFile: bibliographyData.file || '',
        warnings,
        errors
      };

    } catch (error) {
      errors.push(`文献引用处理错误: ${error.message}`);
      return {
        success: false,
        processedText: text,
        citations: [],
        bibliographyItems: [],
        hasBibliography: false,
        bibliographyFile: '',
        warnings,
        errors
      };
    }
  }

  /**
   * 提取所有引用
   * @private
   */
  extractCitations(text, processor, warnings) {
    const citations = [];
    
    // 匹配\cite命令，支持可选的页码参数
    const citeRegex = /\\cite(?:\[([^\]]*)\])?\{([^}]*)\}/g;
    let match;

    while ((match = citeRegex.exec(text)) !== null) {
      const pageInfo = match[1];
      const keyString = match[2];
      
      if (!keyString || keyString.trim() === '') {
        warnings.push('空的cite命令');
        continue;
      }

      const keys = processor.parseCitationKeys(keyString);
      keys.forEach(key => {
        if (key) {
          const number = processor.getCitationNumber(key);
          citations.push({
            key,
            number,
            type: 'cite',
            pageInfo: processor.parsePageInfo(pageInfo),
            position: {
              start: match.index,
              end: match.index + match[0].length,
              originalText: match[0]
            }
          });
        }
      });
    }

    return citations;
  }

  /**
   * 提取文献信息
   * @private
   */
  extractBibliography(text, warnings, errors) {
    const result = {
      hasThebibliography: false,
      hasBibliographyFile: false,
      items: [],
      file: '',
      fullMatch: '',
      fileMatch: ''
    };

    // 1. 检查thebibliography环境
    const beginRegex = /\\begin\{thebibliography\}\{[^}]*\}/;
    const beginMatch = text.match(beginRegex);
    
    if (beginMatch) {
      // 检查环境闭合
      const beginCount = (text.match(/\\begin\{thebibliography\}/g) || []).length;
      const endCount = (text.match(/\\end\{thebibliography\}/g) || []).length;
      
      if (beginCount > endCount) {
        errors.push('未闭合的thebibliography环境');
        return result; // 早期返回，不处理内容
      }
      
      // 如果环境匹配，处理内容
      const thebibRegex = /\\begin\{thebibliography\}\{[^}]*\}([\s\S]*?)\\end\{thebibliography\}/;
      const thebibMatch = text.match(thebibRegex);
      
      if (thebibMatch) {
        result.hasThebibliography = true;
        result.fullMatch = thebibMatch[0];
        
        // 解析bibitem条目
        const bibitemRegex = /\\bibitem\{([^}]*)\}\s*([^\n\\]*(?:\n(?!\\)[^\n\\]*)*)/g;
        const content = thebibMatch[1];
        let itemMatch;
        let itemNumber = 1;

        while ((itemMatch = bibitemRegex.exec(content)) !== null) {
          const key = itemMatch[1].trim();
          const content = itemMatch[2].trim();
          
          if (key && content) {
            result.items.push({
              key,
              number: itemNumber.toString(),
              content,
              position: {
                start: itemMatch.index,
                end: itemMatch.index + itemMatch[0].length,
                originalText: itemMatch[0]
              }
            });
            itemNumber++;
          }
        }
      }
    }

    // 2. 检查bibliography文件
    const bibFileRegex = /\\bibliography\{([^}]*)\}/;
    const fileMatch = text.match(bibFileRegex);
    
    if (fileMatch) {
      result.hasBibliographyFile = true;
      result.file = fileMatch[1].trim();
      result.fileMatch = fileMatch[0];
    }

    return result;
  }

  /**
   * 替换引用为编号链接
   * @private
   */
  replaceCitations(text, citations, bibliographyItems, processor) {
    // 按位置倒序排列，避免替换时位置偏移
    const sortedCitations = [...citations].sort((a, b) => b.position.start - a.position.start);
    
    // 按原始文本分组，处理多key的cite命令
    const citationGroups = new Map();
    
    sortedCitations.forEach(citation => {
      const key = citation.position.originalText;
      if (!citationGroups.has(key)) {
        citationGroups.set(key, {
          citations: [],
          position: citation.position
        });
      }
      citationGroups.get(key).citations.push(citation);
    });

    let processedText = text;

    // 处理每个引用组
    citationGroups.forEach((group, originalText) => {
      const keys = group.citations.map(c => c.key);
      const pageInfo = group.citations[0].pageInfo || '';
      
      // 检查是否所有引用都有对应的文献（只有在有文献条目时才检查）
      const hasBibliographyItems = bibliographyItems.length > 0;
      const allKeysExist = !hasBibliographyItems || keys.every(key => bibliographyItems.find(item => item.key === key));
      
      let formattedCitation;
      if (hasBibliographyItems && !allKeysExist) {
        // 如果有文献条目但某些引用未找到，显示[?]
        formattedCitation = '[?]';
      } else {
        // 格式化引用（使用所有citations而不是bibliographyItems进行编号查找）
        const citationNumbers = keys.map(key => {
          const citation = group.citations.find(c => c.key === key);
          return citation ? citation.number : '?';
        }).filter(n => n !== '?');
        
        if (citationNumbers.length === 0) {
          formattedCitation = '[?]';
        } else {
          const formatted = processor.compressNumberSequence(citationNumbers.map(n => parseInt(n)).sort((a, b) => a - b));
          formattedCitation = pageInfo ? `[${formatted}, ${pageInfo}]` : `[${formatted}]`;
        }
      }
      
      // 生成带链接的引用（如果有对应的文献且单个引用）
      const linkedCitation = keys.length === 1 && hasBibliographyItems && allKeysExist ? 
        processor.generateCitationLink(keys[0], formattedCitation) :
        formattedCitation;
      
      // 替换
      const pos = group.position;
      processedText = processedText.substring(0, pos.start) + 
                    linkedCitation + 
                    processedText.substring(pos.end);
    });

    return processedText;
  }

  /**
   * 处理代码块（```latex格式）
   * 使用占位符系统防止内容被后续层重复处理
   */
  processCodeBlocks(text, blocks, warnings, placeholders = []) {
    let processedText = text;
    let latexBlockIndex = 0;
    
    // 匹配 ```latex ... ``` 格式的代码块
    const codeBlockPattern = /```latex\s*\n([\s\S]*?)\n```/g;
    
    processedText = processedText.replace(codeBlockPattern, (match, content) => {
      const trimmedContent = content.trim();
      
      // 检查是否包含文档结构命令（theorem、proof等环境）
      const documentEnvironments = [
        'theorem', 'lemma', 'proposition', 'corollary',
        'definition', 'example', 'remark', 'proof'
      ];
      
      let hasDocumentEnvironment = false;
      for (const env of documentEnvironments) {
        if (trimmedContent.includes(`\\begin{${env}}`) || trimmedContent.includes(`\\end{${env}}`)) {
          hasDocumentEnvironment = true;
          break;
        }
      }
      
      if (hasDocumentEnvironment) {
        // 如果包含文档环境，直接处理内容而不是创建占位符
        // 这样theorem环境可以被processTheoremEnvironments处理
        return trimmedContent;
      } else {
        // 如果是纯数学内容，创建占位符以便后续转换为$$格式
        const placeholder = `__L2_LATEX_BLOCK_${latexBlockIndex}__`;
        
        // 记录代码块信息
        blocks.push({
          type: 'latex_code_block',
          content: trimmedContent,
          originalMatch: match,
          placeholder: placeholder,
          index: latexBlockIndex
        });
        
        // 存储占位符信息，标记为纯数学内容
        placeholders.push({
          id: placeholder,
          content: `$$\n${trimmedContent}\n$$`,  // 转换为数学格式
          type: 'latex_math_block',
          layer: 2
        });
        
        latexBlockIndex++;
        
        // 返回占位符
        return placeholder;
      }
    });
    
    // 处理可能的其他格式变体（如 ```LaTeX 或 ```tex）
    const alternativePatterns = [
      /```LaTeX\s*\n([\s\S]*?)\n```/g,
      /```tex\s*\n([\s\S]*?)\n```/g,
      /```TeX\s*\n([\s\S]*?)\n```/g
    ];
    
    alternativePatterns.forEach(pattern => {
      processedText = processedText.replace(pattern, (match, content) => {
        // 创建占位符
        const placeholder = `__L2_LATEX_BLOCK_${latexBlockIndex}__`;
        
        blocks.push({
          type: 'latex_code_block',
          content: content.trim(),
          originalMatch: match,
          placeholder: placeholder,
          index: latexBlockIndex
        });
        
        // 存储占位符信息
        placeholders.push({
          id: placeholder,
          content: content.trim(),
          type: 'latex_code_block',
          layer: 2
        });
        
        latexBlockIndex++;
        
        return placeholder;
      });
    });
    
    return processedText;
  }
}

// 统一导出机制：同时支持浏览器和Node.js
if (typeof module !== 'undefined' && module.exports) {
  // Node.js环境（包括Jest测试环境）
  module.exports = BlockElementParser;
  console.log('[BlockElementParser] 模块已加载到Node.js环境');
}

if (typeof window !== 'undefined') {
  // 浏览器环境（包括测试环境中的模拟浏览器）
  window.BlockElementParser = BlockElementParser;
  if (typeof global !== 'undefined') {
    // 测试环境：同时注册到global
    global.BlockElementParser = BlockElementParser;
  }
}
