/**
 * LaTeX HTML映射配置测试
 * 测试 LATEX_HTML_MAPPING 对外接口规范
 */

// 导入被测试的模块
const { LATEX_HTML_MAPPING } = require('../latex-html-mapping.js');

describe('LATEX_HTML_MAPPING Interface Tests', () => {
  
  describe('接口存在性测试', () => {
    test('应该导出 LATEX_HTML_MAPPING 对象', () => {
      expect(LATEX_HTML_MAPPING).toBeDefined();
      expect(typeof LATEX_HTML_MAPPING).toBe('object');
    });

    test('应该不为空对象', () => {
      expect(Object.keys(LATEX_HTML_MAPPING).length).toBeGreaterThan(0);
    });
  });

  describe('文档结构配置测试', () => {
    const documentCommands = ['title', 'author', 'date', 'section', 'subsection', 'subsubsection'];
    
    test.each(documentCommands)('应该包含 %s 命令配置', (command) => {
      expect(LATEX_HTML_MAPPING[command]).toBeDefined();
    });

    test('title 配置应该符合接口规范', () => {
      const titleConfig = LATEX_HTML_MAPPING.title;
      expect(titleConfig).toHaveProperty('tag');
      expect(titleConfig).toHaveProperty('className');
      expect(titleConfig.tag).toBe('h1');
      expect(titleConfig.className).toBe('document-title');
      expect(titleConfig.attributes).toBeDefined();
    });

    test('section 配置应该包含正确的accessibility属性', () => {
      const sectionConfig = LATEX_HTML_MAPPING.section;
      expect(sectionConfig.attributes).toHaveProperty('role', 'heading');
      expect(sectionConfig.attributes).toHaveProperty('aria-level', '2');
    });
  });

  describe('接口规范一致性测试', () => {
    test('所有有效配置项都应该有 tag 属性', () => {
      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        if (!config.ignore && !config.isClosing) {
          expect(config).toHaveProperty('tag');
          expect(typeof config.tag).toBe('string');
        }
      });
    });

    test('所有有效配置项都应该有 className 属性或为特殊类型', () => {
      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        if (!config.ignore && !config.isClosing && !config.selfClosing) {
          expect(config).toHaveProperty('className');
          expect(typeof config.className).toBe('string');
        }
      });
    });

    test('attributes 属性应该为对象类型', () => {
      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        if (config.attributes) {
          expect(typeof config.attributes).toBe('object');
          expect(config.attributes).not.toBeNull();
        }
      });
    });
  });

  describe('特殊配置类型测试', () => {
    test('应该正确处理 ignore 类型配置', () => {
      const ignoreCommands = ['documentclass', 'usepackage', 'geometry'];
      ignoreCommands.forEach(command => {
        expect(LATEX_HTML_MAPPING[command]).toHaveProperty('ignore', true);
      });
    });

    test('应该正确处理容器类型配置', () => {
      const containerConfig = LATEX_HTML_MAPPING.document_begin;
      expect(containerConfig).toHaveProperty('isContainer', true);
      expect(containerConfig.tag).toBe('div');
    });

    test('应该正确处理关闭标签配置', () => {
      const endConfig = LATEX_HTML_MAPPING.document_end;
      expect(endConfig).toHaveProperty('isClosing', true);
      expect(endConfig.tag).toBe('/div');
    });

    test('应该正确处理自闭合标签配置', () => {
      const separatorConfig = LATEX_HTML_MAPPING.maketitle;
      expect(separatorConfig).toHaveProperty('selfClosing', true);
    });
  });

  describe('映射完整性测试', () => {
    test('应该包含所有interface.md中规定的主要配置项', () => {
      // 文档结构
      const documentStructure = ['title', 'author', 'date', 'section', 'subsection', 'subsubsection'];
      documentStructure.forEach(item => {
        expect(LATEX_HTML_MAPPING).toHaveProperty(item);
      });
    });

    test('配置项数量应该合理', () => {
      const configCount = Object.keys(LATEX_HTML_MAPPING).length;
      expect(configCount).toBeGreaterThan(10); // 至少应该有基本配置项
      expect(configCount).toBeLessThan(100); // 不应该过多
    });
  });

  describe('接口数据结构验证', () => {
    test('配置对象应该只包含规范中定义的属性', () => {
      const allowedProperties = [
        'tag', 'className', 'attributes', 'wrapper', 'transform',
        'ignore', 'isContainer', 'isClosing', 'selfClosing'
      ];

      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        const configKeys = Object.keys(config);
        configKeys.forEach(key => {
          expect(allowedProperties).toContain(key);
        });
      });
    });

    test('className 命名应该遵循CSS类命名规范', () => {
      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        if (config.className) {
          // CSS类名应该只包含字母、数字、连字符和下划线
          expect(config.className).toMatch(/^[a-zA-Z][a-zA-Z0-9_-]*$/);
        }
      });
    });
  });

  describe('扩展性测试', () => {
    test('应该支持动态添加新的映射配置', () => {
      const originalKeys = Object.keys(LATEX_HTML_MAPPING);
      
      // 添加新配置
      LATEX_HTML_MAPPING.testCommand = {
        tag: 'span',
        className: 'test-command'
      };

      expect(LATEX_HTML_MAPPING.testCommand).toBeDefined();
      expect(LATEX_HTML_MAPPING.testCommand.tag).toBe('span');

      // 清理测试数据
      delete LATEX_HTML_MAPPING.testCommand;
      expect(Object.keys(LATEX_HTML_MAPPING)).toEqual(originalKeys);
    });
  });

  describe('错误处理和边界条件测试', () => {
    test('访问不存在的配置项应该返回 undefined', () => {
      expect(LATEX_HTML_MAPPING.nonexistentCommand).toBeUndefined();
    });

    test('配置对象应该是可遍历的', () => {
      let count = 0;
      for (let key in LATEX_HTML_MAPPING) {
        count++;
      }
      expect(count).toBeGreaterThan(0);
    });

    test('配置对象应该支持 Object 方法', () => {
      expect(Object.keys(LATEX_HTML_MAPPING)).toBeInstanceOf(Array);
      expect(Object.values(LATEX_HTML_MAPPING)).toBeInstanceOf(Array);
      expect(Object.entries(LATEX_HTML_MAPPING)).toBeInstanceOf(Array);
    });
  });

  describe('模块导出测试', () => {
    test('应该正确导出到 module.exports', () => {
      // 在Node.js环境中应该能够被正确导入
      expect(typeof require('../latex-html-mapping.js')).toBe('object');
    });

    test('导出的对象应该包含 LATEX_HTML_MAPPING', () => {
      const exportedModule = require('../latex-html-mapping.js');
      expect(exportedModule).toHaveProperty('LATEX_HTML_MAPPING');
    });
  });
});