/**
 * LaTeX CSS规则配置测试
 * 测试 LATEX_CSS_RULES 和 CSS_UTILS 对外接口规范
 */

// 导入被测试的模块
const { LATEX_CSS_RULES, CSS_UTILS } = require('../latex-css-rules.js');

describe('LATEX_CSS_RULES Interface Tests', () => {
  
  describe('接口存在性测试', () => {
    test('应该导出 LATEX_CSS_RULES 对象', () => {
      expect(LATEX_CSS_RULES).toBeDefined();
      expect(typeof LATEX_CSS_RULES).toBe('object');
    });

    test('应该导出 CSS_UTILS 工具对象', () => {
      expect(CSS_UTILS).toBeDefined();
      expect(typeof CSS_UTILS).toBe('object');
    });

    test('LATEX_CSS_RULES 应该不为空对象', () => {
      expect(Object.keys(LATEX_CSS_RULES).length).toBeGreaterThan(0);
    });
  });

  describe('主要规则类别测试', () => {
    const expectedClasses = [
      'latex-document',      // 文档容器
      'document-title',      // 标题样式
      'document-author',     // 作者样式
      'document-date',       // 日期样式
      'latex-section',       // 章节样式
      'latex-subsection',    // 子章节样式
      'latex-subsubsection', // 子子章节样式
      'latex-content'        // 内容样式
    ];

    test.each(expectedClasses)('应该包含 %s 样式规则', (className) => {
      expect(LATEX_CSS_RULES[className]).toBeDefined();
    });

    test('文档结构样式应该完整', () => {
      const structureStyles = ['document-title', 'document-author', 'document-date'];
      structureStyles.forEach(style => {
        expect(LATEX_CSS_RULES[style]).toBeDefined();
        expect(typeof LATEX_CSS_RULES[style]).toBe('object');
      });
    });

    test('章节标题样式应该完整', () => {
      const sectionStyles = ['latex-section', 'latex-subsection', 'latex-subsubsection'];
      sectionStyles.forEach(style => {
        expect(LATEX_CSS_RULES[style]).toBeDefined();
        expect(LATEX_CSS_RULES[style]).toHaveProperty('fontSize');
        expect(LATEX_CSS_RULES[style]).toHaveProperty('fontWeight');
        expect(LATEX_CSS_RULES[style]).toHaveProperty('margin');
      });
    });
  });

  describe('接口规范一致性测试', () => {
    test('所有样式规则都应该是对象类型', () => {
      Object.entries(LATEX_CSS_RULES).forEach(([className, styles]) => {
        expect(typeof styles).toBe('object');
        expect(styles).not.toBeNull();
      });
    });

    test('样式对象应该包含有效的CSS属性', () => {
      const validCSSProperties = [
        'fontSize', 'fontWeight', 'fontFamily', 'color', 'background',
        'margin', 'padding', 'textAlign', 'lineHeight', 'maxWidth',
        'width', 'height', 'border', 'textIndent'
      ];

      Object.entries(LATEX_CSS_RULES).forEach(([className, styles]) => {
        Object.keys(styles).forEach(property => {
          expect(validCSSProperties).toContain(property);
        });
      });
    });

    test('CSS值应该为字符串类型', () => {
      Object.entries(LATEX_CSS_RULES).forEach(([className, styles]) => {
        Object.entries(styles).forEach(([property, value]) => {
          expect(typeof value).toBe('string');
        });
      });
    });
  });

  describe('文档容器样式测试', () => {
    test('latex-document 应该包含基础文档样式', () => {
      const docStyle = LATEX_CSS_RULES['latex-document'];
      
      expect(docStyle).toHaveProperty('maxWidth');
      expect(docStyle).toHaveProperty('margin');
      expect(docStyle).toHaveProperty('padding');
      expect(docStyle).toHaveProperty('fontFamily');
      expect(docStyle).toHaveProperty('fontSize');
      expect(docStyle).toHaveProperty('lineHeight');
      expect(docStyle).toHaveProperty('color');
      expect(docStyle).toHaveProperty('background');
    });

    test('文档样式值应该合理', () => {
      const docStyle = LATEX_CSS_RULES['latex-document'];
      
      expect(docStyle.maxWidth).toMatch(/^\d+(\.\d+)?(in|px|em|rem|%)$/);
      expect(docStyle.fontSize).toMatch(/^\d+(\.\d+)?(pt|px|em|rem)$/);
      expect(docStyle.lineHeight).toMatch(/^\d+(\.\d+)?$/);
      expect(docStyle.background).toBe('white');
    });
  });

  describe('标题样式测试', () => {
    test('标题样式应该有适当的层级关系', () => {
      const titleSize = parseFloat(LATEX_CSS_RULES['document-title'].fontSize);
      const sectionSize = parseFloat(LATEX_CSS_RULES['latex-section'].fontSize);
      const subsectionSize = parseFloat(LATEX_CSS_RULES['latex-subsection'].fontSize);
      const subsubsectionSize = parseFloat(LATEX_CSS_RULES['latex-subsubsection'].fontSize);
      
      // 标题大小应该呈递减关系
      expect(titleSize).toBeGreaterThan(sectionSize);
      expect(sectionSize).toBeGreaterThan(subsectionSize);
      expect(subsectionSize).toBeGreaterThan(subsubsectionSize);
    });

    test('标题样式应该包含居中对齐', () => {
      const titleStyles = ['document-title', 'document-author', 'document-date'];
      titleStyles.forEach(style => {
        expect(LATEX_CSS_RULES[style]).toHaveProperty('textAlign', 'center');
      });
    });
  });
});

describe('CSS_UTILS Interface Tests', () => {
  
  describe('工具函数存在性测试', () => {
    test('应该包含 objectToCss 方法', () => {
      expect(CSS_UTILS.objectToCss).toBeDefined();
      expect(typeof CSS_UTILS.objectToCss).toBe('function');
    });

    test('应该包含 generateFullCSS 方法', () => {
      expect(CSS_UTILS.generateFullCSS).toBeDefined();
      expect(typeof CSS_UTILS.generateFullCSS).toBe('function');
    });
  });

  describe('objectToCss 方法测试', () => {
    test('应该正确转换样式对象为CSS字符串', () => {
      const testStyles = {
        fontSize: '16px',
        color: 'red',
        margin: '10px'
      };

      const cssString = CSS_UTILS.objectToCss('test-class', testStyles);
      
      expect(typeof cssString).toBe('string');
      expect(cssString).toContain('.test-class');
      expect(cssString).toContain('font-size: 16px');
      expect(cssString).toContain('color: red');
      expect(cssString).toContain('margin: 10px');
    });

    test('应该正确处理驼峰命名转换', () => {
      const testStyles = {
        fontSize: '16px',
        fontWeight: 'bold',
        backgroundColor: 'blue',
        textAlign: 'center'
      };

      const cssString = CSS_UTILS.objectToCss('test-class', testStyles);
      
      expect(cssString).toContain('font-size');
      expect(cssString).toContain('font-weight');
      expect(cssString).toContain('background-color');
      expect(cssString).toContain('text-align');
    });

    test('应该处理空样式对象', () => {
      const cssString = CSS_UTILS.objectToCss('empty-class', {});
      
      expect(typeof cssString).toBe('string');
      expect(cssString).toContain('.empty-class');
      expect(cssString).toMatch(/\{\s*\}/); // 空的CSS块
    });
  });

  describe('generateFullCSS 方法测试', () => {
    test('应该生成完整的CSS样式表', () => {
      const fullCSS = CSS_UTILS.generateFullCSS();
      
      expect(typeof fullCSS).toBe('string');
      expect(fullCSS.length).toBeGreaterThan(0);
      expect(fullCSS).toContain('LaTeX Parser V3');
    });

    test('生成的CSS应该包含所有配置的样式类', () => {
      const fullCSS = CSS_UTILS.generateFullCSS();
      
      Object.keys(LATEX_CSS_RULES).forEach(className => {
        expect(fullCSS).toContain(`.${className}`);
      });
    });

    test('生成的CSS应该是有效的CSS格式', () => {
      const fullCSS = CSS_UTILS.generateFullCSS();
      
      // 检查CSS基本结构
      expect(fullCSS).toMatch(/\.[a-zA-Z-_]+\s*\{[^}]*\}/); // CSS规则格式
      
      // 检查是否包含CSS属性
      expect(fullCSS).toContain(':');
      expect(fullCSS).toContain(';');
    });

    test('应该能够重复调用而不出错', () => {
      expect(() => {
        CSS_UTILS.generateFullCSS();
        CSS_UTILS.generateFullCSS();
        CSS_UTILS.generateFullCSS();
      }).not.toThrow();
    });
  });

  describe('CSS工具函数边界条件测试', () => {
    test('objectToCss 应该处理特殊字符', () => {
      const testStyles = {
        content: '"Hello World"',
        fontFamily: '"Times New Roman", serif'
      };

      expect(() => {
        CSS_UTILS.objectToCss('special-class', testStyles);
      }).not.toThrow();
    });

    test('应该处理undefined和null值', () => {
      expect(() => {
        CSS_UTILS.objectToCss('test-class', undefined);
      }).not.toThrow();

      expect(() => {
        CSS_UTILS.objectToCss(undefined, {});
      }).not.toThrow();
    });
  });
});

describe('Config模块集成测试', () => {
  
  describe('模块间兼容性测试', () => {
    test('HTML映射的className应该在CSS规则中有对应样式', () => {
      const { LATEX_HTML_MAPPING } = require('../latex-html-mapping.js');
      
      // 检查主要的映射类名是否在CSS规则中存在
      const mappingClassNames = Object.values(LATEX_HTML_MAPPING)
        .filter(config => config.className)
        .map(config => config.className);

      const commonClasses = [
        'document-title', 'document-author', 'document-date',
        'latex-section', 'latex-subsection', 'latex-subsubsection',
        'latex-content'
      ];

      commonClasses.forEach(className => {
        if (mappingClassNames.includes(className)) {
          expect(LATEX_CSS_RULES[className]).toBeDefined();
        }
      });
    });
  });

  describe('错误处理约定测试', () => {
    test('配置模块应该能够处理加载失败情况', () => {
      // 确保配置对象存在且可访问
      expect(LATEX_CSS_RULES).toBeTruthy();
      expect(CSS_UTILS).toBeTruthy();
    });

    test('核心配置项应该始终可用', () => {
      const coreConfigs = ['latex-document', 'document-title', 'latex-section'];
      coreConfigs.forEach(config => {
        expect(LATEX_CSS_RULES[config]).toBeDefined();
      });
    });
  });

  describe('扩展性设计测试', () => {
    test('应该支持动态添加新的CSS规则', () => {
      const originalKeys = Object.keys(LATEX_CSS_RULES);
      
      // 添加新规则
      LATEX_CSS_RULES['test-rule'] = {
        color: 'blue',
        fontSize: '14px'
      };

      expect(LATEX_CSS_RULES['test-rule']).toBeDefined();
      
      // 验证CSS生成包含新规则
      const fullCSS = CSS_UTILS.generateFullCSS();
      expect(fullCSS).toContain('.test-rule');

      // 清理测试数据
      delete LATEX_CSS_RULES['test-rule'];
      expect(Object.keys(LATEX_CSS_RULES)).toEqual(originalKeys);
    });
  });

  describe('模块导出测试', () => {
    test('应该正确导出所有接口', () => {
      const exportedModule = require('../latex-css-rules.js');
      expect(exportedModule).toHaveProperty('LATEX_CSS_RULES');
      expect(exportedModule).toHaveProperty('CSS_UTILS');
    });

    test('导出的接口应该与直接引用一致', () => {
      const exportedModule = require('../latex-css-rules.js');
      expect(exportedModule.LATEX_CSS_RULES).toBe(LATEX_CSS_RULES);
      expect(exportedModule.CSS_UTILS).toBe(CSS_UTILS);
    });
  });
});