/**
 * Config模块集成测试
 * 测试整个Config模块的对外接口和依赖关系
 */

// 导入配置模块
const { LATEX_HTML_MAPPING } = require('../latex-html-mapping.js');
const { LATEX_CSS_RULES, CSS_UTILS } = require('../latex-css-rules.js');

describe('Config Module Integration Tests', () => {
  
  describe('模块完整性测试', () => {
    test('所有配置模块都应该成功加载', () => {
      expect(LATEX_HTML_MAPPING).toBeDefined();
      expect(LATEX_CSS_RULES).toBeDefined();
      expect(CSS_UTILS).toBeDefined();
    });

    test('配置对象应该包含预期的数据结构', () => {
      expect(typeof LATEX_HTML_MAPPING).toBe('object');
      expect(typeof LATEX_CSS_RULES).toBe('object');
      expect(typeof CSS_UTILS).toBe('object');
      
      expect(Object.keys(LATEX_HTML_MAPPING).length).toBeGreaterThan(0);
      expect(Object.keys(LATEX_CSS_RULES).length).toBeGreaterThan(0);
      expect(Object.keys(CSS_UTILS).length).toBeGreaterThan(0);
    });
  });

  describe('配置一致性测试', () => {
    test('HTML映射中的className应该在CSS规则中有对应定义', () => {
      // 收集所有HTML映射中使用的className
      const usedClassNames = new Set();
      Object.values(LATEX_HTML_MAPPING).forEach(config => {
        if (config.className) {
          usedClassNames.add(config.className);
        }
      });

      // 检查这些className是否在CSS规则中定义
      const definedClassNames = new Set(Object.keys(LATEX_CSS_RULES));
      
      usedClassNames.forEach(className => {
        // 如果className在映射中被使用，它应该在CSS规则中被定义
        // 或者是一个已知的不需要样式的类
        const skipClasses = ['title-separator', 'latex-document']; // 可能有特殊处理的类
        if (!skipClasses.includes(className)) {
          expect(definedClassNames.has(className)).toBeTruthy();
        }
      });
    });

    test('CSS规则中的类名应该有对应的HTML映射或合理的用途', () => {
      // 检查CSS规则是否都有用途
      const mappingClassNames = new Set();
      Object.values(LATEX_HTML_MAPPING).forEach(config => {
        if (config.className) {
          mappingClassNames.add(config.className);
        }
      });

      Object.keys(LATEX_CSS_RULES).forEach(className => {
        // 每个CSS类都应该被使用，或者是通用的基础类
        const isUsedInMapping = mappingClassNames.has(className);
        const isBaseClass = ['latex-document', 'latex-content'].includes(className);
        const isTitleClass = className.startsWith('document-') || className.startsWith('title-');
        
        expect(isUsedInMapping || isBaseClass || isTitleClass).toBeTruthy();
      });
    });
  });

  describe('接口规范合规性测试', () => {
    test('HTML映射配置应该符合接口文档规范', () => {
      // 检查interface.md中规定的配置项结构
      Object.entries(LATEX_HTML_MAPPING).forEach(([command, config]) => {
        // 每个配置项都应该有预期的结构
        if (!config.ignore && !config.isClosing) {
          expect(config).toHaveProperty('tag');
          expect(typeof config.tag).toBe('string');
        }
        
        if (config.className) {
          expect(typeof config.className).toBe('string');
        }
        
        if (config.attributes) {
          expect(typeof config.attributes).toBe('object');
        }
        
        if (config.wrapper) {
          expect(typeof config.wrapper).toBe('object');
        }
        
        if (config.transform) {
          expect(typeof config.transform).toBe('function');
        }
      });
    });

    test('CSS规则配置应该符合接口文档规范', () => {
      // 检查CSS规则结构
      Object.entries(LATEX_CSS_RULES).forEach(([className, rules]) => {
        expect(typeof rules).toBe('object');
        expect(rules).not.toBeNull();
        
        // styles 属性应该是对象
        Object.entries(rules).forEach(([property, value]) => {
          expect(typeof value).toBe('string'); // CSS值应该是字符串
        });
      });
    });
  });

  describe('依赖关系验证测试', () => {
    test('配置模块应该是无依赖的基础模块', () => {
      // 通过检查文件内容确认没有外部依赖
      const fs = require('fs');
      const path = require('path');
      
      const mappingFile = fs.readFileSync(
        path.join(__dirname, '../latex-html-mapping.js'), 
        'utf8'
      );
      const cssFile = fs.readFileSync(
        path.join(__dirname, '../latex-css-rules.js'), 
        'utf8'
      );
      
      // 不应该包含 require 其他模块的语句（除了模块导出逻辑）
      const requirePattern = /require\s*\(\s*['"][^'"]*['"]\s*\)/g;
      const mappingRequires = mappingFile.match(requirePattern) || [];
      const cssRequires = cssFile.match(requirePattern) || [];
      
      expect(mappingRequires.length).toBe(0);
      expect(cssRequires.length).toBe(0);
    });

    test('应该提供完整的导出接口', () => {
      // 测试模块导出
      const mappingModule = require('../latex-html-mapping.js');
      const cssModule = require('../latex-css-rules.js');
      
      expect(mappingModule).toHaveProperty('LATEX_HTML_MAPPING');
      expect(cssModule).toHaveProperty('LATEX_CSS_RULES');
      expect(cssModule).toHaveProperty('CSS_UTILS');
    });
  });

  describe('错误处理约定验证', () => {
    test('配置对象应该能处理异常访问', () => {
      // 访问不存在的配置项不应该抛出错误
      expect(() => {
        const nonExistent = LATEX_HTML_MAPPING.nonExistentCommand;
        const invalidCSS = LATEX_CSS_RULES.invalidClass;
      }).not.toThrow();
    });

    test('CSS工具函数应该能处理边界条件', () => {
      expect(() => {
        CSS_UTILS.objectToCss('', {});
        CSS_UTILS.objectToCss('test', null);
        CSS_UTILS.generateFullCSS();
      }).not.toThrow();
    });

    test('核心配置项应该始终可用', () => {
      // 检查核心配置项的存在
      const coreHtmlMappings = ['title', 'section', 'paragraph'];
      const coreCssRules = ['latex-document', 'document-title'];
      
      coreHtmlMappings.forEach(item => {
        expect(LATEX_HTML_MAPPING[item]).toBeDefined();
      });
      
      coreCssRules.forEach(rule => {
        expect(LATEX_CSS_RULES[rule]).toBeDefined();
      });
    });
  });

  describe('扩展性设计验证', () => {
    test('应该支持动态添加新的LaTeX命令映射', () => {
      const originalMappingKeys = Object.keys(LATEX_HTML_MAPPING);
      
      // 添加新命令
      LATEX_HTML_MAPPING.newCommand = {
        tag: 'div',
        className: 'new-command-class'
      };
      
      expect(LATEX_HTML_MAPPING.newCommand).toBeDefined();
      expect(Object.keys(LATEX_HTML_MAPPING).length).toBe(originalMappingKeys.length + 1);
      
      // 清理
      delete LATEX_HTML_MAPPING.newCommand;
      expect(Object.keys(LATEX_HTML_MAPPING)).toEqual(originalMappingKeys);
    });

    test('应该支持自定义CSS主题配置', () => {
      const originalCSSKeys = Object.keys(LATEX_CSS_RULES);
      
      // 添加新主题样式
      LATEX_CSS_RULES['custom-theme'] = {
        color: 'blue',
        fontSize: '16px',
        fontFamily: 'Arial, sans-serif'
      };
      
      expect(LATEX_CSS_RULES['custom-theme']).toBeDefined();
      
      // 验证CSS生成包含新样式
      const generatedCSS = CSS_UTILS.generateFullCSS();
      expect(generatedCSS).toContain('.custom-theme');
      expect(generatedCSS).toContain('color: blue');
      
      // 清理
      delete LATEX_CSS_RULES['custom-theme'];
      expect(Object.keys(LATEX_CSS_RULES)).toEqual(originalCSSKeys);
    });

    test('应该支持配置项的运行时修改', () => {
      // 备份原始配置
      const originalTitle = { ...LATEX_HTML_MAPPING.title };
      
      // 修改配置
      LATEX_HTML_MAPPING.title.className = 'modified-title';
      expect(LATEX_HTML_MAPPING.title.className).toBe('modified-title');
      
      // 恢复原始配置
      LATEX_HTML_MAPPING.title = originalTitle;
      expect(LATEX_HTML_MAPPING.title.className).toBe('document-title');
    });
  });

  describe('性能和内存测试', () => {
    test('配置对象访问应该高效', () => {
      const startTime = process.hrtime.bigint();
      
      // 执行多次配置访问
      for (let i = 0; i < 1000; i++) {
        const titleConfig = LATEX_HTML_MAPPING.title;
        const sectionConfig = LATEX_HTML_MAPPING.section;
        const docStyle = LATEX_CSS_RULES['latex-document'];
      }
      
      const endTime = process.hrtime.bigint();
      const duration = Number(endTime - startTime) / 1000000; // 转换为毫秒
      
      // 1000次访问应该在合理时间内完成（< 10ms）
      expect(duration).toBeLessThan(10);
    });

    test('CSS生成应该在合理时间内完成', () => {
      const startTime = process.hrtime.bigint();
      
      const css = CSS_UTILS.generateFullCSS();
      
      const endTime = process.hrtime.bigint();
      const duration = Number(endTime - startTime) / 1000000;
      
      expect(duration).toBeLessThan(50); // CSS生成应该 < 50ms
      expect(css.length).toBeGreaterThan(100); // 生成的CSS应该有实际内容
    });
  });

  describe('文档示例验证', () => {
    test('interface.md中的使用示例应该有效', () => {
      // 验证文档中的示例代码
      const mapping = LATEX_HTML_MAPPING.textbf;
      
      // 注意：由于当前实现中没有textbf，我们测试实际存在的配置
      const titleMapping = LATEX_HTML_MAPPING.title;
      expect(titleMapping).toHaveProperty('tag');
      expect(titleMapping).toHaveProperty('className');
      expect(titleMapping.tag).toBe('h1');
      expect(titleMapping.className).toBe('document-title');
    });
  });
});