/**
 * LaTeX CSS 样式规则配置
 * 定义各种LaTeX元素的样式
 */

const LATEX_CSS_RULES = {
  // 文档容器
  'latex-document': {
    maxWidth: '8.5in',
    margin: '0 auto',
    padding: '1in',
    fontFamily: '"Times New Roman", "SimSun", serif',
    fontSize: '12pt',
    lineHeight: '1.6',
    color: '#333',
    background: 'white'
  },

  // 标题样式
  'document-title': {
    textAlign: 'center',
    fontSize: '18pt',
    fontWeight: 'bold',
    margin: '1.5em 0 1em 0',
    color: '#000'
  },

  'document-author': {
    textAlign: 'center',
    fontSize: '14pt',
    margin: '0.5em 0',
    color: '#333'
  },

  'document-date': {
    textAlign: 'center',
    fontSize: '12pt',
    margin: '0.5em 0 1.5em 0',
    color: '#666'
  },

  'title-separator': {
    width: '100%',
    height: '1px',
    background: '#ccc',
    margin: '1.5em 0',
    border: 'none'
  },

  // 章节标题
  'latex-section': {
    fontSize: '16pt',
    fontWeight: 'bold',
    margin: '2em 0 1em 0',
    color: '#000'
  },

  'latex-subsection': {
    fontSize: '14pt',
    fontWeight: 'bold',
    margin: '1.5em 0 0.8em 0',
    color: '#000'
  },

  'latex-subsubsection': {
    fontSize: '12pt',
    fontWeight: 'bold',
    margin: '1.2em 0 0.6em 0',
    color: '#000'
  },

  // 内容样式
  'latex-content': {
    margin: '0.8em 0',
    textAlign: 'justify',
    textIndent: '2em'
  }
};

// CSS 生成工具函数
const CSS_UTILS = {
  /**
   * 将样式对象转换为CSS字符串
   */
  objectToCss(className, styleObject) {
    const cssProperties = Object.entries(styleObject)
      .map(([property, value]) => {
        // 将驼峰命名转换为短横线命名
        const cssProperty = property.replace(/[A-Z]/g, match => `-${match.toLowerCase()}`);
        return `  ${cssProperty}: ${value};`;
      })
      .join('\n');
    
    return `.${className} {\n${cssProperties}\n}`;
  },

  /**
   * 生成完整的LaTeX CSS
   */
  generateFullCSS() {
    const cssBlocks = Object.entries(LATEX_CSS_RULES)
      .map(([className, styles]) => this.objectToCss(className, styles));
    
    return `/* LaTeX Parser V3 - 生成的样式 */\n\n${cssBlocks.join('\n\n')}`;
  }
};

// 导出
if (typeof module !== 'undefined' && module.exports) {
  module.exports = { LATEX_CSS_RULES, CSS_UTILS };
} else if (typeof window !== 'undefined') {
  window.LATEX_CSS_RULES = LATEX_CSS_RULES;
  window.CSS_UTILS = CSS_UTILS;
}