/**
 * LaTeX Parser v3 - 浏览器入口文件
 * 
 * 用于浏览器环境的统一入口
 * 确保所有依赖正确加载并提供 LaTeXParser 类
 */

(function(global) {
  'use strict';

  // 检查是否在浏览器环境
  if (typeof window === 'undefined') {
    console.warn('[LaTeXParser] browser.js 应该只在浏览器环境中使用');
    return;
  }

  // 检查必要的依赖解析器
  const requiredParsers = [
    'DocumentParser',
    'BlockElementParser', 
    'InlineElementParser',
    'MathParser',
    'SpecialParser'
  ];
  
  // 检查缺失的解析器
  const missingParsers = requiredParsers.filter(parser => !global[parser]);
  
  if (missingParsers.length > 0) {
    console.error('[LaTeXParser] 缺少必要的解析器:', missingParsers);
    console.error('[LaTeXParser] 请确保在加载 browser.js 之前按顺序加载了以下文件:');
    console.error('  1. core/DocumentParser.js');
    console.error('  2. core/BlockElementParser.js');
    console.error('  3. core/InlineElementParser.js');
    console.error('  4. core/MathParser.js');
    console.error('  5. core/SpecialParser.js');
    console.error('  6. core/LaTeXParser.js');
    throw new Error('LaTeX Parser V3: 依赖加载不完整');
  }

  // 检查主解析器类
  if (!global.LaTeXParser) {
    // 如果 LaTeXParser 还没有被 core/LaTeXParser.js 导出，尝试创建
    if (typeof global.LaTeXParserCore === 'function') {
      global.LaTeXParser = global.LaTeXParserCore;
    } else {
      console.error('[LaTeXParser] LaTeXParser 类未找到');
      console.error('[LaTeXParser] 请确保 core/LaTeXParser.js 已正确加载');
      throw new Error('LaTeX Parser V3: LaTeXParser 类未定义');
    }
  }

  // 提供便捷的工厂函数
  global.createLaTeXParser = function(config) {
    return new global.LaTeXParser(config);
  };

  // 提供版本信息
  global.LaTeXParserVersion = '3.0.0';

  // 提供快捷解析函数
  global.parseLatex = function(content, config) {
    const parser = new global.LaTeXParser(config);
    return parser.parseComplete(content);
  };
})(window);