/**
 * LaTeX解析器稳定接口 v3.0
 * 专为Markdown处理流程设计的V3专用接口
 * 移除所有V2兼容代码，专注于V3五层架构
 * 
 * 设计原则：
 * 1. 接口稳定 - 向后兼容，版本控制严格
 * 2. 错误隔离 - 异常不会影响主流程
 * 3. 性能可预测 - 有超时机制和性能保障
 * 4. 状态无关 - 纯函数设计，无副作用
 * 5. 可插拔 - 可以安全启用/禁用
 */

(function(global) {
  'use strict';

  // 接口版本定义 - 升级到V3
  const INTERFACE_VERSION = '3.0.0';
  const COMPATIBLE_VERSIONS = ['3.0.0']; // 只支持V3版本

  /**
   * LaTeX V3 稳定接口
   * 基于 LaTeX Parser V3 TDD五层架构
   */
  class LaTeXStableInterface {
    constructor() {
      this.version = INTERFACE_VERSION;
      this.isInitialized = false;
      this.v3Parser = null;
      this.config = this.getDefaultConfig();
      this.lastError = null;
    }

    /**
     * 获取默认配置
     */
    getDefaultConfig() {
      return Object.freeze({
        // 基础配置
        enabled: true,
        timeout: 5000,                    // 5秒超时
        maxInputSize: 1024 * 1024,       // 1MB最大输入
        
        // V3特定配置
        mode: 'tolerant',
        enabledLayers: ['document', 'block', 'inline', 'math', 'special'],
        maxRecursionDepth: 100,
        
        // 错误处理
        throwOnError: false,              // 静默失败，不中断主流程
        fallbackToPlainText: true,        // 失败时降级为纯文本
        
        // 性能控制
        enableCache: true,
        maxCacheSize: 100,
        
        // 调试
        debugMode: false,
        logLevel: 'warn'                 // error | warn | info | debug
      });
    }

    /**
     * 初始化V3接口
     */
    async initialize(options = {}) {
      try {
        if (this.isInitialized) {
          return true; // 幂等性：已初始化直接返回成功
        }

        // 合并配置
        const mergedConfig = {
          ...this.config,
          ...options
        };
        
        // 验证并清理配置
        const cleanedConfig = this.cleanConfig(mergedConfig);
        this.config = Object.freeze(cleanedConfig);

        // 验证配置有效性
        if (!this.validateConfig(this.config)) {
          throw new Error('Invalid V3 configuration provided');
        }

        // 检查V3解析器可用性
        if (!global.LaTeXParser) {
          throw new Error('LaTeX Parser V3 not available. Please ensure it is loaded.');
        }

        this.isInitialized = true;
        this.log('info', `LaTeX V3 Interface v${this.version} initialized successfully`);
        return true;

      } catch (error) {
        this.lastError = error;
        this.log('error', `Failed to initialize LaTeX V3 interface: ${error.message}`);
        
        if (this.config.throwOnError) {
          throw error;
        }
        return false;
      }
    }

    /**
     * 核心处理接口 - 使用V3五层架构
     */
    async process(markdownText, context = {}) {
      // 输入验证
      const validationResult = this.validateInput(markdownText);
      if (!validationResult.valid) {
        return this.createErrorResult(validationResult.error, markdownText);
      }

      // 检查是否已启用
      if (!this.config.enabled) {
        return this.createPassthroughResult(markdownText);
      }

      // 检查是否已初始化
      if (!this.isInitialized) {
        this.log('warn', 'LaTeX V3 interface not initialized, initializing now...');
        const initResult = await this.initialize();
        if (!initResult) {
          return this.createErrorResult('V3 initialization failed', markdownText);
        }
      }

      // 检查V3解析器
      if (!global.LaTeXParser) {
        return this.createErrorResult('LaTeX Parser V3 not available', markdownText);
      }

      // 执行V3处理（带超时和错误隔离）
      return this.executeV3Processing(markdownText, context);
    }

    /**
     * V3处理执行
     */
    async executeV3Processing(markdownText, context) {
      return new Promise((resolve) => {
        // 设置超时保护
        const timeoutId = setTimeout(() => {
          this.log('warn', `LaTeX V3 processing timeout after ${this.config.timeout}ms`);
          resolve(this.createTimeoutResult(markdownText));
        }, this.config.timeout);

        try {
          // 创建V3解析器实例
          const parser = new global.LaTeXParser({
            mode: this.config.mode,
            enabledLayers: this.config.enabledLayers,
            maxRecursionDepth: this.config.maxRecursionDepth
          });

          // 执行V3解析
          const result = parser.parseComplete(markdownText);
          
          clearTimeout(timeoutId);
          resolve(this.normalizeV3Result(result, markdownText));

        } catch (error) {
          clearTimeout(timeoutId);
          resolve(this.handleV3ProcessingError(error, markdownText));
        }
      });
    }

    /**
     * 标准化V3处理结果
     */
    normalizeV3Result(v3Result, originalText) {
      if (!v3Result || typeof v3Result !== 'object') {
        return this.createErrorResult('Invalid V3 processing result', originalText);
      }

      return {
        success: v3Result.success !== false,
        processedText: v3Result.processedText || originalText,
        blocks: Array.isArray(v3Result.blocks) ? v3Result.blocks : [],
        metadata: {
          version: this.version,
          architecture: 'five-layer-tdd',
          timestamp: Date.now(),
          inputLength: originalText.length,
          outputLength: (v3Result.processedText || originalText).length,
          blockCount: Array.isArray(v3Result.blocks) ? v3Result.blocks.length : 0,
          processingTime: v3Result.metadata?.processingTime || 0,
          nodeCount: v3Result.metadata?.nodeCount || 0,
          complexityScore: v3Result.metadata?.complexityScore || 0,
          enabledLayers: this.config.enabledLayers
        },
        warnings: v3Result.warnings || [],
        errors: v3Result.errors || []
      };
    }

    /**
     * 输入验证
     */
    validateInput(markdownText) {
      if (typeof markdownText !== 'string') {
        return { valid: false, error: 'Input must be a string' };
      }

      if (markdownText.length > this.config.maxInputSize) {
        return { valid: false, error: 'Input exceeds maximum size limit' };
      }

      return { valid: true };
    }

    /**
     * 配置清理
     */
    cleanConfig(config) {
      const defaultConfig = this.getDefaultConfig();
      const cleanedConfig = {};
      
      // 清理每个配置项
      cleanedConfig.enabled = typeof config.enabled === 'boolean' ? config.enabled : defaultConfig.enabled;
      cleanedConfig.timeout = (typeof config.timeout === 'number' && config.timeout > 0) ? config.timeout : defaultConfig.timeout;
      cleanedConfig.maxInputSize = (typeof config.maxInputSize === 'number' && config.maxInputSize > 0) ? config.maxInputSize : defaultConfig.maxInputSize;
      cleanedConfig.throwOnError = typeof config.throwOnError === 'boolean' ? config.throwOnError : defaultConfig.throwOnError;
      cleanedConfig.fallbackToPlainText = typeof config.fallbackToPlainText === 'boolean' ? config.fallbackToPlainText : defaultConfig.fallbackToPlainText;
      cleanedConfig.enableCache = typeof config.enableCache === 'boolean' ? config.enableCache : defaultConfig.enableCache;
      cleanedConfig.maxCacheSize = (typeof config.maxCacheSize === 'number' && config.maxCacheSize > 0) ? config.maxCacheSize : defaultConfig.maxCacheSize;
      cleanedConfig.debugMode = typeof config.debugMode === 'boolean' ? config.debugMode : defaultConfig.debugMode;
      cleanedConfig.logLevel = ['error', 'warn', 'info', 'debug'].includes(config.logLevel) ? config.logLevel : defaultConfig.logLevel;
      
      // V3特定配置
      cleanedConfig.mode = ['strict', 'tolerant'].includes(config.mode) ? config.mode : defaultConfig.mode;
      cleanedConfig.enabledLayers = Array.isArray(config.enabledLayers) ? config.enabledLayers : defaultConfig.enabledLayers;
      cleanedConfig.maxRecursionDepth = (typeof config.maxRecursionDepth === 'number' && config.maxRecursionDepth > 0) ? config.maxRecursionDepth : defaultConfig.maxRecursionDepth;
      
      return cleanedConfig;
    }

    /**
     * 配置验证
     */
    validateConfig(config) {
      if (!config || typeof config !== 'object') return false;
      if (typeof config.enabled !== 'boolean') return false;
      if (typeof config.timeout !== 'number' || config.timeout <= 0) return false;
      if (typeof config.maxInputSize !== 'number' || config.maxInputSize <= 0) return false;
      
      // V3特定验证
      if (!['strict', 'tolerant'].includes(config.mode)) return false;
      if (!Array.isArray(config.enabledLayers)) return false;
      if (typeof config.maxRecursionDepth !== 'number' || config.maxRecursionDepth <= 0) return false;
      
      return true;
    }

    /**
     * 创建错误结果
     */
    createErrorResult(error, originalText) {
      const errorMessage = typeof error === 'string' ? error : (error?.message || 'Unknown error');
      const safeOriginalText = originalText || '';
      this.log('error', `LaTeX V3 processing error: ${errorMessage}`);

      return {
        success: false,
        processedText: this.config.fallbackToPlainText ? safeOriginalText : '',
        blocks: [],
        metadata: {
          version: this.version,
          architecture: 'five-layer-tdd-error',
          timestamp: Date.now(),
          inputLength: safeOriginalText.length,
          outputLength: safeOriginalText.length,
          blockCount: 0
        },
        warnings: ['LaTeX V3 processing failed, using fallback'],
        errors: [errorMessage]
      };
    }

    /**
     * 创建透传结果
     */
    createPassthroughResult(originalText) {
      const safeOriginalText = originalText || '';
      return {
        success: true,
        processedText: safeOriginalText,
        blocks: [],
        metadata: {
          version: this.version,
          architecture: 'five-layer-tdd-passthrough',
          timestamp: Date.now(),
          inputLength: safeOriginalText.length,
          outputLength: safeOriginalText.length,
          blockCount: 0,
          processingMethod: 'passthrough'
        },
        warnings: ['LaTeX V3 processing disabled'],
        errors: []
      };
    }

    /**
     * 创建超时结果
     */
    createTimeoutResult(originalText) {
      const safeOriginalText = originalText || '';
      return {
        success: false,
        processedText: this.config.fallbackToPlainText ? safeOriginalText : '',
        blocks: [],
        metadata: {
          version: this.version,
          architecture: 'five-layer-tdd-timeout',
          timestamp: Date.now(),
          inputLength: safeOriginalText.length,
          outputLength: safeOriginalText.length,
          blockCount: 0,
          processingMethod: 'timeout-fallback'
        },
        warnings: [`LaTeX V3 processing timeout after ${this.config.timeout}ms`],
        errors: ['Processing timeout']
      };
    }

    /**
     * 处理V3处理错误
     */
    handleV3ProcessingError(error, originalText) {
      this.lastError = error;
      
      if (this.config.throwOnError) {
        throw error;
      }

      return this.createErrorResult(error, originalText);
    }

    /**
     * 日志记录
     */
    log(level, message) {
      const levels = { error: 0, warn: 1, info: 2, debug: 3 };
      const configLevel = levels[this.config.logLevel] || 1;
      const messageLevel = levels[level] || 1;

      if (messageLevel <= configLevel) {
        const timestamp = new Date().toISOString();
        const prefix = `[LaTeXV3Interface-${this.version}]`;
        console[level] || console.log(`${timestamp} ${prefix} ${message}`);
      }
    }

    // =====================================================
    // 公共接口方法 - V3专用
    // =====================================================

    /**
     * 获取V3接口版本信息
     */
    getVersion() {
      return {
        version: this.version,
        architecture: 'five-layer-tdd',
        compatibleVersions: COMPATIBLE_VERSIONS,
        isInitialized: this.isInitialized,
        v3Available: !!global.LaTeXParser
      };
    }

    /**
     * 检查V3接口健康状况
     */
    healthCheck() {
      return {
        healthy: this.isInitialized && !this.lastError && !!global.LaTeXParser,
        version: this.version,
        architecture: 'five-layer-tdd',
        initialized: this.isInitialized,
        v3Available: !!global.LaTeXParser,
        lastError: this.lastError ? this.lastError.message : null,
        enabledLayers: this.config.enabledLayers,
        mode: this.config.mode,
        timestamp: Date.now()
      };
    }

    /**
     * 获取当前V3配置
     */
    getConfig() {
      return { ...this.config }; // 返回配置副本
    }

    /**
     * 安全重新配置V3
     */
    reconfigure(newConfig) {
      const mergedConfig = { ...this.config, ...newConfig };
      const cleanedConfig = this.cleanConfig(mergedConfig);
      
      if (this.validateConfig(cleanedConfig)) {
        this.config = Object.freeze(cleanedConfig);
        this.log('info', 'V3 Configuration updated successfully');
        return true;
      } else {
        this.log('error', 'Invalid V3 configuration provided');
        return false;
      }
    }

    /**
     * 禁用V3接口
     */
    disable() {
      this.config = Object.freeze({ ...this.config, enabled: false });
      this.log('info', 'LaTeX V3 interface disabled');
    }

    /**
     * 启用V3接口
     */
    enable() {
      this.config = Object.freeze({ ...this.config, enabled: true });
      this.log('info', 'LaTeX V3 interface enabled');
    }
  }

  /**
   * 全局V3单例实例
   */
  let globalV3Interface = null;

  /**
   * 获取全局LaTeX V3接口实例
   */
  function getLatexInterface(options = {}) {
    if (!globalV3Interface) {
      globalV3Interface = new LaTeXStableInterface();
      
      // 自动初始化V3接口
      globalV3Interface.initialize(options).catch(error => {
        console.warn('[LaTeXV3Interface] Auto-initialization failed:', error.message);
      });
    }
    return globalV3Interface;
  }

  /**
   * 重置全局V3接口（主要用于测试）
   */
  function resetLatexInterface() {
    globalV3Interface = null;
  }

  // 导出到全局作用域
  global.LaTeXStableInterface = LaTeXStableInterface;
  global.getLatexInterface = getLatexInterface;
  global.resetLatexInterface = resetLatexInterface;

  // Node.js 模块导出
  if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
      LaTeXStableInterface,
      getLatexInterface,
      resetLatexInterface,
      INTERFACE_VERSION
    };
  }


})(typeof window !== 'undefined' ? window : global);