// i18n本地化功能

// 存储所有语言的翻译数据
let translationsCache = {};

// 加载指定语言的翻译
async function loadTranslations(language) {
  if (translationsCache[language]) {
    return translationsCache[language];
  }
  
  try {
    let url;
    if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.getURL) {
      url = chrome.runtime.getURL(`_locales/${language}/messages.json`);
    } else {
      // 非扩展环境，使用相对路径
      url = `../_locales/${language}/messages.json`;
    }
    
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`Failed to load language: ${language}`);
    }
    const messages = await response.json();
    translationsCache[language] = messages;
    return messages;
  } catch (error) {
    console.error('Error loading translations:', error);
    // 如果加载失败，尝试加载英文
    if (language !== 'en') {
      return loadTranslations('en');
    }
    return {};
  }
}

// 获取翻译文本
function getTranslation(messages, key) {
  return messages[key]?.message || key;
}

// 初始化页面本地化
async function initializeI18n() {
  // 获取当前语言
  const currentLang = await getCurrentLanguage();
  
  // 加载翻译
  const messages = await loadTranslations(currentLang);
  
  // 获取所有需要翻译的元素
  const elements = document.querySelectorAll('[data-i18n]');
  elements.forEach(element => {
    const messageKey = element.getAttribute('data-i18n');
    const message = getTranslation(messages, messageKey);
    if (message) {
      element.textContent = message;
    }
  });
  
  // 处理占位符文本
  const placeholderElements = document.querySelectorAll('[data-i18n-placeholder]');
  placeholderElements.forEach(element => {
    const messageKey = element.getAttribute('data-i18n-placeholder');
    const message = getTranslation(messages, messageKey);
    if (message) {
      element.placeholder = message;
    }
  });
  
  // 处理标题属性
  const titleElements = document.querySelectorAll('[data-i18n-title]');
  titleElements.forEach(element => {
    const messageKey = element.getAttribute('data-i18n-title');
    const message = getTranslation(messages, messageKey);
    if (message) {
      element.title = message;
    }
  });
  
  // 设置页面方向（RTL支持）
  setPageDirection(currentLang);
  
  // 更新页面标题
  if (messages.popupTitle) {
    document.title = messages.popupTitle.message;
  }
}

// 获取当前语言设置
function getCurrentLanguage() {
  return new Promise((resolve) => {
    // 检查是否在扩展环境中
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
      chrome.storage.sync.get(['language'], (result) => {
        if (result.language && result.language !== 'auto') {
          resolve(result.language);
        } else {
          // 自动检测浏览器语言
          const browserLang = chrome.i18n ? chrome.i18n.getUILanguage() : navigator.language;
          resolve(mapBrowserLanguage(browserLang));
        }
      });
    } else {
      // 非扩展环境，使用浏览器语言或localStorage
      try {
        const storedLang = localStorage.getItem('language');
        if (storedLang && storedLang !== 'auto') {
          resolve(storedLang);
        } else {
          const browserLang = navigator.language || navigator.userLanguage || 'en';
          resolve(mapBrowserLanguage(browserLang));
        }
      } catch (e) {
        console.error('Failed to read language from localStorage:', e);
        const browserLang = navigator.language || navigator.userLanguage || 'en';
        resolve(mapBrowserLanguage(browserLang));
      }
    }
  });
}

// 将浏览器语言映射到支持的语言
function mapBrowserLanguage(browserLang) {
  const langMap = {
    'zh': 'zh_CN',
    'zh-CN': 'zh_CN',
    'zh-TW': 'zh_CN',
    'es': 'es',
    'es-ES': 'es',
    'es-MX': 'es',
    'fr': 'fr',
    'fr-FR': 'fr',
    'de': 'de',
    'de-DE': 'de',
    'ja': 'ja',
    'ja-JP': 'ja',
    'ko': 'ko',
    'ko-KR': 'ko',
    'ru': 'ru',
    'ru-RU': 'ru',
    'pt': 'pt',
    'pt-BR': 'pt',
    'pt-PT': 'pt',
    'it': 'it',
    'it-IT': 'it',
    'ar': 'ar',
    'ar-SA': 'ar',
    'hi': 'hi',
    'hi-IN': 'hi'
  };
  
  // 尝试精确匹配
  if (langMap[browserLang]) {
    return langMap[browserLang];
  }
  
  // 尝试匹配语言代码前缀
  const langPrefix = browserLang.split('-')[0];
  if (langMap[langPrefix]) {
    return langMap[langPrefix];
  }
  
  // 默认使用英语
  return 'en';
}

// 设置语言
function setLanguage(language) {
  return new Promise((resolve) => {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
      chrome.storage.sync.set({ language: language }, () => {
        resolve();
      });
    } else {
      // 非扩展环境，使用localStorage
      try {
        localStorage.setItem('language', language);
        resolve();
      } catch (e) {
        console.error('Failed to save language to localStorage:', e);
        resolve(); // 继续执行，不要阻塞
      }
    }
  });
}

// 重新加载扩展以应用新语言
function reloadExtension() {
  if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.reload) {
    chrome.runtime.reload();
  } else {
    // 非扩展环境，重新加载页面
    window.location.reload();
  }
}

// 设置页面方向（RTL或LTR）
function setPageDirection(language) {
  const rtlLanguages = ['ar', 'he', 'fa', 'ur'];
  const isRTL = rtlLanguages.includes(language);
  
  // 设置HTML元素的dir属性
  document.documentElement.setAttribute('dir', isRTL ? 'rtl' : 'ltr');
  
  // 如果是RTL语言，还需要调整一些特殊的样式
  if (isRTL) {
    document.documentElement.classList.add('rtl-lang');
  } else {
    document.documentElement.classList.remove('rtl-lang');
  }
}

// 导出函数供其他模块使用
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    initializeI18n,
    getCurrentLanguage,
    setLanguage,
    reloadExtension,
    setPageDirection,
    loadTranslations,
    getTranslation
  };
}