/**
 * 隐藏iframe渲染器 - 专注核心功能
 * 1. 创建唯一隐藏iframe打开renderer.html
 * 2. 处理PDF/Word转换和进度消息
 */
class HiddenIframeRender {
  static instance = null;

  constructor() {
    if (HiddenIframeRender.instance) {
      return HiddenIframeRender.instance;
    }

    this.iframe = null;
    this.isReady = false;
    this.pendingRequests = new Map();
    this.eventListeners = new Map();

    HiddenIframeRender.instance = this;
  }

  static getInstance() {
    if (!this.instance) {
      this.instance = new HiddenIframeRender();
    }
    return this.instance;
  }

  async initialize() {
    if (this.isReady) return;
    if (this.initPromise) return this.initPromise;

    this.initPromise = this._doInitialize();
    await this.initPromise;
    this.isReady = true;
  }

  async _doInitialize() {

    // 清理调试状态（localStorage在浏览器环境中总是可用）
    localStorage.removeItem('md2pdf_debug_mode');
    localStorage.removeItem('md2pdf_debug_iframe');

    // 移除旧iframe
    if (this.iframe) {
      this.iframe.remove();
      this.iframe = null;
    }

    // 设置消息监听（在创建iframe之前）
    this.setupMessageListener();

    // 创建新iframe
    this.iframe = document.createElement('iframe');
    this.iframe.id = 'md2pdf-hidden-renderer';

    // 调试模式：显示iframe
    const debugMode = localStorage.getItem('md2pdf_debug_iframe') === 'true';

    if (debugMode) {
      // 显示iframe用于调试
      this.iframe.style.cssText = `
        position: fixed !important;
        right: 10px !important;
        top: 10px !important;
        width: 800px !important;
        height: 600px !important;
        visibility: visible !important;
        opacity: 1 !important;
        pointer-events: auto !important;
        z-index: 9999 !important;
        display: block !important;
        border: 2px solid red !important;
        background: white !important;
        box-shadow: 0 4px 6px rgba(0,0,0,0.1) !important;
      `;
    } else {
      // 正常模式：隐藏iframe
      this.iframe.style.cssText = `
        position: fixed !important;
        left: -9999px !important;
        top: -9999px !important;
        width: 1px !important;
        height: 1px !important;
        visibility: hidden !important;
        opacity: 0 !important;
        pointer-events: none !important;
        z-index: -1 !important;
      `;
    }

    // 添加iframe错误监听
    this.iframe.onerror = (error) => {
      console.error('[HiddenIframeRender] iframe加载失败:', error);
    };

    // 添加到页面（在设置src之前）
    document.body.appendChild(this.iframe);

    // 设置iframe源
    const rendererUrl = chrome.runtime?.getURL
      ? chrome.runtime.getURL('renderer/renderer.html')
      : '../renderer/renderer.html';

    this.iframe.src = rendererUrl;

    // 等待iframe加载，然后请求就绪状态
    return new Promise((resolve, reject) => {
      const cleanup = () => clearTimeout(timeout);
      const timeout = setTimeout(() => {
        console.error('[HiddenIframeRender] 初始化超时');
        reject(new Error('Renderer initialization timeout'));
      }, 15000); // 15秒超时

      this.once('ready', () => {
        cleanup();
        resolve();
      });

      // iframe加载完成后，请求就绪状态
      this.iframe.addEventListener('load', () => {
        setTimeout(() => {
          this.sendMessage('requestReady').catch(err => {
            // console.log('[HiddenIframeRender] requestReady发送失败，这是正常的:', err.message);
          });
        }, 1000);
      });
    });
  }

  setupMessageListener() {
    if (this.messageHandler) return; // 避免重复监听

    this.messageHandler = (event) => {
      // 过滤无效消息
      if (!event.data || typeof event.data !== 'object' || event.data.source !== 'md2pdf-renderer') {
        return;
      }

      const { action, requestId, data, error } = event.data;

      // 处理ready消息 - 根据README.md规范
      if (action === 'rendererReady') {
        this.emit('ready');
        return;
      }

      // 处理各种消息类型
      switch (action) {
        case 'nextStep':
          // 进度消息（仅在调试模式记录）
          if (localStorage.getItem('md2pdf_debug_iframe') === 'true') {
            // console.log(`[HiddenIframeRender] 进度更新: ${data.message}`);
          }
          this.emit('progress', data);
          break;

        case 'pdfComplete':
        case 'wordComplete':
          // console.log(`[HiddenIframeRender] ${action}: ${data.filename}`);
          this.emit(action, data);
          this._resolveRequest(requestId, data);
          break;

        case 'pdfError':
        case 'wordError':
          console.error(`[HiddenIframeRender] ${action}: ${data.error}`);
          this.emit(action, data);
          this._rejectRequest(requestId, data.error);
          break;

        default:
          // 处理其他请求响应
          if (requestId) {
            if (error) {
              this._rejectRequest(requestId, error);
            } else {
              this._resolveRequest(requestId, data);
            }
          }
      }
    };

    window.addEventListener('message', this.messageHandler);
  }

  sendMessage(action, data = {}) {
    return new Promise((resolve, reject) => {
      const requestId = `${action}-${Date.now()}`;

      this.pendingRequests.set(requestId, { resolve, reject });

      // 发送消息
      const message = {
        source: 'md2pdf-iframe-manager',
        action: action,
        requestId: requestId
      };

      if (action !== 'requestReady') {
        message.data = data;
      }

      this.iframe.contentWindow.postMessage(message, '*');

      // 超时处理
      setTimeout(() => {
        if (this.pendingRequests.has(requestId)) {
          this.pendingRequests.delete(requestId);
          reject(new Error(`Request timeout: ${action}`));
        }
      }, 300000); // 5分钟超时
    });
  }

  // 辅助方法：解析请求
  _resolveRequest(requestId, data) {
    if (!requestId) return;
    const request = this.pendingRequests.get(requestId);
    if (request) {
      this.pendingRequests.delete(requestId);
      request.resolve(data);
    }
  }

  // 辅助方法：拒绝请求
  _rejectRequest(requestId, error) {
    if (!requestId) return;
    const request = this.pendingRequests.get(requestId);
    if (request) {
      this.pendingRequests.delete(requestId);
      request.reject(new Error(error));
    }
  }

  // PDF转换
  async convertToPDF(markdown, filename, options = {}) {
    if (!this.isReady) {
      await this.initialize();
    }

    // 构建PDF转换数据 - 根据README.md规范
    const data = {
      markdown: markdown,
      filename: filename.endsWith('.pdf') ? filename : `${filename}.pdf`,
      options: {
        pageSize: options.pageSize || 'a4',
        orientation: options.orientation || 'portrait',
        scale: options.scale || 2,
        backgroundColor: options.backgroundColor || '#ffffff',
        paginate: options.paginate !== false, // 默认启用分页
        ...options
      }
    };

    // 发送转换请求并等待完成
    const result = await this.sendMessage('generatePDF', data);

    // 如果返回的是blob，直接下载
    if (result instanceof Blob) {
      this.downloadBlob(result, data.filename);
    }

    return { success: true, filename: data.filename };
  }

  // Word转换
  async convertToWord(markdown, filename, options = {}) {
    if (!this.isReady) {
      await this.initialize();
    }

    // 构建Word转换数据 - 根据README.md规范
    const data = {
      markdown: markdown,
      filename: filename.endsWith('.docx') ? filename : `${filename}.docx`,
      options: {
        pageSize: options.pageSize || 'A4',
        orientation: options.orientation || 'portrait',
        ...options
      }
    };

    // 发送转换请求并等待完成
    const result = await this.sendMessage('generateWord', data);

    // 如果返回的是blob，直接下载
    if (result instanceof Blob) {
      this.downloadBlob(result, data.filename);
    }

    return { success: true, filename: data.filename };
  }

  // 兼容旧API
  async convertMarkdownToPDF(markdown, filename, progressCallback, options = {}) {
    if (progressCallback) {
      // 临时监听进度事件
      const progressHandler = (data) => {
        progressCallback(`处理中: ${data.message || '转换PDF'}`);
      };
      this.on('progress', progressHandler);

      try {
        const result = await this.convertToPDF(markdown, filename, options);
        this.off('progress', progressHandler);
        return result;
      } catch (error) {
        this.off('progress', progressHandler);
        throw error;
      }
    }
    return this.convertToPDF(markdown, filename, options);
  }

  async convertMarkdownToWord(markdown, filename, progressCallback, options = {}) {
    if (progressCallback) {
      // 临时监听进度事件
      const progressHandler = (data) => {
        progressCallback(`处理中: ${data.message || '转换Word'}`);
      };
      this.on('progress', progressHandler);

      try {
        const result = await this.convertToWord(markdown, filename, options);
        this.off('progress', progressHandler);
        return result;
      } catch (error) {
        this.off('progress', progressHandler);
        throw error;
      }
    }
    return this.convertToWord(markdown, filename, options);
  }

  downloadBlob(blob, filename) {
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  // 事件系统
  on(event, listener) {
    if (!this.eventListeners.has(event)) {
      this.eventListeners.set(event, []);
    }
    this.eventListeners.get(event).push(listener);
  }


  once(event, listener) {
    const onceWrapper = (...args) => {
      this.off(event, onceWrapper);
      listener(...args);
    };
    this.on(event, onceWrapper);
  }

  off(event, listener) {
    if (!this.eventListeners.has(event)) return;
    const listeners = this.eventListeners.get(event);
    const index = listeners.indexOf(listener);
    if (index > -1) {
      listeners.splice(index, 1);
    }
  }

  emit(event, ...args) {
    if (!this.eventListeners.has(event)) return;
    const listeners = this.eventListeners.get(event).slice();
    listeners.forEach(listener => {
      try {
        listener(...args);
      } catch (error) {
        console.error(`Event listener error for ${event}:`, error);
      }
    });
  }

  // 清理方法
  destroy() {
    // 清理事件监听器
    if (this.messageHandler) {
      window.removeEventListener('message', this.messageHandler);
      this.messageHandler = null;
    }

    // 清理iframe
    if (this.iframe) {
      this.iframe.remove();
      this.iframe = null;
    }

    // 清理调试标题栏（使用精确选择器）
    const debugTitleBar = document.querySelector('div[style*="position: fixed"][style*="z-index: 10000"]');
    if (debugTitleBar && debugTitleBar.textContent === '[DEBUG] MD2PDF Renderer iframe') {
      debugTitleBar.remove();
    }

    // 清理状态
    this.isReady = false;
    this.pendingRequests.clear();
    this.eventListeners.clear();

    // 清理单例
    HiddenIframeRender.instance = null;
  }
}

// 导出全局实例
window.HiddenIframeRender = HiddenIframeRender;

// 创建全局实例
const globalInstance = HiddenIframeRender.getInstance();

// 兼容旧接口 - 统一指向同一实例
window.DocumentGenerator = HiddenIframeRender;
window.IframeRendererManager = HiddenIframeRender;
window.hiddenIframeRender = globalInstance;
window.documentGenerator = globalInstance;