/**
 * 增强型滚动同步系统 - 分层降级架构
 * 优先锚点、次之百分比，确保在任何情况下都能实现同步
 * 
 * 同步策略优先级：
 * 1. 精确锚点同步 (Smart Anchor Sync)
 * 2. 简化锚点同步 (Simple Anchor Sync)  
 * 3. 滚动百分比同步 (Percentage Sync)
 * 4. 行号比例同步 (Line Ratio Sync)
 */

class EnhancedScrollSync {
  constructor() {
    this.config = {
      channelId: 'md2pdf-enhanced-scroll-sync',
      debounceDelay: 100,
      maxRetries: 3,
      fallbackEnabled: true,
      debug: localStorage.getItem('md2pdf_sync_debug') === 'true'
    };
    
    // 同步状态管理
    this.syncState = {
      lastSyncTime: 0,
      currentStrategy: null,
      retryCount: 0,
      isScrolling: false,
      scrollSource: null
    };
    
    // 容器和提取器
    this.container = null;
    this.isEditor = false;
    this.textExtractor = null;
    
    // 锚点缓存
    this.anchors = {
      smart: new Map(),      // 精确锚点（随机片段）
      simple: new Map(),     // 简单锚点（行首、标题等）
      fallback: new Map()    // 降级锚点（关键字）
    };
    
    this.initialize();
  }
  
  /**
   * 初始化系统
   */
  initialize() {
    this.identifyContainer();
    
    if (!this.container) {
      setTimeout(() => this.initialize(), 500);
      return;
    }
    
    this.setupListeners();
    this.buildAllAnchors();
  }
  
  /**
   * 识别容器类型
   */
  identifyContainer() {
    // 编辑器识别
    const textarea = document.querySelector('#markdown-input, textarea.markdown-editor');
    if (textarea) {
      this.container = textarea;
      this.isEditor = true;
      this.textExtractor = new EnhancedTextareaExtractor(textarea);
      return;
    }
    
    // 预览区域识别
    const renderContainer = document.querySelector('#render-container');
    if (renderContainer) {
      this.container = renderContainer;
      this.isEditor = false;
      this.textExtractor = new EnhancedDOMExtractor(renderContainer);
      return;
    }
    
  }
  
  /**
   * 设置事件监听器
   */
  setupListeners() {
    // 滚动事件监听
    const handleScroll = this.debounce(() => {
      if (this.syncState.scrollSource !== 'remote') {
        this.handleLocalScroll();
      }
    }, this.config.debounceDelay);
    
    if (this.container.tagName === 'TEXTAREA') {
      this.container.addEventListener('scroll', handleScroll, { passive: true });
    } else {
      window.addEventListener('scroll', handleScroll, { passive: true });
    }
    
    // 消息监听
    this.setupMessageListener();
    
    // 内容变化监听（重建锚点）
    if (this.isEditor) {
      this.container.addEventListener('input', this.debounce(() => {
        this.buildAllAnchors();
      }, 1000));
    }
    
  }
  
  /**
   * 构建所有层级的锚点
   */
  buildAllAnchors() {
    if (!this.isEditor) {
      return;
    }
    
    const startTime = performance.now();
    
    // 清空旧锚点
    this.anchors.smart.clear();
    this.anchors.simple.clear();
    this.anchors.fallback.clear();
    
    const lines = this.container.value.split('\n');
    
    // 第1层：精确锚点（优化的随机片段）
    this.buildSmartAnchors(lines);
    
    // 第2层：简单锚点（标题、行首等）
    this.buildSimpleAnchors(lines);
    
    // 第3层：降级锚点（关键词）
    this.buildFallbackAnchors(lines);
    
    const endTime = performance.now();
  }
  
  /**
   * 构建第1层：精确锚点（改进的随机片段算法）
   */
  buildSmartAnchors(lines) {
    lines.forEach((line, lineIndex) => {
      const cleanLine = line.trim();
      
      // 跳过无效行
      if (cleanLine.length < 8 || 
          cleanLine.startsWith('```') || 
          cleanLine.startsWith('#') ||  // 标题留给简单锚点
          cleanLine.includes('mermaid') || 
          cleanLine.includes('latex')) {
        return;
      }
      
      // 生成确定性片段（非随机）
      const fragments = this.generateDeterministicFragments(cleanLine, lineIndex);
      
      fragments.forEach(fragment => {
        this.anchors.smart.set(fragment.text, {
          line: lineIndex,
          type: 'smart',
          confidence: fragment.confidence
        });
      });
    });
    
  }
  
  /**
   * 构建第2层：简单锚点（标题、列表等）
   */
  buildSimpleAnchors(lines) {
    
    lines.forEach((line, lineIndex) => {
      const cleanLine = line.trim();
      
      // 标题锚点
      if (cleanLine.startsWith('#')) {
        const titleText = cleanLine.replace(/^#+\s*/, '').trim();
        if (titleText.length > 3) {
          this.anchors.simple.set(titleText, {
            line: lineIndex,
            type: 'title',
            confidence: 0.9
          });
        }
      }
      
      // 列表项锚点
      else if (cleanLine.match(/^[-*+]\s+/) || cleanLine.match(/^\d+\.\s+/)) {
        const listText = cleanLine.replace(/^[-*+\d.\s]+/, '').trim();
        if (listText.length > 5) {
          this.anchors.simple.set(listText.substring(0, 20), {
            line: lineIndex,
            type: 'list',
            confidence: 0.7
          });
        }
      }
      
      // 行首锚点（每10行取一个）
      else if (lineIndex % 10 === 0 && cleanLine.length > 10) {
        const lineStart = cleanLine.substring(0, 15).trim();
        this.anchors.simple.set(lineStart, {
          line: lineIndex,
          type: 'lineStart',
          confidence: 0.6
        });
      }
    });
    
  }
  
  /**
   * 构建第3层：降级锚点（关键词）
   */
  buildFallbackAnchors(lines) {
    
    const keywords = ['function', 'class', 'const', 'let', 'var', 'import', 'export', 
                     '函数', '方法', '接口', '实现', '配置', '设置', '测试', '示例'];
    
    lines.forEach((line, lineIndex) => {
      const cleanLine = line.trim().toLowerCase();
      
      keywords.forEach(keyword => {
        if (cleanLine.includes(keyword)) {
          const keywordAnchor = `${keyword}_line_${lineIndex}`;
          this.anchors.fallback.set(keywordAnchor, {
            line: lineIndex,
            type: 'keyword',
            confidence: 0.4,
            keyword: keyword
          });
        }
      });
    });
    
  }
  
  /**
   * 生成确定性文本片段（替代随机算法）
   */
  generateDeterministicFragments(line, lineIndex) {
    const fragments = [];
    const minLength = 8;
    const maxLength = 20;
    
    // 策略1：取行首片段
    if (line.length >= minLength) {
      const startFragment = line.substring(0, Math.min(maxLength, line.length)).trim();
      if (this.isValidFragment(startFragment)) {
        fragments.push({
          text: startFragment,
          confidence: 0.8
        });
      }
    }
    
    // 策略2：取行尾片段
    if (line.length >= minLength * 2) {
      const endFragment = line.substring(Math.max(0, line.length - maxLength)).trim();
      if (this.isValidFragment(endFragment)) {
        fragments.push({
          text: endFragment,
          confidence: 0.7
        });
      }
    }
    
    // 策略3：取中间片段
    if (line.length >= minLength * 3) {
      const midStart = Math.floor((line.length - maxLength) / 2);
      const midFragment = line.substring(midStart, midStart + maxLength).trim();
      if (this.isValidFragment(midFragment)) {
        fragments.push({
          text: midFragment,
          confidence: 0.6
        });
      }
    }
    
    return fragments;
  }
  
  /**
   * 验证片段有效性
   */
  isValidFragment(fragment) {
    return fragment.length >= 6 && 
           /[\u4e00-\u9fa5a-zA-Z]{4,}/.test(fragment) &&
           !fragment.includes('```') &&
           !fragment.includes('$');
  }
  
  /**
   * 处理本地滚动（发送端）- 双向同步
   */
  handleLocalScroll() {
    const sourceType = this.isEditor ? '编辑器' : '预览区域';
    
    this.syncState.scrollSource = 'local';
    this.syncState.lastSyncTime = Date.now();
    
    // 无论是编辑器还是预览区域，都尝试同步策略
    this.attemptSync();
    
    setTimeout(() => {
      this.syncState.scrollSource = null;
    }, 200);
  }
  
  /**
   * 尝试同步（分层策略）
   */
  attemptSync() {
    // 根据环境类型选择不同的策略优先级
    let strategies;
    
    if (this.isEditor) {
      // 编辑器：优先锚点，次之百分比
      strategies = [
        { name: 'smart', method: this.attemptSmartSync.bind(this) },
        { name: 'simple', method: this.attemptSimpleSync.bind(this) },
        { name: 'percentage', method: this.attemptPercentageSync.bind(this) },
        { name: 'lineRatio', method: this.attemptLineRatioSync.bind(this) }
      ];
    } else {
      // 预览区域：优先百分比，次之DOM锚点
      strategies = [
        { name: 'percentage', method: this.attemptPercentageSync.bind(this) },
        { name: 'domAnchor', method: this.attemptDomAnchorSync.bind(this) },
        { name: 'lineRatio', method: this.attemptLineRatioSync.bind(this) }
      ];
    }
    
    for (const strategy of strategies) {
      const result = strategy.method();
      if (result.success) {
        this.syncState.currentStrategy = strategy.name;
        this.broadcastSync(result.data, strategy.name);
        return;
      } else {
      }
    }
    
  }
  
  /**
   * 策略1：精确锚点同步
   */
  attemptSmartSync() {
    const visibleAnchors = this.getVisibleAnchors(this.anchors.smart);
    
    if (visibleAnchors.length === 0) {
      return { success: false, reason: '无可见精确锚点' };
    }
    
    return {
      success: true,
      data: {
        type: 'smart',
        anchors: visibleAnchors,
        viewport: this.textExtractor.getViewportInfo()
      }
    };
  }
  
  /**
   * 策略2：简单锚点同步
   */
  attemptSimpleSync() {
    const visibleAnchors = this.getVisibleAnchors(this.anchors.simple);
    
    if (visibleAnchors.length === 0) {
      return { success: false, reason: '无可见简单锚点' };
    }
    
    return {
      success: true,
      data: {
        type: 'simple',
        anchors: visibleAnchors,
        viewport: this.textExtractor.getViewportInfo()
      }
    };
  }
  
  /**
   * 策略3：滚动百分比同步（兜底方案）
   */
  attemptPercentageSync() {
    const scrollInfo = this.textExtractor.getScrollInfo();
    
    return {
      success: true,  // 百分比同步总是成功
      data: {
        type: 'percentage',
        scrollTop: scrollInfo.scrollTop,
        scrollHeight: scrollInfo.scrollHeight,
        clientHeight: scrollInfo.clientHeight,
        percentage: scrollInfo.percentage
      }
    };
  }
  
  /**
   * 策略4：行号比例同步（最后保险）
   */
  attemptLineRatioSync() {
    const lineInfo = this.textExtractor.getLineInfo();
    
    return {
      success: true,  // 行号同步总是成功
      data: {
        type: 'lineRatio',
        currentLine: lineInfo.currentLine,
        totalLines: lineInfo.totalLines,
        lineRatio: lineInfo.ratio
      }
    };
  }
  
  /**
   * 策略5：DOM锚点同步（预览区域专用）
   */
  attemptDomAnchorSync() {
    if (this.isEditor) {
      return { success: false, reason: '编辑器不支持DOM锚点同步' };
    }
    
    // 从DOM中提取可见元素作为锚点
    const domAnchors = this.extractDomAnchors();
    
    if (domAnchors.length === 0) {
      return { success: false, reason: '无可见DOM锚点' };
    }
    
    return {
      success: true,
      data: {
        type: 'domAnchor',
        anchors: domAnchors,
        viewport: this.textExtractor.getViewportInfo()
      }
    };
  }
  
  /**
   * 从DOM中提取锚点
   */
  extractDomAnchors() {
    const anchors = [];
    const viewport = this.textExtractor.getViewportInfo();
    
    // 查找可见的标题元素
    const headings = this.container.querySelectorAll('h1, h2, h3, h4, h5, h6');
    headings.forEach(heading => {
      const rect = heading.getBoundingClientRect();
      const absoluteTop = window.pageYOffset + rect.top;
      
      // 检查是否在可见区域附近
      if (this.isElementNearViewport(heading, viewport)) {
        const text = heading.textContent.trim();
        if (text.length > 3 && text.length < 100) {
          const relativePosition = this.calculateRelativePosition(absoluteTop, viewport);
          anchors.push({
            text: text,
            element: heading.tagName.toLowerCase(),
            position: relativePosition,
            absoluteTop: absoluteTop,
            confidence: 0.9
          });
        }
      }
    });
    
    // 查找可见的段落元素（如果标题不够）
    if (anchors.length < 2) {
      const paragraphs = this.container.querySelectorAll('p, li');
      paragraphs.forEach(p => {
        const rect = p.getBoundingClientRect();
        const absoluteTop = window.pageYOffset + rect.top;
        
        if (this.isElementNearViewport(p, viewport)) {
          const text = p.textContent.trim();
          if (text.length > 10 && text.length < 100) {
            const shortText = text.substring(0, 50).trim();
            const relativePosition = this.calculateRelativePosition(absoluteTop, viewport);
            anchors.push({
              text: shortText,
              element: p.tagName.toLowerCase(),
              position: relativePosition,
              absoluteTop: absoluteTop,
              confidence: 0.6
            });
          }
        }
      });
    }
    
    // 按置信度排序，只返回前3个
    return anchors
      .sort((a, b) => b.confidence - a.confidence)
      .slice(0, 3);
  }
  
  /**
   * 检查元素是否接近视窗
   */
  isElementNearViewport(element, viewport) {
    const rect = element.getBoundingClientRect();
    const viewportHeight = window.innerHeight;
    
    // 元素在视窗上下50%范围内都认为是"接近"
    return rect.top >= -viewportHeight * 0.5 && 
           rect.top <= viewportHeight * 1.5;
  }
  
  /**
   * 计算相对位置
   */
  calculateRelativePosition(absoluteTop, viewport) {
    const scrollTop = viewport.scrollTop;
    const clientHeight = viewport.clientHeight;
    
    // 计算元素在当前视窗中的相对位置
    const relativeTop = absoluteTop - scrollTop;
    return Math.max(0, Math.min(1, relativeTop / clientHeight));
  }
  
  /**
   * 获取可见锚点
   */
  getVisibleAnchors(anchorMap) {
    const viewport = this.textExtractor.getViewportInfo();
    const visibleAnchors = [];
    
    anchorMap.forEach((position, text) => {
      if (this.textExtractor.isLineVisible(position.line, viewport)) {
        const relativePosition = this.textExtractor.getRelativePosition(position.line, viewport);
        visibleAnchors.push({
          text: text,
          line: position.line,
          position: relativePosition,
          confidence: position.confidence,
          type: position.type
        });
      }
    });
    
    // 按置信度和位置排序
    return visibleAnchors
      .sort((a, b) => b.confidence - a.confidence)
      .slice(0, 3);  // 只取前3个最可靠的锚点
  }
  
  /**
   * 广播同步消息
   */
  broadcastSync(data, strategy) {
    const message = {
      type: this.config.channelId,
      action: 'sync',
      strategy: strategy,
      data: data,
      timestamp: Date.now(),
      source: this.isEditor ? 'editor' : 'preview'
    };
    
    this.broadcast(message);
  }
  
  /**
   * 消息广播（多渠道）
   */
  broadcast(message) {
    // PostMessage到父窗口和iframe
    if (window.parent !== window) {
      try {
        window.parent.postMessage(message, '*');
      } catch (e) {
      }
    }
    
    // 发送到所有iframe
    document.querySelectorAll('iframe').forEach(iframe => {
      try {
        iframe.contentWindow?.postMessage(message, '*');
      } catch (e) {}
    });
    
    // BroadcastChannel备份
    if (window.BroadcastChannel) {
      try {
        const channel = new BroadcastChannel(this.config.channelId);
        channel.postMessage(message);
        channel.close();
      } catch (e) {}
    }
  }
  
  /**
   * 设置消息监听器
   */
  setupMessageListener() {
    window.addEventListener('message', (event) => {
      if (event.data?.type === this.config.channelId) {
        this.handleRemoteSync(event.data);
      }
    });
  }
  
  /**
   * 处理远程同步（接收端）- 双向同步
   */
  handleRemoteSync(message) {
    const myType = this.isEditor ? 'editor' : 'preview';
    const messageSource = message.source;
    
    // 忽略自己发送的消息
    if (messageSource === myType) {
      return;
    }
    
    // 防止循环同步：如果正在发送本地同步，忽略远程消息
    if (this.syncState.scrollSource === 'local') {
      return;
    }
    
    const targetType = this.isEditor ? '编辑器' : '预览区域';
    this.syncState.scrollSource = 'remote';
    
    // 根据策略类型执行同步
    switch (message.strategy) {
      case 'smart':
        this.executeSmartSync(message.data);
        break;
      case 'simple':
        this.executeSimpleSync(message.data);
        break;
      case 'percentage':
        this.executePercentageSync(message.data);
        break;
      case 'lineRatio':
        this.executeLineRatioSync(message.data);
        break;
      case 'domAnchor':
        this.executeDomAnchorSync(message.data);
        break;
      default:
        console.log('[EnhancedScrollSync] ❌ 未知同步策略:', message.strategy);
    }
    
    setTimeout(() => {
      this.syncState.scrollSource = null;
    }, 300);
  }
  
  /**
   * 执行精确锚点同步
   */
  executeSmartSync(data) {
    console.log('[EnhancedScrollSync] 执行精确锚点同步');
    this.executeAnchorSync(data.anchors, 'smart');
  }
  
  /**
   * 执行简单锚点同步
   */
  executeSimpleSync(data) {
    console.log('[EnhancedScrollSync] 执行简单锚点同步');
    this.executeAnchorSync(data.anchors, 'simple');
  }
  
  /**
   * 执行锚点同步（通用方法）
   */
  executeAnchorSync(anchors, type) {
    if (!anchors || anchors.length === 0) {
      console.log('[EnhancedScrollSync] ❌ 无效锚点数据');
      return;
    }
    
    console.log(`[EnhancedScrollSync] 搜索 ${anchors.length} 个${type}锚点`);
    
    // 尝试匹配锚点
    const matches = this.findAnchorMatches(anchors);
    
    if (matches.length > 0) {
      console.log(`[EnhancedScrollSync] ✅ 找到 ${matches.length} 个匹配`);
      this.scrollToMatch(matches[0]);  // 使用最佳匹配
    } else {
      console.log('[EnhancedScrollSync] ❌ 未找到锚点匹配，降级到百分比同步');
      // 自动降级到百分比同步
      this.executePercentageSync({
        percentage: anchors[0].position || 0.5
      });
    }
  }
  
  /**
   * 执行百分比同步 - 区分编辑器和预览环境
   */
  executePercentageSync(data) {
    if (this.isEditor) {
      // 编辑器环境：滚动textarea
      this.executeEditorPercentageSync(data);
    } else {
      // 预览环境：滚动window
      this.executePreviewPercentageSync(data);
    }
  }
  
  /**
   * 编辑器百分比同步
   */
  executeEditorPercentageSync(data) {
    const textarea = this.container;
    const maxScrollTop = Math.max(0, textarea.scrollHeight - textarea.clientHeight);
    const targetScrollTop = data.percentage * maxScrollTop;
    textarea.scrollTop = targetScrollTop;
  }
  
  /**
   * 预览环境百分比同步
   */
  executePreviewPercentageSync(data) {
    const containerHeight = this.container.scrollHeight || document.documentElement.scrollHeight;
    const viewportHeight = window.innerHeight;
    const maxScrollTop = Math.max(0, containerHeight - viewportHeight);
    const targetScrollTop = data.percentage * maxScrollTop;
    
    window.scrollTo({
      top: targetScrollTop,
      behavior: 'smooth'
    });
  }
  
  /**
   * 执行行号比例同步 - 区分编辑器和预览环境
   */
  executeLineRatioSync(data) {
    if (this.isEditor) {
      // 编辑器环境：滚动textarea
      const textarea = this.container;
      const maxScrollTop = Math.max(0, textarea.scrollHeight - textarea.clientHeight);
      const targetScrollTop = data.lineRatio * maxScrollTop;
      
      textarea.scrollTop = targetScrollTop;
    } else {
      // 预览环境：滚动window
      const containerHeight = this.container.scrollHeight || document.documentElement.scrollHeight;
      const targetScrollTop = data.lineRatio * containerHeight;
      
      window.scrollTo({
        top: targetScrollTop,
        behavior: 'smooth'
      });
    }
  }
  
  /**
   * 执行DOM锚点同步
   */
  executeDomAnchorSync(data) {
    if (!data.anchors || data.anchors.length === 0) {
      // 降级到百分比同步
      this.executePercentageSync({ percentage: 0.5 });
      return;
    }
    
    // 如果是编辑器接收DOM锚点，需要在markdown中搜索对应文本
    if (this.isEditor) {
      this.executeDomToEditorSync(data.anchors);
    } else {
      // 预览区域接收DOM锚点，直接在DOM中搜索
      this.executeAnchorSync(data.anchors, 'dom');
    }
  }
  
  /**
   * DOM锚点到编辑器同步（特殊处理）
   */
  executeDomToEditorSync(domAnchors) {
    const textarea = this.container;
    const content = textarea.value;
    const lines = content.split('\n');
    
    // 尝试在markdown中找到匹配的文本
    let bestMatch = null;
    let bestMatchScore = 0;
    
    domAnchors.forEach(domAnchor => {
      const searchText = domAnchor.text;
      
      // 在每一行中搜索
      lines.forEach((line, lineIndex) => {
        if (line.includes(searchText)) {
          const score = searchText.length / line.length; // 匹配度
          if (score > bestMatchScore) {
            bestMatch = {
              line: lineIndex,
              position: domAnchor.position,
              confidence: score * domAnchor.confidence
            };
            bestMatchScore = score;
          }
        }
      });
    });
    
    if (bestMatch) {
      // 计算目标滚动位置
      const lineHeight = this.textExtractor.getLineHeight();
      const targetScrollTop = bestMatch.line * lineHeight - (bestMatch.position * textarea.clientHeight);
      const maxScrollTop = textarea.scrollHeight - textarea.clientHeight;
      const finalScrollTop = Math.max(0, Math.min(targetScrollTop, maxScrollTop));
      
      textarea.scrollTop = finalScrollTop;
    } else {
      // 使用第一个DOM锚点的位置作为百分比
      const percentage = domAnchors[0].position || 0.5;
      const targetScrollTop = percentage * (textarea.scrollHeight - textarea.clientHeight);
      textarea.scrollTop = Math.max(0, targetScrollTop);
    }
  }
  
  /**
   * 查找锚点匹配
   */
  findAnchorMatches(remoteAnchors) {
    const matches = [];
    
    remoteAnchors.forEach(anchor => {
      const element = this.searchTextInDOM(anchor.text);
      if (element) {
        const rect = element.getBoundingClientRect();
        matches.push({
          element: element,
          position: window.pageYOffset + rect.top,
          remotePosition: anchor.position,
          confidence: anchor.confidence || 0.5,
          text: anchor.text
        });
      }
    });
    
    return matches.sort((a, b) => b.confidence - a.confidence);
  }
  
  /**
   * 在DOM中搜索文本
   */
  searchTextInDOM(searchText) {
    const walker = document.createTreeWalker(
      this.container,
      NodeFilter.SHOW_TEXT,
      null,
      false
    );
    
    let node;
    while (node = walker.nextNode()) {
      if (node.textContent.includes(searchText)) {
        return node.parentElement;
      }
    }
    
    return null;
  }
  
  /**
   * 滚动到匹配位置
   */
  scrollToMatch(match) {
    const viewportHeight = window.innerHeight;
    let targetScrollTop;
    
    // 智能位置计算
    if (match.remotePosition <= 0.1) {
      targetScrollTop = Math.max(0, match.position - viewportHeight * 0.1);
    } else if (match.remotePosition >= 0.9) {
      targetScrollTop = Math.max(0, match.position - viewportHeight * 0.9);
    } else {
      targetScrollTop = Math.max(0, match.position - match.remotePosition * viewportHeight);
    }
    
    console.log(`[EnhancedScrollSync] 🎯 滚动到匹配位置: ${targetScrollTop}px`);
    
    window.scrollTo({
      top: targetScrollTop,
      behavior: 'smooth'
    });
  }
  
  /**
   * 防抖函数
   */
  debounce(func, delay) {
    let timeoutId;
    return (...args) => {
      clearTimeout(timeoutId);
      timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
  }
}

/**
 * 增强型Textarea提取器
 */
class EnhancedTextareaExtractor {
  constructor(textarea) {
    this.textarea = textarea;
  }
  
  getViewportInfo() {
    const lineHeight = this.getLineHeight();
    const scrollTop = this.textarea.scrollTop;
    const clientHeight = this.textarea.clientHeight;
    const totalLines = this.textarea.value.split('\n').length;
    
    const startLine = Math.floor(scrollTop / lineHeight);
    const endLine = Math.min(totalLines - 1, Math.ceil((scrollTop + clientHeight) / lineHeight));
    
    return {
      startLine,
      endLine,
      scrollTop,
      clientHeight,
      lineHeight,
      totalLines,
      visibleLines: endLine - startLine + 1
    };
  }
  
  getScrollInfo() {
    return {
      scrollTop: this.textarea.scrollTop,
      scrollHeight: this.textarea.scrollHeight,
      clientHeight: this.textarea.clientHeight,
      percentage: this.textarea.scrollTop / Math.max(1, this.textarea.scrollHeight - this.textarea.clientHeight)
    };
  }
  
  getLineInfo() {
    const viewport = this.getViewportInfo();
    const currentLine = Math.floor(viewport.startLine + viewport.visibleLines / 2);
    
    return {
      currentLine,
      totalLines: viewport.totalLines,
      ratio: currentLine / Math.max(1, viewport.totalLines - 1)
    };
  }
  
  isLineVisible(line, viewport) {
    return line >= viewport.startLine && line <= viewport.endLine;
  }
  
  getRelativePosition(line, viewport) {
    return (line - viewport.startLine) / Math.max(1, viewport.visibleLines - 1);
  }
  
  getLineHeight() {
    const style = window.getComputedStyle(this.textarea);
    const lineHeight = parseFloat(style.lineHeight);
    return isNaN(lineHeight) ? parseFloat(style.fontSize) * 1.2 : lineHeight;
  }
}

/**
 * 增强型DOM提取器
 */
class EnhancedDOMExtractor {
  constructor(container) {
    this.container = container;
  }
  
  getViewportInfo() {
    return {
      scrollTop: window.pageYOffset,
      clientHeight: window.innerHeight,
      scrollHeight: document.documentElement.scrollHeight
    };
  }
  
  getScrollInfo() {
    const scrollTop = window.pageYOffset;
    const scrollHeight = document.documentElement.scrollHeight;
    const clientHeight = window.innerHeight;
    
    return {
      scrollTop,
      scrollHeight,
      clientHeight,
      percentage: scrollTop / Math.max(1, scrollHeight - clientHeight)
    };
  }
  
  isLineVisible(line, viewport) {
    // DOM环境中行号概念不明确，始终返回true
    return true;
  }
  
  getRelativePosition(line, viewport) {
    // DOM环境中返回默认相对位置
    return 0.5;
  }
  
  getLineInfo() {
    // DOM环境中的行号信息基于滚动位置估算
    const scrollInfo = this.getScrollInfo();
    const estimatedTotalLines = Math.max(100, Math.floor(scrollInfo.scrollHeight / 20)); // 假设平均行高20px
    const currentLine = Math.floor(scrollInfo.percentage * estimatedTotalLines);
    
    return {
      currentLine,
      totalLines: estimatedTotalLines,
      ratio: scrollInfo.percentage
    };
  }
}

// 全局实例
if (typeof window !== 'undefined') {
  window.enhancedScrollSync = new EnhancedScrollSync();
  
  // 调试工具
  window.debugEnhancedSync = {
    showAnchors: () => {
      const sync = window.enhancedScrollSync;
    },
    testSync: (strategy = 'smart') => {
      const sync = window.enhancedScrollSync;
      sync.attemptSync();
    },
    getStats: () => {
      const sync = window.enhancedScrollSync;
      return {
        smartAnchors: sync.anchors.smart.size,
        simpleAnchors: sync.anchors.simple.size,
        fallbackAnchors: sync.anchors.fallback.size,
        currentStrategy: sync.syncState.currentStrategy,
        isEditor: sync.isEditor
      };
    }
  };
}